# coding: UTF-8

# Copyright (C)Nintendo All rights reserved.
#
# These coded instructions, statements, and computer programs contain proprietary
# information of Nintendo and/or its licensed developers and are protected by
# national and international copyright laws. They may not be disclosed to third
# parties or copied or duplicated in any form, in whole or in part, without the
# prior written consent of Nintendo.
#
# The content herein is highly confidential and should be handled accordingly.

import argparse
import codecs
import gettext
import locale
import os
import re
import sys

# clang の python binding をインポートします。
APP_ROOT   = os.path.dirname( os.path.abspath(__file__) )
SIGLO_ROOT = os.path.normpath( APP_ROOT + '/../../../..' )
CLANG_ROOT = os.path.normpath( SIGLO_ROOT + '/Externals/Binaries/libclang' )
sys.path.append( CLANG_ROOT + '/src/bindings/python' )
from clang import cindex
cindex. Config.set_library_path( CLANG_ROOT + '/bin' )

# ログレベルの定義です。
LOG_LEVEL_DEBUG             = 1
LOG_LEVEL_VERBOSE           = 2
LOG_LEVEL_WARNING           = 3
LOG_LEVEL_ERROR             = 4

# 規約レベルの定義です。
ERROR_LEVEL_UNKNOWN         = 0
ERROR_LEVEL_INFORMATION     = 1
ERROR_LEVEL_RECOMMENDED     = 2
ERROR_LEVEL_REQUIRED        = 3
ERROR_LEVEL_FATAL           = 4
MESSAGES_FOR_ERROR_LEVEL    = [
    u'UNKNOWN',
    u'INFO',
    u'RECOMMENDED',
    u'REQUIRED',
    u'FATAL'
]

# 有効な拡張子のリストです。
VALID_EXTENSIONS            = [ '.c', '.cpp', '.h' ]
SOURCE_EXTENSIONS           = [ '.c', '.cc', '.cp', '.cpp', '.cxx', '.c++' ]
HEADER_EXTENSIONS           = [ '.h', '.hh', '.hp', '.hpp', '.hxx', '.h++' ]

# Unicode 関連の規約で BOM の有無を設定する定数です。
BOM_REQUIRED                = True
BOM_FORBIDDEN               = False

# 改行コードの指定に使用する定数です。
LINEFEED_CR                 = 0
LINEFEED_LF                 = 1
LINEFEED_CRLF               = 2

# インデントタイプを指定する定数です。
INDENT_TAB                  = 0
INDENT_SPACE                = 1

# アクセス修飾子を表す定数です。
ACCESS_SPECIFIER_UNKNOWN    = -1    # 不明です。
ACCESS_SPECIFIER_NONE       = 0     # まだ指定されていません。
ACCESS_SPECIFIER_PUBLIC     = 1     # public です。
ACCESS_SPECIFIER_PROTECTED  = 2     # protected です。
ACCESS_SPECIFIER_PRIVATE    = 3     # private です。

# 標準ヘッダです。
C89_HEADERS = [
    'assert.h', 'ctype.h', 'errno.h', 'float.h', 'limits.h', 'locale.h', 'math.h', 'setjmp.h',
    'signal.h', 'stdarg.h', 'stddef.h', 'stdio.h', 'stdlib.h', 'string.h', 'time.h'
]
C95_HEADERS = [
    'iso646.h', 'wchar.h', 'wctype.h'
]
C99_HEADERS = [
    'complex.h', 'fenv.h', 'inttypes.h', 'stdalign.h', 'stdbool.h', 'stdint.h', 'tgmath.h',
]
C11_HEADERS = [
    'stdatomic.h', 'stdnoreturn.h', 'threads.h', 'uchar.h',
]
CPP_C_HEADERS = [
    'cassert', 'cctype', 'cerrno', 'cfenv', 'cfloat', 'cinttypes', 'ciso646', 'climits',
    'clocale', 'cmath', 'csetjmp', 'csignal', 'cstdarg', 'cstdbool', 'cstddef', 'cstdint',
    'cstdio', 'cstdlib', 'cstring', 'ctgmath', 'ctime', 'cuchar', 'cwchar', 'cwctype'
]
CPP03_HEADERS = [
    'algorithm', 'bitset', 'complex', 'deque', 'exception', 'functional', 'iterator', 'limits',
    'list', 'locale', 'map', 'memory', 'new', 'numeric', 'queue', 'set', 'stack', 'stdexcept',
    'string', 'typeinfo', 'utility', 'valarray', 'vector'
]
CPP11_HEADERS = [
    'atomic', 'array', 'chrono', 'codecvt', 'condition_variable', 'forward_list', 'future',
    'initializer_list', 'mutex', 'random', 'ratio', 'regex', 'system_error', 'thread', 'tuple',
    'typeindex', 'type_traits', 'unordered_map', 'unordered_set'
]
C_HEADERS   = C89_HEADERS + C95_HEADERS + C99_HEADERS + C11_HEADERS
CPP_HEADERS = CPP_C_HEADERS + CPP03_HEADERS + CPP11_HEADERS

# NOLINT の検索に使用する正規表現です。
# 特定の行に対する規約チェックを無効化するには NOLINT(category),
# ファイル全体で規約チェックを無効化するには NOLINT_FILE(category) をコメントとして記述します。
NOLINT_PATTERN = re.compile(
    r'\bNOLINT(_FILE)?\s*(?:\(\s*([0-9a-zA-Z_]*(?:/[0-9a-zA-Z_]+)?)\s*\))?(.*)$')

# clang に渡すオプションです。
build_options               = [

    # 警告を有効化します。
    '-Wall',
    '-Wimplicit-fallthrough',

    # C++11 のソースコードとして解析します。
    '-xc++',
    '-std=c++11'
]

# clang のエラーを無視して規約チェックを継続するオプションです。
syntax_error_ignored        = False

# ログの詳細度を表すレベルです。
log_level                   = LOG_LEVEL_WARNING

# ログ出力のエンコーディングです。
encoding                    = None

# ログ出力の言語です。
language                    = 'en'

# 規約チェックのレベルです。
error_level                 = ERROR_LEVEL_REQUIRED

# 無視するエラーを登録するフィルタです。
ignores                     = []

# 検証するエラーを登録するフィルタです。 ignores よりも優先されます。
only                        = []

# ヘッダを再帰的にチェックします。
recursive                   = False

# ヘッダを再帰的にチェックする範囲を指定するルートパスです。
root                        = None

# 規約チェックから除外するファイルのリストです。
exclude_files               = []

# libclang の構文解析インタフェースです。
index                       = cindex.Index.create()

# 規約チェックの結果を記憶しておくリストです。
source_caches               = {}
checked_results             = []
error_count                 = 0
warning_count               = 0

# 適用するコーディング規約です。
checkers                    = []

# 翻訳オブジェクトです。
_ = gettext.translation(
        domain    = 'CppCodingRuleChecker',
        localedir = os.path.join(os.path.dirname(__file__), 'Locale'),
        languages = ['en']
    ).ugettext

def normalize_path(path):
    """パスを正規化します。

    Args:
        path:               正規化対象のパスです。
    Returns:
        区切り文字に / を使用する絶対パスです。
    """
    return os.path.abspath(path).replace('\\', '/')

class SourceFile(object):
    """ソースコードを格納するデータ型です。

    raw_body は生のソースコードです。
    body は raw_body から BOM を取り除いた utf-8 のソースコードです。

    raw_lines は body を行毎に分割したものです。
    lines は raw_lines から改行コードが除去されています。
    simple_lines は lines からさらにコメントや文字列が除去されています。
    """

    _ESCAPE_PATTERN = re.compile(r'\\([abnrftv\\?\'"]|\d+|x[0-9A-Fa-f]+)')
    _STRING_PATTERN = re.compile(r'"[^"]*"')
    _CHAR_PATTERN   = re.compile(r'\'.\'')

    def __init__(self):
        self.path         = None
        self.raw_body     = None
        self.body         = None
        self.raw_lines    = []
        self.lines        = []
        self.simple_lines = []
        self.nolints      = None
        self.tu           = None
        self.is_processed = False

    def process(self, tu):
        """規約チェックを実行しやすいようにソースコードを加工します。

        Args:
            tu:             TranslationUnit です。
        """
        if not self.is_processed:
            self.tu = tu
            self._create_nolint_list()
            self._split_lines()
            self.is_processed = True

    def get_full_extent(self):
        """ソースコード全体を表す SourceRange を取得します。

        Returns:
            ソースコード全体を表す SourceRange です。
        """
        file  = cindex.File.from_name(self.tu, self.path)
        start = cindex.SourceLocation.from_offset(self.tu, file, 0)
        end   = cindex.SourceLocation.from_offset(self.tu, file, len(self.body) - 1)
        return cindex.SourceRange.from_locations(start, end)

    def _create_nolint_list(self):
        self.nolints = {}
        tokens  = self.tu.get_tokens(extent = self.get_full_extent())
        for token in tokens:
            if token.kind != cindex.TokenKind.COMMENT:
                continue
            lines = token.spelling.splitlines()
            for i in xrange(len(lines)):
                nolint_list = self._create_nolint(lines[i], token.location.line + i)
            for nolint in nolint_list:
                if not self.nolints.has_key(nolint.line):
                    self.nolints[nolint.line] = []
                self.nolints[nolint.line].append(nolint)

    def _create_nolint(self, comment, linenum):
        nolints = []
        match   = NOLINT_PATTERN.search(comment)
        while match:
            if match.group(1):
                nolints.append(NoLint(match.group(2), 0))
            else:
                nolints.append(NoLint(match.group(2), linenum))
            match = NOLINT_PATTERN.search(match.group(3))
        return nolints

    def _split_lines(self):
        # libclang の行番号と対応させるため、先頭にはダミーの行を挟みます。
        keep_ends = True
        del self.raw_lines[:]
        self.raw_lines.append('')
        self.raw_lines.extend(self.body.splitlines(keep_ends))

        # 全ての行から末尾の改行文字を取り除きます。
        del self.lines[:]
        del self.simple_lines[:]
        for raw_line in self.raw_lines:
            line = raw_line.rstrip('\r\n')
            self.lines.append(line)
            self.simple_lines.append(line)

        # コメントと文字列を取り除きます。
        self._remove_comment(self.simple_lines)
        self._remove_string(self.simple_lines)

    def _remove_comment(self, lines):
        comments = []
        tokens   = self.tu.get_tokens(extent = self.get_full_extent())
        for token in tokens:
            if token.kind == cindex.TokenKind.COMMENT:
                comments.append(token)
            for comment in reversed(comments):
                self._remove_range(lines, comment.extent)

    def _remove_string(self, lines):
        for i in xrange(len(lines)):
            lines[i] = self._ESCAPE_PATTERN.sub('', lines[i])
            lines[i] = self._STRING_PATTERN.sub('""', lines[i])
            lines[i] = self._CHAR_PATTERN.sub('\'\'', lines[i])

    def _remove_range(self, lines, range):
        start_raw = range.start.line
        end_raw   = range.end.line
        start_col = range.start.column - 1
        end_col   = range.end.column - 1
        if start_raw == end_raw:
            lines[start_raw] = lines[start_raw][:start_col] + lines[start_raw][end_col:]
        else:
            lines[start_raw] = lines[start_raw][:start_col]
            for i in xrange(start_raw + 1, end_raw):
                lines[i] = ''
            lines[end_raw] = lines[end_raw][end_col:]

class Error(object):
    """規約チェックのエラーの詳細を格納するデータ型です。
    """

    def __init__(self, category, level, message, file, line):
        """規約チェックのエラーを生成するコンストラクタです。

        Args:
            category:       規約のカテゴリです。
            level:          規約レベルです。
            message:        エラーメッセージです。
            file:           規約違反を発見したファイルです。
            line:           規約違反の発見位置を表す行番号です。
        """
        self.category   = category
        self.level      = level
        self.message    = message
        self.file       = file
        self.line       = line

class NoLint(object):
    """規約チェックの抑制に関する情報を格納します。
    """

    def __init__(self, category, line):
        self.category   = category
        self.line       = line

class CheckedResult(object):
    """規約チェックの結果を保存しておくデータ型です。
    """

    def __init__(self, path):
        self.path       = path
        self.errors     = []

class NestingState(object):
    """ネスト状態を記憶するクラスです。
    """

    def __init__(self):
        self.stack      = []
        self.namespaces = []
        self.classes    = []
        self.functions  = []

    def reset(self):
        del self.stack[:]
        del self.namespaces[:]
        del self.classes[:]
        del self.functions[:]

    def enter(self, node):
        if node.kind == cindex.CursorKind.NAMESPACE:
            self.stack.append(node)
            self.namespaces.append(node)
        elif is_class_like(node):
            self.stack.append(node)
            self.classes.append(node)
        elif is_function_like(node):
            self.stack.append(node)
            self.functions.append(node)

    def leave(self, node):
        if node.kind == cindex.CursorKind.NAMESPACE:
            self.stack.pop()
            self.namespaces.pop()
        elif is_class_like(node):
            self.stack.pop()
            self.classes.pop()
        elif is_function_like(node):
            self.stack.pop()
            self.functions.pop()

    @property
    def is_anonymous(self):
        for namespace in self.namespaces:
            if namespace.spelling == '':
                return True
        return False

    @property
    def is_global(self):
        return len(self.stack) == 0

def log(level, message):
    """ログメッセージを標準エラー出力します。

    Args:
        level:              ログレベルです。
        message:            出力するログメッセージです。
    """
    if log_level <= level:
        if encoding:
            sys.stderr.write(message.encode(encoding))
        elif sys.stderr.encoding:
            sys.stderr.write(message)
        else:
            sys.stderr.write(message.encode('cp932'))

def debug(message, linefeed = True):
    """ログメッセージを標準エラー出力します。

    Args:
        message:            出力するログメッセージです。
        linefeed:           自動的に改行を付加する設定です。
    """
    text = message + u'\n' if linefeed else message
    log(LOG_LEVEL_DEBUG, text)

def verbose(message, linefeed = True):
    """ログメッセージを標準エラー出力します。

    Args:
        message:            出力するログメッセージです。
        linefeed:           自動的に改行を付加する設定です。
    """
    text = message + u'\n' if linefeed else message
    log(LOG_LEVEL_VERBOSE, text)

def warning(message):
    """ログメッセージを標準エラー出力します。

    Args:
        message:            出力するログメッセージです。
    """
    global warning_count
    log(LOG_LEVEL_WARNING, u'WARNING: ' + message + u'\n')
    warning_count += 1

def error(name, level, message, source, line = 0):
    """規約違反を表すエラーを生成します。
       この関数では現在のファイルに対するエラーを蓄積するだけであり、
       実際に出力するためには print_errors を実行します。

    Args:
        name:               規約の名前です。
        level:              規約レベルです。
        message:            エラーメッセージです。
        source:             ソースファイルです。
        line:               行番号です。
    """
    global error_count

    result   = checked_results[-1]
    nolints  = source.nolints
    filename = source.path

    # コマンドラインオプションで指定された規約レベル以下であれば弾きます。
    if level < error_level:
        return

    # コマンドラインオプションで指定できるエラーカテゴリによるフィルタリングです。
    if only:
        for filter in only:
            if not name.startswith(filter):
                return
    elif ignores:
        for filter in ignores:
            if name.startswith(filter):
                return

    # その行を対象とする、あるいは全体を対象とする NOLINT が指定されている場合には弾きます。
    if nolints.has_key(0):
        for nolint in nolints[0]:
            if nolint.category is None or name.startswith(nolint.category):
                return
    if line != 0 and nolints.has_key(line):
        for nolint in nolints[line]:
            if nolint.category is None or name.startswith(nolint.category):
                return

    # ここまで到達したエラーは出力すべきエラーなので記憶しておきます。
    result.errors.append(Error(name, level, message, filename, line))
    error_count += 1

def print_errors(result):
    """現在のファイルに対する規約チェックの結果を出力します。

    Args:
        result:             規約チェックの結果です。
    """
    # ファイル名と行番号でソートします。
    sorted_errors = sorted(result.errors, cmp=lambda x, y:
                           cmp(x.file, y.file) or cmp(x.line, y.line))

    # エラーメッセージを出力します。
    for error in sorted_errors:
        text = u"ERROR: {file} L.{line} ({level}) {error_message} [{category}]\n".format(
               file=error.file, line=error.line, level=MESSAGES_FOR_ERROR_LEVEL[error.level],
               error_message=error.message, category=error.category)
        log(LOG_LEVEL_ERROR, text)

def remove_bom(raw_body):
    """ソースファイルから BOM を取り除きます。

    Args:
        raw_body:           加工されていない生のソースファイルです。
    Returns:
        BOM を取り除いた UTF-8 のソースファイルです。
    """
    if len(raw_body) > 0 and raw_body[0] == u'\ufeff':
        body = raw_body[1:]
    else:
        body = raw_body
    return body

def read_source(path, bom_remove = False):
    """ソースコードを取得します。

    Args:
        path:               対象のファイルのパスです。
        bom_remove:         BOM を除去したい場合に True を指定します。
    Returns:
        取得したソースコードです。
    """
    if not source_caches.has_key(path):
        source = SourceFile()
        try:
            source.path     = path
            source.raw_body = codecs.open(path, 'r', 'utf_8', 'strict').read()
            body            = remove_bom(source.raw_body) if bom_remove else source.raw_body
            source.body     = body.encode('utf_8')
        except:
            return None
        source_caches[path] = source
        return source
    else:
        return source_caches[path]

def get_children(node, kind = None, name = None):
    """libclang の AST ノードから子ノードのリストを生成します。

    Args:
        node:               親ノードです。
        kind:               検索するノードのタイプです。
        name:               検索するノードの名前です。
    Returns:
        子ノードのリストです。
    """
    children = []
    for child in node.get_children():
        if (kind is None or child.kind == kind) and (name is None or child.spelling == name):
            children.append(child)
    return children

def find_child(node, kind):
    """libclang の AST ノードから指定されたタイプの子ノードを検索します。

    Args:
        node:               親ノードです。
        kind:               検索するノードのタイプです。
    Returns:
        見つかった子ノードです。見つからなかった場合は None です。
    """
    for child in node.get_children():
        if child.kind == kind:
            return child

def get_tokens(node):
    """libclang の AST ノードからトークンのリストを生成します。

    Args:
        node:               親ノードです。
    Returns:
        トークンのリストです。
    """
    tokens = []
    for token in node.get_tokens():
        tokens.append(token)
    return tokens

def is_class(node):
    """指定された AST ノードがクラスであるか否かを判定します。

    Args:
        node                判定対象の ast ノードです。
    Returns:
        クラスであれば True です。テンプレートクラスも含みますが、構造体は含みません。
    """
    if node.kind == cindex.CursorKind.CLASS_DECL:
        return True
    elif (node.kind == cindex.CursorKind.CLASS_TEMPLATE or
          node.kind == cindex.CursorKind.CLASS_TEMPLATE_PARTIAL_SPECIALIZATION):
        count = 0
        for token in node.get_tokens():
            if token.kind == cindex.TokenKind.PUNCTUATION:
                if token.spelling == '<':
                    count += 1
                elif token.spelling == '>':
                    count -= 1
            elif token.kind == cindex.TokenKind.KEYWORD and count == 0:
                if token.spelling == 'class':
                    return True
                elif token.spelling == 'struct':
                    return False
        return False

def is_struct(node):
    """指定された AST ノードが構造体であるか否かを判定します。

    Args:
        node                判定対象の ast ノードです。
    Returns:
        構造体であれば True です。テンプレートの構造体も含みますが、クラスは含みません。
    """
    if node.kind == cindex.CursorKind.STRUCT_DECL:
        return True
    elif (node.kind == cindex.CursorKind.CLASS_TEMPLATE or
          node.kind == cindex.CursorKind.CLASS_TEMPLATE_PARTIAL_SPECIALIZATION):
        return is_class(node) == False

def is_union(node):
    """指定された AST ノードが共用体であるか否かを判定します。

    Args:
        node                判定対象の ast ノードです。
    Returns:
        共用体であれば True です。
    """
    return node.kind == cindex.CursorKind.UNION_DECL

def is_class_like(node):
    """指定された AST ノードがクラスであるか否かを判定します。

    Args:
        node                判定対象の ast ノードです。
    Returns:
        クラスか構造体であれば True です。テンプレートも含みます。
    """

    return node.kind in [cindex.CursorKind.CLASS_DECL,
                         cindex.CursorKind.STRUCT_DECL,
                         cindex.CursorKind.CLASS_TEMPLATE,
                         cindex.CursorKind.CLASS_TEMPLATE_PARTIAL_SPECIALIZATION,
                         cindex.CursorKind.UNION_DECL]

def is_type_decl(node):
    """指定された AST ノードが型定義であるか否かを判定します。

    Args:
        node                判定対象の ast ノードです。
    Returns:
        クラスか型定義であれば True です。
    """
    return node.kind in [cindex.CursorKind.CLASS_DECL,
                         cindex.CursorKind.STRUCT_DECL,
                         cindex.CursorKind.CLASS_TEMPLATE,
                         cindex.CursorKind.CLASS_TEMPLATE_PARTIAL_SPECIALIZATION,
                         cindex.CursorKind.ENUM_DECL,
                         cindex.CursorKind.TYPEDEF_DECL,
                         cindex.CursorKind.TYPE_ALIAS_DECL,
                         cindex.CursorKind.UNION_DECL]

def is_function_like(node):
    """指定された AST ノードが関数のであるか否かを判定します。

    Args:
        node:               判定対象の ast ノードです。
    Returns:
        関数, メソッド, コンストラクタ, デストラクタのいずれかであれば True です。
        テンプレート関数やテンプレートメソッドも含みます。
    """
    return node.kind in [ cindex.CursorKind.FUNCTION_DECL,
                          cindex.CursorKind.FUNCTION_TEMPLATE,
                          cindex.CursorKind.CXX_METHOD,
                          cindex.CursorKind.CONSTRUCTOR,
                          cindex.CursorKind.DESTRUCTOR,
                          cindex.CursorKind.CONVERSION_FUNCTION ]

def is_static(node):
    """指定された定義に static 指定子が含まれるか否かを判定します。

    Args:
        node                判定対象の ast ノードです。
    Returns:
        static が指定されていれば True です。
    """
    # TODO: 怪しい。もっとまともな判定方法はない？
    for token in node.get_tokens():
        if token.spelling == 'static':
            return True
        elif token.spelling == '{' or token.spelling == '(':
            return False
    return False

def convert_node_to_specifier(node):
    """AST ノードに対応するアクセス修飾子を取得します。

    Args:
        node                対象の ast ノードです。
    Returns:
        AST ノードがアクセス修飾子であれば対応する定数 ACCESS_SPECIFIER_XXX を返します。
        異なる種類のノードであれば ACCESS_SPECIFIER_UNKNOWN を返します。
    """
    if node.kind != cindex.CursorKind.CXX_ACCESS_SPEC_DECL:
        return ACCESS_SPECIFIER_UNKNOWN
    else:
        name = node.access_specifier.name
        if name == 'PUBLIC':
            return ACCESS_SPECIFIER_PUBLIC
        elif name == 'PROTECTED':
            return ACCESS_SPECIFIER_PROTECTED
        elif name == 'PRIVATE':
            return ACCESS_SPECIFIER_PRIVATE
        else:
            return ACCESS_SPECIFIER_UNKNOWN

class Checker(object):
    """規約チェッカの基底クラスです。

    新しい規約チェッカを追加するためには Checker クラスを継承したクラスを追加し、
    その中で hook_* 関数をオーバロードします。
    """

    def hook_setup(self, source):
        """各翻訳単位の規約チェックで最初に 1 度だけ実行されます。

        Args:
            source:         ソースコードです。
        Returns:
            規約チェックの結果です。
        """
        pass


    def hook_begin(self, source):
        """各ファイルの規約チェックで最初に 1 度だけ実行されます。

        Args:
            source:         ソースコードです。
        Returns:
            規約チェックの結果です。
        """
        pass

    def hook_warning(self, warning, source):
        """警告を元に規約チェックを行います。

        Args:
            warning:        libclang の警告です。 cindex.Diagnostic に相当します。
            source:         ソースコードです。
        """
        pass

    def hook_token(self, token, source):
        """トークンに対する規約チェックです。

        Args:
            token:          走査中のトークンです。 Token に相当します。
            source:         ソースコードです。
        """
        pass

    def hook_include(self, include_file, source):
        """ヘッダファイルのインクルードに対応する規約チェックです。

        Args:
            include_file:   インクルードされたファイルを示す cindex.FileInclusion です。
            source:         ソースコードです。
        Returns:
            規約チェックの結果です。
        """
        pass

    def hook_node_enter(self, node, source, skip):
        """libclang によって生成された AST に対する規約チェックです。子を辿る前に実行されます。

        Args:
            node:           走査中の AST ノードです。 cindex.Cursor に相当します。
            source:         ソースコードです。
        Returns:
            規約チェックの結果です。
        """
        pass

    def hook_node_leave(self, node, source, skip):
        """libclang によって生成された AST に対する規約チェックです。子を辿った後に実行されます。

        Args:
            node:           走査中の AST ノードです。 cindex.Cursor に相当します。
            source:         ソースコードです。
        Returns:
            規約チェックの結果です。
        """
        pass

    def hook_line(self, linenum, raw_line, line, simple_line, source):
        """各行の規約チェックの最初に 1 度だけ実行されます。

        Args:
            linenum:        1 から始まる行番号です。
            raw_line:       ソースコードから読み込んだ文字列をそのまま格納しています。
            line:           raw_line から改行コードを除去した文字列です。
            simple_line:    line からさらにコメントと文字列リテラルが除去されています。
            source:         ソースコードです。
        Returns:
            規約チェックの結果です。
        """
        pass

    def hook_end(self, source):
        """各ファイルの規約チェックで最後に 1 度だけ実行されます。

        Args:
            source:         ソースコードです。
        Returns:
            規約チェックの結果です。
        """
        pass

    def hook_cleanup(self, source):
        """各翻訳単位の規約チェックで最後に 1 度だけ実行されます。

        Args:
            source:         ソースコードです。
        Returns:
            規約チェックの結果です。
        """
        pass

class PathChecker(Checker):
    """
    ファイルパスやファイル名、拡張子の検証を行います。
    """

    # モジュール全体をインクルードするヘッダファイル e.g.) os.h
    _PATTERN_MODULE = re.compile(r'^([a-z][a-zA-Z0-9]*)(\.h)$')

    # 通常のファイル e.g.) os_Mutex.h, os_SimpleLock.cpp, os_Mutex-os.win32.cpp
    _PATTERN_FILE   = re.compile(
        r'^([a-z][a-z0-9]*)_([A-Z][-.a-zA-Z0-9]*)$')

    # 実装分岐関連のファイル e.g.) build_Compiler.clang.h
    _PATTERN_BUILD   = re.compile(
        r'^(build)_([A-Z][a-zA-Z0-9]*)(\.[a-z0-9_]+)*?$')

    def hook_begin(self, source):

        # 入力ファイルのパスを解析して分割します。
        path      = source.path
        dir, file = os.path.split(path)
        dirs      = dir.split('/')
        name, ext = os.path.splitext(file)

        # 拡張子の規約チェックです。
        if ext not in VALID_EXTENSIONS:
            error('file/extension', ERROR_LEVEL_REQUIRED,
                _(u'ファイルの拡張子には {extensions} のいずれかを使用してください。').format(
                  extensions = ', '.join(VALID_EXTENSIONS)), source)

        # ファイル名とディレクトリ階層の規約チェックです。
        match_module = self._PATTERN_MODULE.match(file)
        match_build  = self._PATTERN_BUILD.match(name)
        match_file   = self._PATTERN_FILE.match(name)
        module       = None
        if match_module:
            self._check_module_header(dirs, name, ext, source)
        elif match_build:
            module = 'TargetConfigs'
            self._check_file_header(dirs, module, name, ext, source)
        elif match_file:
            module = match_file.group(1)
            self._check_file_header(dirs, module, name, ext, source)
        else:
            error('file/name', ERROR_LEVEL_REQUIRED,
                _(u'ファイル名が命名規則に違反しています。'), source)

    def _check_module_header(self, dirs, name, ext, source):

        # モジュール全体をインクルードするヘッダは Include/nn 直下です。
        parent = dirs.pop() if dirs else ''
        granpa = dirs.pop() if dirs else ''
        if name != 'nn' and (parent != 'nn' or granpa != 'Include'):
            error('file/path', ERROR_LEVEL_REQUIRED,
                _(u'ファイルのディレクトリ構成を確認してください。'), source)

    def _check_file_header(self, dirs, module, name, ext, source):

        # detail ディレクトリは存在しても無視します。
        parent  = dirs.pop() if dirs else ''
        if parent == 'detail':
            parent = dirs.pop() if dirs else ''
        granpa  = dirs.pop() if dirs else ''

        # 親ディレクトリがモジュール名に一致する必要があります。
        if (parent != module or granpa == 'detail'):
            error('file/path', ERROR_LEVEL_REQUIRED,
                _(u'ファイルのディレクトリ構成を確認してください。'), source)

class NamingChecker(Checker):
    """
    命名規則の規約チェックを行います。
    """

    _PATTERN_CAMEL              = re.compile(r'^[a-z][a-zA-Z0-9]*$')
    _PATTERN_PASCAL             = re.compile(r'^[A-Z][a-zA-Z0-9]*$')
    _PATTERN_PASCAL_OR_NUMBER   = re.compile(r'^[A-Z0-9][a-zA-Z0-9]*$')
    _PATTERN_SNAKE              = re.compile(r'^[a-z][a-z0-9]*(?:_[a-z][a-z0-9]*)*$')
    _PATTERN_SCREAMING_SNAKE    = re.compile(r'^[A-Z][A-Z0-9]*(?:_[A-Z0-9]+)*$')
    _PATTERN_LOWER              = re.compile(r'^[a-z][a-z0-9]*$')
    _PATTERN_UPPER              = re.compile(r'^[A-Z][A-Z0-9]*$')
    _PATTERN_GLOBAL_SYMBOL      = re.compile(r'^[a-z0-9]+[A-Z][a-zA-Z0-9]*$')

    _IGNORE_FUNCTIONS = [
        'allocate', 'assign', 'begin', 'cbegin', 'cend', 'clear', 'construct', 'crbegin',
        'crend', 'deallocate', 'destroy', 'emplace', 'empty', 'end', 'erase', 'generate',
        'get_allocator', 'insert', 'is_always_equal', 'lock', 'lock_shared', 'max', 'max_size',
        'min', 'now', 'param', 'propagate_on_container_copy_assignment',
        'propagate_on_container_swap', 'propagete_on_container_move_assignment', 'rbegin',
        'rend', 'reset', 'select_on_container_copy_construction', 'size', 'swap', 'try_lock',
        'try_lock_for', 'try_lock_shared', 'try_lock_shared_for', 'try_lock_shared_until',
        'try_lock_until', 'unlock', 'unlock_shared',

        #'capacity', 'reserver',
        #'back', 'front', 'merge', 'pop_back', 'push_back', 'pop_front', 'push_front',
        #'remove_if', 'remove', 'resize', 'reverse', 'sort', 'splice', 'unique'
    ]

    _IGNORE_TYPES = [
        'allocator_type', 'clock', 'const_iterator', 'const_pointer', 'const_reference',
        'const_reverse_iterator', 'const_void_pointer', 'difference_type', 'duration',
        'is_steady', 'iterator', 'key_comp', 'key_compare', 'key_type', 'other', 'period',
        'pointer', 'reference', 'rep', 'result_type', 'reverse_iterator', 'size_type',
        'time_point', 'value_comp', 'value_compare', 'value_type', 'void_pointer',

        #'iterator_category'
    ]

    def __init__(self):
        self.nest   = NestingState()
        self.define = False

    def hook_setup(self, source):
        self.nest.reset()

    def hook_begin(self, source):
        self.define = False

    def hook_token(self, token, source):
        # define 識別子の直後の識別子がマクロ名です。
        if token.kind == cindex.TokenKind.IDENTIFIER:
            if token.spelling == 'define':
                self.define = True
            elif self.define:
                self._check_macro(token, source)
                self.define = False

    def hook_node_enter(self, node, source, skip):

        # 命名規則の規約チェックです。
        if not skip:
            if node.kind == cindex.CursorKind.NAMESPACE:
                self._check_namespace(node, source)
            elif node.kind in [ cindex.CursorKind.FUNCTION_DECL,
                                cindex.CursorKind.FUNCTION_TEMPLATE,
                                cindex.CursorKind.CXX_METHOD ]:
                self._check_function(node, source)
            elif is_type_decl(node):
                self._check_type(node, source)
            elif node.kind == cindex.CursorKind.TEMPLATE_TYPE_PARAMETER:
                self._check_template_param(node, source)
            elif node.kind == cindex.CursorKind.ENUM_CONSTANT_DECL:
                self._check_enum_constant(node, source)
            elif node.kind == cindex.CursorKind.VAR_DECL:
                if node.type.spelling.startswith('const'):
                    self._check_const(node, source)
                else:
                    self._check_var(node, source)
            elif node.kind == cindex.CursorKind.PARM_DECL:
                self._check_param(node, source)
            elif node.kind == cindex.CursorKind.FIELD_DECL:
                self._check_field(node, source)

        # ネスト状態を記憶します。
        self.nest.enter(node)

    def hook_node_leave(self, node, source, skip):
        self.nest.leave(node)

    def _check_namespace(self, node, source):
        if not self._validate(node.spelling, self._PATTERN_CAMEL):
            error('name/namespace', ERROR_LEVEL_REQUIRED,
                _(u'名前空間 {name} は {rule} の命名規則に違反しています。').format(
                  name = node.spelling, rule = 'camelCase'),
                  source, node.location.line)

    def _check_function(self, node, source):
        if node.spelling.startswith('operator'):
            pass
        elif self.nest.is_global and not is_static(node):
            if ((not self._validate(node.spelling, self._PATTERN_GLOBAL_SYMBOL)) and
                (not self._validate(node.spelling, self._PATTERN_PASCAL))):
                error('name/function', ERROR_LEVEL_REQUIRED,_(
                      u'名前空間外の関数 {name} は命名規則に違反しています。 '\
                      u'{example} のように本来の名前空間に相当する単語を先頭に付加します。'\
                      u'ただし、非公開のシンボルであることを表す detail は常に 2 番目です。')
                      .format(name    = node.spelling,
                              example = 'nnmoduleFunction, nndetailModuleFunction'),
                      source, node.location.line)
        else:
            if not self._validate(node.spelling, self._PATTERN_PASCAL,
                                  ignores = self._IGNORE_FUNCTIONS):
                error('name/function', ERROR_LEVEL_REQUIRED,
                    _(u'関数 {name} は {rule} の命名規則に違反しています。').format(
                      name = node.spelling, rule = 'PascalCase'),
                      source, node.location.line)

    def _check_type(self, node, source):
        if not self._validate(node.spelling, self._PATTERN_PASCAL,
                              ignores = self._IGNORE_TYPES):
            error('name/type', ERROR_LEVEL_REQUIRED,
                _(u'型 {name} は {rule} の命名規則に違反しています。').format(
                  name = node.spelling, rule = 'PascalCase'),
                  source, node.location.line)

    def _check_template_param(self, node, source):
        if not self._validate(node.spelling, self._PATTERN_PASCAL):
            error('name/tparam', ERROR_LEVEL_REQUIRED, _(
                  u'テンプレートパラメータ {name} は {rule} の命名規則に違反しています。 '\
                  u'{example} のような一文字からなる名前を使用しても構いません。').format(
                  name = node.spelling, rule = 'PascalCase', example = 'T, U'),
                  source, node.location.line)

    def _check_enum_constant(self, node, source):
        parent = node.lexical_parent
        if parent and parent.kind == cindex.CursorKind.ENUM_DECL:
            prefix = parent.spelling + '_'
            if not self._validate(node.spelling, self._PATTERN_PASCAL_OR_NUMBER,
                                  prefix = prefix):
                error('name/enum', ERROR_LEVEL_REQUIRED, _(
                      u'列挙子 {name} は命名規則に違反しています。 '\
                      u'{example} のように型名 + アンダースコアを接頭辞にします。').format(
                      name = node.spelling, example = prefix + 'EnumConstant'),
                      source, node.location.line)

    def _check_macro(self, token, source):
        if not self._validate(token.spelling, self._PATTERN_SCREAMING_SNAKE, prefix = 'NN_'):
            error('name/macro', ERROR_LEVEL_REQUIRED, _(
                  u'マクロ {name} は命名規則に違反しています。'\
                  u'{example} のように名前空間に相当する単語を連結して接頭辞とします。'\
                  u'ただし、非公開のシンボルであることを表す DETAIL は常に 2 番目です。').
                  format(name    = token.spelling,
                         example = 'NN_MODULE_MACRO, NN_DETAIL_MODULE_MACRO'),
                  source, token.location.line)

    def _check_const(self, node, source):
        if len(self.nest.functions) > 0 and not is_static(node):
            self._check_auto_var(node, source)
        elif not self._validate(node.spelling, self._PATTERN_PASCAL):
            error('name/const', ERROR_LEVEL_REQUIRED,
                _(u'定数 {name} は命名規則 {rule} に違反しています。 ').
                  format(name = node.spelling, rule = 'PascalCase'),
                  source, node.location.line)

    def _check_var(self, node, source):
        if len(self.nest.functions) > 0:
            if is_static(node):
                self._check_function_local_var(node, source)
            else:
                self._check_auto_var(node, source)
        elif len(self.nest.classes) > 0:
            self._check_class_var(node, source)
        elif self.nest.is_anonymous or is_static(node):
            self._check_file_local_var(node, source)
        elif len(self.nest.namespaces) > 0:
            self._check_namespace_var(node, source)
        elif find_child(node, cindex.CursorKind.TYPE_REF):
            self._check_class_var(node, source)
        else:
            self._check_global_var(node, source)

    def _check_function_local_var(self, node, source):
        if not self._validate(node.spelling, self._PATTERN_PASCAL, prefix = 's_p?'):
            error('name/variable', ERROR_LEVEL_REQUIRED,
                _(u'関数ローカルな静的変数 {name} は命名規則 {rule} に違反しています。 ').
                  format(name = node.spelling, rule = 's_PascalCase'),
                  source, node.location.line)

    def _check_auto_var(self, node, source):
        if '_' in node.spelling:
            error('name/variable', ERROR_LEVEL_REQUIRED,
                _(u'自動変数の名前 {name} にアンダースコアを使用することはできません。').
                  format(name = node.spelling),
                  source, node.location.line)
        elif not self._validate(node.spelling, self._PATTERN_CAMEL):
            error('name/variable', ERROR_LEVEL_RECOMMENDED,
                _(u'自動変数 {name} は命名規則 {rule} に違反しています。 ').format(
                  name = node.spelling, rule = 'camelCase'),
                  source, node.location.line)

    def _check_class_var(self, node, source):
        if not self._validate(node.spelling, self._PATTERN_PASCAL, prefix = 'g_p?'):
            error('name/variable', ERROR_LEVEL_REQUIRED,
                _(u'クラス変数 {name} は命名規則 {rule} に違反しています。 ').
                  format(name = node.spelling, rule = 'g_PascalCase'),
                  source, node.location.line)

    def _check_file_local_var(self, node, source):
        if not self._validate(node.spelling, self._PATTERN_PASCAL, prefix = 'g_p?'):
            error('name/variable', ERROR_LEVEL_REQUIRED,
                _(u'ファイルローカルな変数 {name} は命名規則 {rule} に違反しています。 ').
                  format(name = node.spelling, rule = 'g_PascalCase'),
                  source, node.location.line)

    def _check_namespace_var(self, node, source):
        if not self._validate(node.spelling, self._PATTERN_PASCAL, prefix = 'g_p?'):
            error('name/variable', ERROR_LEVEL_REQUIRED,
                _(u'名前空間内の外部変数 {name} は命名規則 {rule} に違反しています。 ').
                  format(name = node.spelling, rule = 'g_PascalCase'),
                  source, node.location.line)

    def _check_global_var(self, node, source):
        if ((not self._validate(node.spelling, self._PATTERN_GLOBAL_SYMBOL)) and
            (not self._validate(node.spelling, self._PATTERN_PASCAL, prefix = 'g_p?'))):
            error('name/variable', ERROR_LEVEL_REQUIRED, _(
                  u'名前空間外の外部変数 {name} は命名規則に違反しています。 '\
                  u'{example} のように本来の名前空間に相当する単語を先頭に付加します。'\
                  u'ただし、非公開のシンボルであることを表す detail は常に 2 番目です。').
                  format(name    = node.spelling,
                         example = 'nnmoduleVariable, nndetailModuleVariable'),
                  source, node.location.line)

    def _check_param(self, node, source):
        if '_' in node.spelling:
            error('name/param', ERROR_LEVEL_REQUIRED,
                _(u'仮引数の名前 {name} にアンダースコアを使用することはできません。').
                  format(name = node.spelling),
                  source, node.location.line)
        elif not self._validate(node.spelling, self._PATTERN_CAMEL):
            error('name/param', ERROR_LEVEL_RECOMMENDED,
                _(u'仮引数 {name} は命名規則 {rule} に違反しています。 ').format(
                  name = node.spelling, rule = 'camelCase'),
                  source, node.location.line)

    def _check_field(self, node, source):
        spec = node.access_specifier.name.lower()
        if spec == "private":
            if not self._validate(node.spelling, self._PATTERN_PASCAL, prefix = 'm_p?'):
                error('name/field', ERROR_LEVEL_REQUIRED,
                    _(u'private なメンバ変数 {name} は命名規則 {rule} に違反しています。 ').
                      format(name = node.spelling, rule = 'm_PascalCase'),
                      source, node.location.line)
        elif spec == 'protected':
            if not self._validate(node.spelling, self._PATTERN_CAMEL):
                error('name/field', ERROR_LEVEL_REQUIRED,
                    _(u'protected なメンバ変数 {name} は命名規則 {rule} に違反しています。 ').
                      format(name = node.spelling, rule = 'camellCase'),
                      source, node.location.line)
        elif is_struct(self.nest.classes[-1]):
            if not self._validate(node.spelling, self._PATTERN_CAMEL, prefix = '_?'):
                error('name/field', ERROR_LEVEL_REQUIRED, _(
                      u'public なメンバ変数 {name} は命名規則 {rule} に違反しています。 '\
                      u'ユーザのアクセスを意図しない構造体メンバには、'\
                      u'接頭辞にアンダースコアを付加することも認められています。').
                      format(name = node.spelling, rule = 'camellCase'),
                      source, node.location.line)
        else:
            if not self._validate(node.spelling, self._PATTERN_CAMEL):
                error('name/field', ERROR_LEVEL_REQUIRED, _(
                      u'public なメンバ変数 {name} は命名規則 {rule} に違反しています。 ').
                      format(name = node.spelling, rule = 'camellCase'),
                      source, node.location.line)

    def _validate(self, name, pattern, prefix = None, ignores = []):

        # テンプレートの < 以降は削除します。
        match = re.match(r'^([^<]*)(<.+)$', name)
        if match:
            name = match.group(1)

        # 無視リストを検証します。空の名前も認めます。
        if not name or name in ignores:
            return True

        # 接頭辞の規約チェックです。
        if prefix:
            prefix  = '^(' + prefix +')'
            match   = re.match(prefix, name)
            if not match:
                return False
            name = name[len(match.group(1)):]

        # 接頭辞を除く名前本体の規約チェックです。
        return pattern.match(name)

class BlacklistNameChecker(Checker):
    """
    禁止された名前を使用していないことを検証します。
    """

    _RESERVED_C = [
        "char", "short", "int", "long", "signed", "unsigned", "float", "double", "struct",
        "union", "enum", "volatile", "const", "restrict", "auto", "extern", "static",
        "register", "inline", "sizeof", "typedef", "void", "if", "else", "switch", "case",
        "default", "for", "while", "do", "goto", "continue", "break", "return"
    ]

    _RESERVED_CPP = [
        "asm", "bool", "true", "false", "wchar_t", "class", "mutable", "friend", "explicit",
        "virtual", "public", "protected", "private", "operator", "this", "try", "catch",
        "new", "delete", "dynamic_cast", "static_cast", "const_cast", "reinterpret_cast",
        "typeid", "throw", "template", "typename", "export", "namespace", "using",
        "nullptr", "char16_t", "char32_t", "thread_local", "constexpr", "alignof",
        "decltype", "noexcept", "static_assert", "alignas", "final", "override"
    ]

    _ALTERNATIVE = [
        "and", "and_eq", "bitand", "bitor", "compl", "not", "not_eq", "or",
        "or_eq", "xor", "xor_eq"
    ]

    _STDLIB = [
        "std", "string", "vector", "map", "bitset", "errno"
    ]

    def __init__(self):
        self.filter = set(self._RESERVED_C + self._RESERVED_CPP + self._ALTERNATIVE + self._STDLIB)

    def hook_node_enter(self, node, source, skip):
        if skip:
            return
        if node.kind in [ cindex.CursorKind.NAMESPACE,
                          cindex.CursorKind.FUNCTION_DECL,
                          cindex.CursorKind.FUNCTION_TEMPLATE,
                          cindex.CursorKind.CXX_METHOD,
                          cindex.CursorKind.CLASS_DECL,
                          cindex.CursorKind.STRUCT_DECL,
                          cindex.CursorKind.CLASS_TEMPLATE,
                          cindex.CursorKind.CLASS_TEMPLATE_PARTIAL_SPECIALIZATION,
                          cindex.CursorKind.ENUM_DECL,
                          cindex.CursorKind.TYPEDEF_DECL,
                          cindex.CursorKind.UNION_DECL,
                          cindex.CursorKind.TEMPLATE_TYPE_PARAMETER,
                          cindex.CursorKind.ENUM_CONSTANT_DECL,
                          cindex.CursorKind.VAR_DECL,
                          cindex.CursorKind.FIELD_DECL ]:
            if node.spelling in self.filter:
                error('name/reserved', ERROR_LEVEL_REQUIRED,
                    _(u'名前 {name} は使用を禁止されています。').
                      format(name = node.spelling),
                      source, node.location.line)

class BooleanNamingChecker(Checker):
    """
    真偽値変数の命名を検証します。
    """

    _BOOL_PATTERN = re.compile(r'^bool[\s*&]*$')
    _NAMING_PATTERN = re.compile(r'^(m_|s_|g_|_)?p?(is|Is|IS|has|Has|HAS|can|Can|CAN)')

    def __init__(self):
        self.nest   = NestingState()

    def hook_setup(self, source):
        self.nest.reset()

    def hook_node_enter(self, node, source, skip):
        if ((not skip) and
            (node.kind in [cindex.CursorKind.VAR_DECL,
                           cindex.CursorKind.PARM_DECL,
                           cindex.CursorKind.FIELD_DECL]) and
            (self._BOOL_PATTERN.match(node.type.spelling)) and
            (not self.nest.is_global or is_static(node))):
            if len(node.spelling) > 0 and not self._NAMING_PATTERN.match(node.spelling):
                error('name/boolean', ERROR_LEVEL_REQUIRED, _(
                     u'真偽値変数 {name} は命名規則に違反しています。'\
                     u'接頭辞に {prefix} のいずれかを指定してください').format(
                     name = node.spelling, prefix = 'is, has, can'),
                     source, node.location.line)
        self.nest.enter(node)

    def hook_node_leave(self, node, source, skip):
        self.nest.leave(node)

class CountNamingChecker(Checker):
    """
    〇〇の数に対する命名規則を検証します。
    """

    _PATTERN_NAME     = re.compile(r'(Num)(Min|Max)?$')
    _PATTERN_MIN_MAX  = re.compile(r'(Min|Max)(Count|Length|Num|Size)$')

    def hook_node_enter(self, node, source, skip):
        if skip:
            return
        if node.kind in [ cindex.CursorKind.FUNCTION_DECL,
                          cindex.CursorKind.FUNCTION_TEMPLATE,
                          cindex.CursorKind.CXX_METHOD,
                          cindex.CursorKind.VAR_DECL ]:
            self._check_impl(node, source)

    def _check_impl(self, node, source):
        match_name   = self._PATTERN_NAME.search(node.spelling)
        match_minmax = self._PATTERN_MIN_MAX.search(node.spelling)
        if match_name:
            error('name/count', ERROR_LEVEL_INFORMATION, _(
                  u'数に対する命名では意味が曖昧な {name} よりも '\
                  u'{example} などを使用することが推奨されています。').format(
                  name = match_name.group(1), example = 'Count, Length'),
                  source, node.location.line)
        elif match_minmax:
            minmax = match_minmax.group(1)
            count  = match_minmax.group(2)
            error('name/count', ERROR_LEVEL_INFORMATION, _(
                  u'最大数や最小数に対する命名では {name} よりも '\
                  u' {example} を使用することが推奨されています。').format(
                  name = minmax + count, example = count + minmax),
                  source, node.location.line)

class Utf8BomChecker(Checker):
    """
    ソースファイルの BOM の有無を確認します。
    """

    def __init__(self, isBomRequired):
        """
        Args:
            isBomRequired:  BOM の指定を必須にする場合は BOM_REQUIRED を、
                            BOM の指定を禁止する場合は BOM_FORBIDDEN を指定します。
        """
        self.isBomRequired = isBomRequired

    def hook_begin(self, source):
        if self.isBomRequired:
            if source.raw_body[0] != u'\ufeff':
                error('style/encoding', ERROR_LEVEL_REQUIRED,
                    _(u'ソースファイルのエンコーディングは UTF-8 (BOM 付き) に設定してください。'),
                      source)
        else:
            if source.raw_body[0] == u'\ufeff':
                error('style/encoding', ERROR_LEVEL_REQUIRED,
                    _(u'ソースファイルのエンコーディングは UTF-8 (BOM 無し) に設定してください。'),
                      source)

class LinefeedChecker(Checker):
    """
    改行コードが意図した設定になっていることを確認します。
    """

    def __init__(self, linefeed):
        """
        Args:
            linefeed:       要求する改行コードとして LINEFEED_CR, LINEFEED_LF, LINEFEED_CRLF
                            の中から 1 つを選択して指定します。
        """
        self.linefeed = linefeed
        self.has_error = False

    def hook_begin(self, source):
        self.has_error = False

    def hook_line(self, linenum, raw_line, line, simple_line, source):

        # このエラーは 1 つのファイルに対して 1 度だけ表示します。
        if self.has_error:
            return

        # 改行コードの種類に応じた判定です。
        linefeed = None
        if self.linefeed == LINEFEED_CR:
            if raw_line.endswith('\n'):
                linefeed = u'CR (\\r)'
        elif self.linefeed == LINEFEED_LF:
            if raw_line.endswith('\r') or line.endswith('\r\n'):
                linefeed = u'LF (\\n)'
        elif self.linefeed == LINEFEED_CRLF:
            if (raw_line.endswith('\r') or
               (raw_line.endswith('\n') and not raw_line.endswith('\r\n'))):
                linefeed = u'CRLF (\\r\\n)'

        # エラーメッセージを出力します。
        if linefeed:
            error('style/linefeed', ERROR_LEVEL_REQUIRED,
                _(u'改行コードには {linefeed} を使用してください。').format(linefeed = linefeed),
                  source)
            self.has_error = True

class IndentChecker(Checker):
    """
    インデントに指定された方法が使用されていることを確認します。
    """

    _PATTERN_PREPROCESSOR_FORBIDDEN_INDENT = re.compile(r'#\s+(\w+)')

    def __init__(self, indent_type, width = 0):
        """
        Args:
            indent_type:    インデントの種類を INDENT_TAB, INDENT_SPACE から選択します。
            width:          INDENT_SPACE の場合に、インデント幅を指定します。
        """
        self.indent_type = indent_type
        self.width       = width

    def hook_line(self, linenum, raw_line, line, simple_line, source):

        # 行頭のインデントの規約チェックです。
        has_error = None
        if self.indent_type == INDENT_TAB:
            has_error = self._check_tab(linenum, simple_line, source)
        else:
            has_error = self._check_space(linenum, simple_line, self.width, source)

        # プリプロセッサの # 前後のインデントの規約チェックです。
        if not has_error:
            self._check_preprocessor(linenum, simple_line, source)

    def _check_tab(self, linenum, line, source):

        # 行の先頭が半角空白で始まっている場合はインデントに空白が使用されているとみなします。
        if 0 < len(line) and line[0] == ' ' and line[-1] != ' ':
            error('style/indent', ERROR_LEVEL_REQUIRED,
                _(u'インデントにはタブを使用してください。'),
                  source, linenum)
            return True

    def _check_space(self, linenum, line, width, source):

        # タブが含まれないことを確認します。
        if line.find('\t') >= 0:
            error('style/indent', ERROR_LEVEL_REQUIRED,
                _(u'インデントには {width} 文字の半角空白を使用してください。').format(width=width),
                  source, linenum)
            self.has_error = True
            return True

        # 引数や初期化リスト、式の途中での改行を例外として考えるのであれば
        # 厳密なインデントの検証は難しいので、行頭の空白数が width 未満でないことを確認します。
        line_width  = len(line)
        space_count = 0
        while space_count < line_width and line[space_count] == ' ':
            space_count += 1
        if 0 < space_count and space_count < width:
            error('style/indent', ERROR_LEVEL_REQUIRED,
                _(u'インデントには {width} 文字の半角空白を使用してください。').format(width=width),
                  source, linenum)
            self.has_error = True
            return True

    def _check_preprocessor(self, linenum, line, source):
        match = self._PATTERN_PREPROCESSOR_FORBIDDEN_INDENT.search(line)
        if match:
            error('style/indent', ERROR_LEVEL_RECOMMENDED,
                _(u'プリプロセッサ #{preprocessor} は # の後ろでインデントされています。').
                  format(preprocessor = match.group(1)),
                  source, linenum)
            return True

class BinaryOperatorSpacingChecker(Checker):
    """
    二項演算子の前後のスペーシングを確認します。
    """

    def __init__(self, operators):
        """
        Args:
            operators:      規約チェックの対象となる演算子のリストを指定します。
        """
        self.operators = operators

    def hook_token(self, token, source):
        if (token.cursor.kind == cindex.CursorKind.BINARY_OPERATOR and
            token.spelling in self.operators):
            start     = token.extent.start
            end       = token.extent.end
            prev_char = source.body[start.offset-1]
            next_char = source.body[end.offset]
            if prev_char != ' ' or next_char not in [' ', "\n", "\r"]:
                error('style/spacing', ERROR_LEVEL_REQUIRED,
                    _(u'二項演算子 {op} の前後には空白が必要です。').format(op=token.spelling),
                      source, token.location.line)

class NewlineBraceChecker(Checker):
    """
    波括弧が新しい行に単独で配置されていることを確認します。
    """

    class BraceState(object):
        def __init__(self):
            self.open_prev   = None
            self.open_brace  = None
            self.open_next   = None
            self.close_prev  = None
            self.close_brace = None
            self.close_next  = None
            self.keyword     = None

    def __init__(self):
        self.prev    = None
        self.keyword = None
        self.stack   = []

    def hook_begin(self, source):
        self.prev    = None
        self.keyword = None
        del self.stack[:]

    def hook_end(self, source):
        if len(self.stack) > 0:
            self.stack[-1].close_next = None
            self._check_brace(source.simple_lines, self.stack[-1], source)
            self.stack.pop()

    def hook_token(self, token, source):

        # 波括弧に対応するキーワードを記憶しておきます。
        if token.kind == cindex.TokenKind.KEYWORD:
            self.keyword = token
        elif token.spelling == '}' or token.spelling == ';':
            self.keyword = None

        # 波括弧の次のトークンに対する処理です。
        if self.prev and self.prev.kind == cindex.TokenKind.PUNCTUATION:
            if self.prev.spelling.endswith('{'):
                self.stack[-1].open_next = token
            elif self.prev.spelling.endswith('}'):
                self.stack[-1].close_next = token
                self._check_brace(source.simple_lines, self.stack[-1], source)
                self.stack.pop()

        # 波括弧トークンに対する処理です。
        if token.kind == cindex.TokenKind.PUNCTUATION:
            if token.spelling.endswith('{'):
                self.stack.append(self.BraceState())
                self.stack[-1].open_prev   = self.prev
                self.stack[-1].open_brace  = token
                self.stack[-1].keyword     = self.keyword
            elif token.spelling.endswith('}'):
                self.stack[-1].close_prev  = self.prev
                self.stack[-1].close_brace = token

        self.prev = token

    def _check_brace(self, lines, state, source):

        # 式を形成する波括弧 { } は規約の対象外とします。
        if ((state.open_prev.spelling == '=') or
            (state.open_prev.spelling == ',') or
            (state.close_next and state.close_next.spelling == ',')):
            return

        # 波括弧 { } の中が空で同じ行に書かれている場合にも規約の対象外です。
        if ((state.open_next.spelling == '}') and
            (state.open_brace.extent.start.line == state.close_brace.extent.start.line)):
            return

        # 名前空間に対する波括弧 { } は規約の対象外です。
        keyword = state.keyword.spelling if state.keyword else  ''
        if keyword == 'namespace':
            return

        # 波括弧 { の規約チェックです。新しい行に配置されていることを確認します。
        prev  = state.open_prev
        brace = state.open_brace
        next  = state.open_next
        n1    = prev.extent.end.line
        n2    = brace.extent.end.line
        n3    = next.extent.end.line
        line  = lines[n1]
        if n1 == n2 or (n2 == n3 and next.kind != cindex.TokenKind.COMMENT):
            error('style/brace', ERROR_LEVEL_REQUIRED,
                _(u"波括弧 { は新しい行に配置してください。"), source, n2)
            return

        # 波括弧 } の規約チェックです。新しい行に配置されていることを確認します。
        # ただし、 do-while 文の while は波括弧 } の後ろに置くべきで、
        # セミコロンは波括弧 } の後ろに置いても構いません。
        prev  = state.close_prev
        brace = state.close_brace
        next  = state.close_next
        n1    = prev.extent.end.line
        n2    = brace.extent.end.line
        n3    = next.extent.end.line if next else 0
        line  = lines[n2]
        if n1 == n2:
            error('style/brace', ERROR_LEVEL_REQUIRED,
                _(u"波括弧 } は新しい行に配置してください。"), source, n2)
        elif n2 == n3:
            if ((keyword != 'do') and
                (next is None or next.spelling != ';') and
                (next is None or next.kind != cindex.TokenKind.COMMENT) and
                (keyword not in ['class', 'enum', 'struct', 'union'])):
                error('style/brace', ERROR_LEVEL_REQUIRED,
                    _(u"波括弧 } は新しい行に配置してください。"), source, n2)
        elif n2 < n3 and keyword == 'do':
            error('style/brace', ERROR_LEVEL_REQUIRED,
                _(u"do-while 文の while は波括弧 } と同じ行に配置してください。"),
                  source, n3)

class EofLineChecker(Checker):
    """
    ファイルの末尾が改行で終わっていることを確認します。
    """

    def hook_begin(self, source):
        line_num   = len(source.raw_lines)
        last_line  = source.raw_lines[line_num - 1]
        if not last_line.endswith('\r') and not last_line.endswith('\n'):
            error('style/eof', ERROR_LEVEL_REQUIRED,
                _(u'ファイルの末尾は改行でなければなりません。'),
                  source, line_num - 1)

class LineWidthChecker(Checker):
    """
    1 行の長さが指定された文字数を超えていないことを確認します。
    """

    def __init__(self, max_width):
        """
        Args:
            max_width:      規約で認められる 1 行の最大長を指定します。
                            ASCII は 1 文字、それ以外は 2 文字としてカウントします。
        """
        self.max_width = max_width

    def hook_line(self, linenum, raw_line, line, simple_line, source):
        if self.count_width(line) > self.max_width:
            error('style/linewidth', ERROR_LEVEL_RECOMMENDED,
                _(u'1 行の長さが {width} 文字を超えています。').format(width = self.max_width),
                  source, linenum)

    def count_width(self, line):
        width = 0
        for c in line.decode('utf_8'):
            if ord(c) <= '\u007e':  # ASCII
                width = width + 1
            else:                   # not ASCII
                width = width + 2
        return width

class CopyrightChecker(Checker):
    """
    著作権ヘッダの有無を確認します。
    """

    def __init__(self, author, min_line, max_line):
        """
        Args:
            author:         著作者です。この名前が著作権ヘッダに含まれる必要があります。
            min_line:       著作権表記を探索する範囲を 1 から始まる行番号で指定します。
            max_line:       著作権表記を探索する範囲を 1 から始まる行番号で指定します。
        """
        self.author        = author
        self.min_line      = min_line
        self.max_line      = max_line
        self.pattern       = re.compile(r'\bCopyright\b.+\b%s\b' % author)
        self.has_copyright = False

    def hook_begin(self, source):
        self.has_copyright = False

    def hook_line(self, linenum, raw_line, line, simple_line, source):
        if self.min_line <= linenum and linenum <= self.max_line:
            if self.pattern.search(line):
                self.has_copyright = True

    def hook_end(self, source):
        if not self.has_copyright:
            error('style/copyright', ERROR_LEVEL_REQUIRED,
                _(u'ファイルの先頭に {author} を含む著作権表記がありません。').format(
                  author = self.author), source)

class IncludeGuardChecker(Checker):
    """
    インクルードガードの有無を確認します。
    """

    _PATTERN_PRAGMA = re.compile(r'#\s*pragma\s+once\b')

    def __init__(self):
        self.is_header = False
        self.count     = 0

    def hook_begin(self, source):
        name, ext      = os.path.splitext(source.path)
        self.is_header = ext in HEADER_EXTENSIONS
        self.count     = 0

    def hook_line(self, linenum, raw_line, line, simple_line, source):
        if self._PATTERN_PRAGMA.search(simple_line):
            self.count += 1

    def hook_end(self, source):
        if self.is_header and self.count == 0:
            error('style/include_guard', ERROR_LEVEL_REQUIRED,
                _(u'インクルードガード #pragma once が見つかりませんでした。'), source)

class MultipleVariableDeclarationChecker(Checker):
    """
    1 行で複数の変数を宣言していないことを確認します。
    """

    def __init__(self):
        self.stack     = []
        self.linenum   = 0

    def hook_setup(self, source):
        del self.stack[:]
        self.linenum   = 0

    def hook_node_enter(self, node, source, skip):
        count = len(self.stack)
        if 0 < count and self.linenum != node.location.line:
            if not skip and 2 <= count:
                error('style/var_decl', ERROR_LEVEL_REQUIRED,
                    _(u'1 行で {count} 個の変数 {vars} が定義されました。'\
                      u'1 つの文で複数の変数を宣言しないでください。'\
                      u'また、複数の宣言文を同じ行に書くこともできません。').
                      format(count = count, vars = ', '.join(self.stack)),
                      source, self.linenum)
            del self.stack[:]
        if node.kind == cindex.CursorKind.VAR_DECL:
            self.stack.append(node.spelling)
            self.linenum = node.location.line

class SwitchDefaultChecker(Checker):
    """
    switch 文における default の省略を禁止します。
    """

    class SwitchState(object):
        def __init__(self):
            self.has_default = False

    def __init__(self):
        self.stack = []

    def hook_setup(self, source):
        del self.stack[:]

    def hook_node_enter(self, node, source, skip):
        if node.kind == cindex.CursorKind.SWITCH_STMT:
            self.stack.append(self.SwitchState())
        elif node.kind == cindex.CursorKind.DEFAULT_STMT:
            self.stack[-1].has_default = True

    def hook_node_leave(self, node, source, skip):
        if node.kind == cindex.CursorKind.SWITCH_STMT:
            if not skip and not self.stack[-1].has_default:
                error('style/switch_default', ERROR_LEVEL_REQUIRED, _(
                      u'switch 文の default を省略することは禁止されています。 '\
                      u'default ラベルに到達しないことが明らかであっても、'\
                      u'その表明のために NN_UNEXPECTED_DEFAULT を '\
                      u'default の直後に記述してください。'),
                      source, node.location.line)
            self.stack.pop()

class SwitchFallthroughChecker(Checker):
    """
    switch 文における fall-through を禁止します。
    """

    _PATTERN = re.compile(r'\b(switch|case|break|return|NN_FALL_THROUGH)\b')

    def hook_warning(self, warning, source):
        if warning.option == '-Wimplicit-fallthrough':
            if not self.is_explicit(warning, source.simple_lines):
                error('style/fall_through', ERROR_LEVEL_REQUIRED, _(
                      u'switch 文の fall-through は原則禁止されています。'\
                      u'やむをえない場合は NN_FALL_THROUGH マクロを記述子、'\
                      u'意図を明示してください。'),
                      source, warning.location.line)

    def is_explicit(self, warning, lines):
        i = warning.location.line - 1
        while i > 0:
            match = self._PATTERN.search(lines[i])
            if match:
                return match.group(1) == 'NN_FALL_THROUGH'
            i -= 1

class AccessSpecifierChecker(Checker):
    """
    以下の規約チェックを実施します。
    * アクセス修飾子の記述順序が public, protected, private の順になっていること
    * クラスにおいてアクセス修飾子が省略されていないこと (構造体は規約の対象外)
    """

    class ClassState(object):
        def __init__(self, is_class):
            self.is_class  = is_class
            self.specifier = ACCESS_SPECIFIER_NONE
            self.has_error = False

    def __init__(self):
        self.stack = []

    def hook_setup(self, source):
        del self.stack[:]

    def hook_node_enter(self, node, source, skip):

        if (not skip and
              len(self.stack) > 0 and
              self.stack[-1].is_class and
              self.stack[-1].specifier == ACCESS_SPECIFIER_NONE and
              not self.stack[-1].has_error and
              node.kind in [cindex.CursorKind.VAR_DECL,
                            cindex.CursorKind.FIELD_DECL,
                            cindex.CursorKind.FUNCTION_TEMPLATE,
                            cindex.CursorKind.CXX_METHOD,
                            cindex.CursorKind.CONVERSION_FUNCTION,
                            cindex.CursorKind.CLASS_DECL,
                            cindex.CursorKind.CLASS_TEMPLATE_PARTIAL_SPECIALIZATION,
                            cindex.CursorKind.STRUCT_DECL,
                            cindex.CursorKind.CLASS_TEMPLATE,
                            cindex.CursorKind.UNION_DECL,
                            cindex.CursorKind.TYPEDEF_DECL,
                            cindex.CursorKind.ENUM_DECL]):
            error('style/access_specifier', ERROR_LEVEL_REQUIRED,
                _(u'クラス定義で private の記述を省略することは禁止されています。'),
                  source, node.location.line)
            self.stack[-1].has_error = True

        if is_class(node):
            self.stack.append(self.ClassState(True))
        elif is_struct(node) or is_union(node):
            self.stack.append(self.ClassState(False))
        elif node.kind == cindex.CursorKind.CXX_ACCESS_SPEC_DECL:
            prev = self.stack[-1].specifier
            self.stack[-1].specifier = convert_node_to_specifier(node)
            if not skip and self.stack[-1].specifier < prev:
                error('style/access_specifier', ERROR_LEVEL_RECOMMENDED,
                    _(u'アクセス修飾子は public, protected, private の順に記述してください。'),
                      source, node.location.line)

    def hook_node_leave(self, node, source, skip):
        if is_class(node) or is_struct(node):
            self.stack.pop()

class TemplateKeywordChecker(Checker):
    """
    テンプレート初期化子で class を使用せず typename を使用していることを確認します。
    """

    def hook_node_enter(self, node, source, skip):
        if not skip and node.kind == cindex.CursorKind.TEMPLATE_TYPE_PARAMETER:
            tokens  = get_tokens(node)
            keyword = tokens[0].spelling if tokens else ''
            if keyword == 'class':
                error('style/typename', ERROR_LEVEL_RECOMMENDED,
                    _(u'テンプレート初期化子では class ではなく typename を使用してください。'),
                      source, node.location.line)

class AlternativeTokenChecker(Checker):
    """
    代替表記の使用を禁止します。
    """

    _ERROR_PATTERN = re.compile(r'^\s*#\s*error\b')

    def hook_token(self, token, source):
        if self._is_alternative_token(token, source):
            error('style/alt_tokens', ERROR_LEVEL_REQUIRED,
                _(u'代替トークン {token} の使用は禁止されています。').format(
                  token = token.spelling),
                  source, token.location.line)

    def _is_alternative_token(self, token, source):

        # error プリプロセッサはメッセージ内の誤検出を防止するため無視します。
        if self._ERROR_PATTERN.search(source.simple_lines[token.location.line]):
            return False

        if token.kind == cindex.TokenKind.PUNCTUATION:
            return token.spelling in ['??(', '??)', '<:', ':>', '<%', '%>', '%:', '%:%:']
        elif token.kind == cindex.TokenKind.KEYWORD:
            return token.spelling in ['and', 'bitor', 'or', 'xor', 'compl', 'bitand',
                                      'and_eq', 'or_eq', 'xor_eq', 'not', 'not_eq']
        return False

class ForbiddenTypeChecker(Checker):
    """
    禁止されている型を使用していないことを確認します。
    """

    _PATTERN = re.compile(
        r'\b(short|long|signed|unsigned|wchar_t|intptr_t|u?intmax_t|'\
           r'u?int_least\d+_t|u?int_fast\d+_t)\b')

    def hook_node_enter(self, node, source, skip):
        if skip:
            return
        if node.kind in [ cindex.CursorKind.VAR_DECL,
                          cindex.CursorKind.PARM_DECL,
                          cindex.CursorKind.FIELD_DECL ]:
            self._check_impl(node, source)

    def _check_impl(self, node, source):
        if self._PATTERN.match(node.type.spelling):
            error('type/buildin', ERROR_LEVEL_REQUIRED,
                _(u'型 {type} の使用は禁止されています。').format(
                  type = node.type.spelling),
                  source, node.location.line)

class BitFieldChecker(Checker):
    """
    ビットフィールドを使用していないことを確認します。
    """

    _PADDING_PATTERN = re.compile(r'\bNN_PADDING[1-8]\b')

    def hook_node_enter(self, node, source, skip):
        if skip:
            return
        if node.kind == cindex.CursorKind.FIELD_DECL and node.is_bitfield():
            self._check_impl(node, source)

    def _check_impl(self, node, source):
        line = source.simple_lines[node.location.line]
        if not self._PADDING_PATTERN.search(line):
            error('type/bitfield', ERROR_LEVEL_REQUIRED, _(
                  u'移植性を確保するためビットフィールドの使用は禁止されています。'\
                  u'メンバ変数 {name} の定義を確認してください。').format(
                  name = node.spelling),
                  source, node.location.line)

class ParamReassignmentChecker(Checker):
    """
    仮引数に値を再割り当てしていないことを検証します。
    """

    class FunctionState(object):
        def __init__(self):
            self.node   = None
            self.params = []

    def __init__(self):
        self.stacks = []

    def hook_setup(self, source):
        del self.stacks[:]

    def hook_node_enter(self, node, source, skip):
        if is_function_like(node):
            self.stacks.append(self.FunctionState())
            self.stacks[-1].node = node
            for child in get_children(node, kind = cindex.CursorKind.PARM_DECL):
                self.stacks[-1].params.append(child)
        elif node.kind == cindex.CursorKind.BINARY_OPERATOR and not skip:
            op   = None
            decl = find_child(node, cindex.CursorKind.DECL_REF_EXPR)
            for token in node.get_tokens():
                if token.spelling == '=':
                    op = token
            if op and decl and 0 < len(self.stacks):
                self._check_impl(self.stacks[-1], decl, source)
        elif node.kind == cindex.CursorKind.COMPOUND_ASSIGNMENT_OPERATOR and not skip:
            decl = find_child(node, cindex.CursorKind.DECL_REF_EXPR)
            if decl and 0 < len(self.stacks):
                self._check_impl(self.stacks[-1], decl, source)

    def hook_node_leave(self, node, source, skip):
        if is_function_like(node):
            self.stacks.pop()

    def _check_impl(self, func, node, source):
        for param in func.params:
            if param.spelling == node.spelling:
                error('variable/reassign', ERROR_LEVEL_RECOMMENDED,
                    _(u'仮引数 {name} の変更は推奨されません。').format(
                      name = node.spelling),
                      source, node.location.line)

class ReferenceChecker(Checker):
    """
    非 const 参照の引数を利用していないことを検証します。
    """

    def hook_node_enter(self, node, source, skip):
        if skip:
            return
        if node.kind == cindex.CursorKind.PARM_DECL:
            self._check_impl(node, source)

    def _check_impl(self, node, source):
        name = node.type.spelling
        if name.endswith('&') and not name.endswith('&&') and not name.startswith('const'):
            error('function/references', ERROR_LEVEL_REQUIRED,
                _(u'仮引数 {name} は非 const 参照です。').format(name = node.spelling),
                  source, node.location.line)

class DefaultParamChecker(Checker):
    """
    デフォルト引数を利用していないことを検証します。
    """

    def hook_node_enter(self, node, source, skip):
        if skip:
            return
        if node.kind == cindex.CursorKind.PARM_DECL:
            self._check_impl(node, source)

    def _check_impl(self, node, source):
        for token in node.get_tokens():
            if token.spelling == '=':
                error('function/default', ERROR_LEVEL_RECOMMENDED, _(
                      u'デフォルト引数の使用は推奨されません。'\
                      u'引数 {name} の定義を確認してください。').format(name = node.spelling),
                      source, node.location.line)

class NoexceptChecker(Checker):
    """
    無例外指定の有無を検証します。
    """

    def hook_node_enter(self, node, source, skip):
        if skip:
            return
        if is_function_like(node):
            self._check_impl(node, source)

    def _check_impl(self, node, source):
        has_body = False
        for token in node.get_tokens():
            # AST に無関係な NN_NOEXCEPT はマクロ展開されない？
            if token.kind == cindex.TokenKind.IDENTIFIER and token.spelling == 'NN_NOEXCEPT':
                return
            elif token.spelling == '(':
                has_body = True
            elif token.spelling == '{':
                break

        # 正しく関数のトークンを取得できた場合にだけ検証を行います。
        if has_body:
            error('function/noexcept', ERROR_LEVEL_REQUIRED,
                 _(u'関数 {name} に対する NN_NOEXCEPT が見つかりません。').format(
                  name = node.spelling), source, node.location.line)

class PodChecker(Checker):
    """
    struct が POD (Plain Old Data) 型であることを確認します。
    """

    def hook_node_enter(self, node, source, skip):
        if skip:
            return
        if node.kind == cindex.CursorKind.STRUCT_DECL:
            children = get_children(node)
            if (not node.type.is_pod() and len(children) > 0):
                error('class/nonpod', ERROR_LEVEL_REQUIRED, _(
                      u'構造体 {name} は non-POD 型です。'\
                      u'構造体は POD 型でなければなりません。').format(name = node.spelling),
                      source, node.location.line)

class ExplicitConstructorChecker(Checker):
    """
    一引数コンストラクタが explicit, NN_IMPLICIT のいずれかをもつことを検証します。
    """

    def __init__(self):
        self.stack = []

    def hook_setup(self, source):
        del self.stack[:]

    def hook_node_enter(self, node, source, skip):
        # 規約チェックの対象となる一引数のコンストラクタを探します。
        # ただし、ここではコピーコンストラクタやムーブコンストラクタを対象から外します。
        if is_class_like(node):
            self.stack.append(node)
        elif not skip and node.kind == cindex.CursorKind.CONSTRUCTOR and len(self.stack) > 0:
            params = get_children(node, kind = cindex.CursorKind.PARM_DECL)
            if len(params) == 1:
                type = params[0].type
                if node.spelling not in type.spelling:
                    self._check_explicit(node, source)

    def hook_node_leave(self, node, source, skip):
        if is_class_like(node):
            self.stack.pop()

    def _check_explicit(self, node, source):

        # トークン explicit を探索します。
        for token in node.get_tokens():
            if token.kind == cindex.TokenKind.KEYWORD and token.spelling == 'explicit':
                return

        # NN_IMPLICIT はトークンとしては発見できないため、付近の行を探索します。
        for i in xrange(2):
            if node.location.line > i:
                line = source.simple_lines[node.location.line - i]
                if 'NN_IMPLICIT' in line:
                    return

        # explicit も NN_IMPLICIT も見つからなかった場合は規約違反です。
        error('class/explicit', ERROR_LEVEL_REQUIRED, _(
              u'一引数コンストラクタ {name} には explicit が必要です。'\
              u'暗黙的な変換が目的の場合は代わりに NN_IMPLICIT を指定してください。').format(
              name = node.spelling), source, node.location.line)

class PrivateFieldChecker(Checker):
    """
    クラスのメンバ変数が全て private になっていることを検証します。
    """

    def __init__(self):
        self.stack = []

    def hook_setup(self, source):
        del self.stack[:]

    def hook_node_enter(self, node, source, skip):
        if is_class_like(node):
            self.stack.append(node)
        elif not skip and node.kind == cindex.CursorKind.FIELD_DECL:
            self._check_impl(node, source)

    def hook_node_leave(self, node, source, skip):
        if is_class_like(node):
            self.stack.pop()

    def _check_impl(self, node, source):

        # クラス以外は規約のチェック対象になりません。
        if not is_class(self.stack[-1]):
            return

        # 定数も規約のチェック対象から外します。
        elif node.type.is_const_qualified():
            return

        # protected, public メンバはエラーです。
        access = node.access_specifier.name.lower()
        if access in ['public', 'protected']:
            error('class/access_specifier', ERROR_LEVEL_REQUIRED, _(
                  u'class のメンバ変数 {name} のアクセス制御に {access} が指定されました。'\
                  u'定数ではないメンバ変数のアクセス制御は全て private にしてください。').format(
                  name = node.spelling, access = access),
                  source, node.location.line)

class MemberInitializerChecker(Checker):
    """
    メンバ変数の初期化リストについて規約チェックを行います。
    """

    def hook_warning(self, warning, source):
        if warning.option == '-Wreorder':
            error('class/initializer', ERROR_LEVEL_RECOMMENDED,
                _(u'初期化リストの初期化順序はメンバ変数の宣言順序と一致させてください。'),
                  source, warning.location.line)

    def hook_node_enter(self, node, source, skip):
        if skip:
            return
        if node.kind == cindex.CursorKind.CONSTRUCTOR:
            self._check_constructor(node, source)

    def _check_constructor(self, node, source):
        stack = []
        isRhs = False
        for child in node.get_children():
            if isRhs:
                self._check_expr(stack, child, source)
                isRhs = False
            elif child.kind == cindex.CursorKind.MEMBER_REF:
                isRhs = True
                stack.append(child.spelling)

    def _check_expr(self, stack ,node, source):
        name = self._include_self_member(stack, node)
        if name:
            error('class/initializer', ERROR_LEVEL_RECOMMENDED, _(
                  u'メンバ変数 {referrer} の初期化は {refered} に依存しています。'\
                  u'初期化リストの初期化順に依存性をもたせないでください。').format(
                  referrer = stack[-1], refered = name),
                  source, node.location.line)

    def _include_self_member(self, stack, node):
        children = get_children(node)
        if node.kind == cindex.CursorKind.MEMBER_REF_EXPR:
            if len(children) == 0 and node.spelling in stack:
                return node.spelling
        for child in children:
            name = self._include_self_member(stack, child)
            if name:
                return name

class ConstIntegerMacroChecker(Checker):
    """
    整数定数マクロの使用を禁止します。
    """

    _DEFINE_PATTERN = re.compile(r'^\s*#\s*define\s+([a-zA-Z0-9_]+)\s+(.+)\s*$')

    _CONFIG_PATTERN = re.compile(r'^NN(?:_[A-Z0-9]+)*_CONFIG(?:_[A-Z0-9]+)+$')

    _VERSION_PATTERN = re.compile(r'^NN_SDK_VERSION(?:_[A-Z0-9]+)+$')

    _UNARY_PATTERN = re.compile(r'^[+\-~](.+)$')

    _BINARY_PATTERN = re.compile(r'^(.+)(?:\+|\*|/|%|-|<<|>>|&|\||\^)(.+)$')

    _PAREN_PATTERN = re.compile(r'^\((.+)\)$')

    _LITERAL_PATTERN = re.compile(
        r'^(?:[0-9]+[lLuU]*|0x[0-9a-fA-F]+[lLuU]*|0[0-7]+[lLuU]*|sizeof\s*\([^\)]+\))$')

    def __init__(self):
        self.is_macro = False
        self.line     = 0
        self.name     = None
        self.value    = None

    def hook_begin(self, source):
        self.is_macro = False
        self.line     = 0
        self.name     = None
        self.value    = None

    def hook_line(self, linenum, raw_line, line, simple_line, source):
        if self.is_macro:
            self.value += simple_line
        else:
            match = self._DEFINE_PATTERN.match(simple_line)
            if match:
                self.is_macro = True
                self.line     = linenum
                self.name     = match.group(1)
                self.value    = match.group(2)

        if self.is_macro:
            if self.value.endswith("\\"):
                self.value = self.value[:-1]
            else:
                self._check_macro(self.line, self.name, self.value, source)
                self.is_macro = False

    def _check_macro(self, line, name, value, source):
        if (self._is_const_integer(value) and
            not self._CONFIG_PATTERN.match(name) and
            not self._VERSION_PATTERN.match(name)):
            error('preprocessor/const', ERROR_LEVEL_REQUIRED, _(
                  u'{name} は整数定数マクロです。実装分岐以外の用途では'\
                  u'列挙型や const の利用を検討してください。').format(name = name),
                  source, line)

    def _is_const_integer(self, value):

        # 前後の空白は邪魔なので取り除きます。
        value = value.strip()

        # 単項演算子を取り除きます。
        match = self._UNARY_PATTERN.match(value)
        if match:
            return self._is_const_integer(match.group(1))

        # 括弧を取り除きます。
        match = self._PAREN_PATTERN.match(value)
        if match:
            return self._is_const_integer(match.group(1))

        # 二項演算子を取り除きます。
        match = self._BINARY_PATTERN.match(value)
        if match:
            return (self._is_const_integer(match.group(1)) and
                    self._is_const_integer(match.group(2)))

        # 整数リテラルをマッチングします。
        return self._LITERAL_PATTERN.match(value)

class CompilerExtensionChecker(Checker):
    """
    コンパイラ拡張の使用を禁止します。
    """

    def hook_node_enter(self, node, source, skip):
        if skip:
            return
        if node.kind in [cindex.CursorKind.VAR_DECL, cindex.CursorKind.FIELD_DECL]:
            self._check_storage(node, source)

    def _check_storage(self, node, source):
        sean = None
        for token in node.get_tokens():
            name = token.spelling
            if name in ['const', 'volatile', 'signed', 'unsigned', 'auto',
                        'void', 'bool', 'char', 'short', 'int', 'long', 'float', 'double',
                        'int8_t', 'int16_t', 'int32_t', 'int64_t',
                        'uint8_t', 'uint16_t', 'uint32_t', 'uint64_t',
                        'size_t', 'ptrdiff_t', 'uintptr_t', 'char16_t', 'char32_t',
                        'Bit8', 'Bit16', 'Bit32', 'Bit64',
                        'BitPack8', 'BitPack16', 'BitPack32', 'TimeSpan']:
                sean = name
            elif name in ['extern', 'mutable', 'register', 'static', 'thread_local']:
                if sean:
                    error('extension/storage_class', ERROR_LEVEL_REQUIRED,
                        _(u'ストレージクラス {storage} を {type} の後ろに置くことはできません。').
                          format(storage = name, type = sean),
                          source, node.location.line)
                else:
                    return
            else:
                sean = None

class FunctionLengthChecker(Checker):
    """
    関数の物理行数が上限を超えていないことを検証します。
    """

    def __init__(self, required, recommended):
        """
        Args:
            required:       規約で認められる関数の最大長（物理行数）を指定します。
            recommended:    規約で認められる関数の最大長（物理行数）を指定します。
        """
        self.required    = required
        self.recommended = recommended

    def hook_node_enter(self, node, source, skip):
        if skip:
            return
        if is_function_like(node):
            count = self._count_lines(node, source)
            if count > self.required:
                error('impl/function_size', ERROR_LEVEL_RECOMMENDED,
                    _(u'関数 {name} は物理行数が {length} 行を超えています。').format(
                      name = node.spelling, length = self.required),
                      source, node.location.line)
            elif count > self.recommended:
                error('impl/function_size', ERROR_LEVEL_INFORMATION,
                    _(u'関数 {name} は物理行数が {length} 行を超えています。').format(
                      name = node.spelling, length = self.recommended),
                      source, node.location.line)

    def _count_lines(self, func, source):

        # 関数本体を取得します。
        children = get_children(func, kind = cindex.CursorKind.COMPOUND_STMT)
        if not children:
            return 0
        compound = children[0]
        start    = compound.extent.start
        last     = compound.extent.end

        # 物理行数をカウントします。
        count = 0
        for i in xrange(start.line + 1, last.line):
            line = source.simple_lines[i].strip()
            if len(line) > 0:
                count += 1
        return count

class StdlibChecker(Checker):
    """
    特定の標準ライブラリの使用を禁止します。
    """

    _LOCALE_HEADER = [
        'locale.h', 'locale', 'clocale'
    ]

    _CONTAINER_HEADER = [
        'array', 'deque', 'forward_list', 'list', 'map', 'queue', 'set',
        'stack', 'string', 'unordered_map', 'unordered_set', 'valarray', 'vector'
    ]

    _FORBIDDEN_FUNCTIONS = [
        # C 標準
        'asctime', 'ctime', 'getenv', 'localeconv', 'rand', 'setlocale', 'strerror', 'strtok',

        # C++ 標準
        'inplace_merge', 'stable_sort'
    ]

    def hook_include(self, include_file, source):
        if include_file.depth == 1:
            dir, header = os.path.split(include_file.include.name)
            if header in self._CONTAINER_HEADER:
                error('extra/stdlib', ERROR_LEVEL_REQUIRED,
                    _(u'コンテナ {name} の使用は禁止されています。').format(
                      name = header), source, include_file.location.line)
            elif header in self._LOCALE_HEADER:
                error('extra/stdlib', ERROR_LEVEL_REQUIRED,
                    _(u'{name} に含まれるロケール機能の使用は禁止されています。').format(
                      name = header), source, include_file.location.line)

    def hook_node_enter(self, node, source, skip):
        if skip:
            return
        if (node.kind == cindex.CursorKind.CALL_EXPR and
            node.spelling in self._FORBIDDEN_FUNCTIONS):
             error('extra/stdlib', ERROR_LEVEL_REQUIRED,
                 _(u'標準関数 {name} の使用は禁止されています。').format(
                   name = node.spelling), source, node.location.line)

class Cpp11Checker(Checker):
    """
    許可されていない C++11 の機能の使用を禁止します。現状、以下の機能のみ対応しています。
    * alignas
    * alignof
    * constexpr
    * initializer_list
    * noexcept
    * noreturn
    * override
    * static_assert
    """

    def __init__(self):
        self.macro_line    = 0
        self.previous_token = None

    def hook_begin(self, source):
        self.macro_line    = 0
        self.previous_token = None

    def hook_token(self, token, source):
        if (token.kind == cindex.TokenKind.IDENTIFIER and
            token.spelling in ['define', 'include']):
            line = token.location.line
            while line < len(source.simple_lines):
                if not re.search(r'\\\s*$', source.simple_lines[line]):
                    self.macro_line = line
                    break
                line += 1
        elif self.macro_line < token.location.line and self.previous_token:
            if token.kind == cindex.TokenKind.KEYWORD:
                self._check_keyword(self.previous_token, token, source)
            elif token.kind == cindex.TokenKind.IDENTIFIER:
                self._check_identifier(self.previous_token, token, source)
        self.previous_token = token

    def _check_keyword(self, prev, token, source):
        name = token.spelling
        line = token.location.line
        if name == 'alignas':
            error('c++11/alignas', ERROR_LEVEL_REQUIRED, _(
                  u'直接 alignas を使用することは禁止されています。'
                  u'代わりに NN_ALIGNAS を使用してください。'), source, line)
        elif name == 'alignof':
            error('c++11/alignof', ERROR_LEVEL_REQUIRED, _(
                  u'直接 alignof を使用することは禁止されています。'
                  u'代わりに NN_ALIGNOF を使用してください。'), source, line)
        elif name == 'constexpr':
            error('c++11/constexpr', ERROR_LEVEL_REQUIRED,
                _(u'constexpr の使用は禁止されています。'), source, line)
        elif name == 'noexcept':
            error('c++11/noexcept', ERROR_LEVEL_REQUIRED, _(
                  u'直接 noexcept を使用することは禁止されています。'
                  u'代わりに NN_NOEXCEPT を使用してください。'), source, line)
        elif name == 'static_assert':
            error('c++11/static_assert', ERROR_LEVEL_REQUIRED, _(
                  u'直接 static_assert を使用することは禁止されています。'\
                  u'代わりに NN_STATIC_ASSERT を使用してください。'), source, line)
        elif name == 'override':
            error('c++11/override', ERROR_LEVEL_REQUIRED, _(
                  u'直接 override を使用することは禁止されています。'
                  u'代わりに NN_OVERRIDE を使用してください。'), source, line)

    def _check_identifier(self, prev, token, source):
        name = token.spelling
        line = token.location.line
        if name == 'noreturn' and prev.spelling == '[':
            error('c++11/noreturn', ERROR_LEVEL_REQUIRED, _(
                  u'直接 [[noreturn]] を使用することは禁止されています。'
                  u'代わりに NN_NORETURN を使用してください。'), source, line)
        elif name == 'initializer_list':
            error('c++11/initializer_list', ERROR_LEVEL_REQUIRED,
                _(u'std::initializer_list の使用は禁止されています。'), source, line)

class ExplicitConversionChecker(Checker):
    """
    変換関数に explicit, NN_IMPLICIT のいずれかが付加されていることを検証します。
    """

    def __init__(self):
        self.stack = []

    def hook_setup(self, source):
        del self.stack[:]

    def hook_node_enter(self, node, source, skip):
        # 規約チェックの対象となる変換関数を探します。
        if is_class_like(node):
            self.stack.append(node)
        elif not skip:
            if node.kind == cindex.CursorKind.CONVERSION_FUNCTION and len(self.stack) > 0:
                self._check_explicit(node, source)

    def hook_node_leave(self, node, source, skip):
        if is_class_like(node):
            self.stack.pop()

    def _check_explicit(self, node, source):

        # マクロはトークンとして発見できないため、付近の行を探索します。
        for i in xrange(2):
            if node.location.line > i:
                line = source.simple_lines[node.location.line - i]
                if 'NN_IMPLICIT' in line or 'NN_EXPLICIT_OPERATOR' in line:
                    return

        # NN_EXPLICIT_OPERATOR も NN_IMPLICIT も見つからなかった場合は規約違反です。
        error('c++11/explicit', ERROR_LEVEL_REQUIRED, _(
              u'変換関数 {name} は NN_EXPLICIT_OPERATOR で定義してください。'\
              u'暗黙的な変換が目的の場合は代わりに NN_IMPLICIT を指定してください。').format(
              name = node.spelling), source, node.location.line)

class UsingDirectiveChecker(Checker):
    """
    using 指令の使用を検出して警告を出します。
    """

    def hook_node_enter(self, node, source, skip):
        if skip:
            return
        if node.kind in [cindex.CursorKind.USING_DIRECTIVE,
                         cindex.CursorKind.USING_DECLARATION]:
            error('others/using', ERROR_LEVEL_REQUIRED,
                _(u'using 指令の使用は禁止されています。'),
                  source, node.location.line)

class FileLocalChecker(Checker):
    """
    ファイルローカルな定義に static を使用している場合に警告をだします。
    """

    def __init__(self):
        self.stack = []

    def hook_setup(self, source):
        del self.stack[:]

    def hook_node_enter(self, node, source, skip):
        if node.kind in [cindex.CursorKind.VAR_DECL,
                         cindex.CursorKind.FUNCTION_DECL]:
            if not skip and len(self.stack) == 0:
                self._check_impl(node, source)

        if is_class_like(node) or is_function_like(node):
            self.stack.append(node)

    def hook_node_leave(self, node, source, skip):
        if is_class_like(node) or is_function_like(node):
            self.stack.pop()

    def _check_impl(self, node, source):
        if is_static(node):
            error('others/filescope', ERROR_LEVEL_REQUIRED, _(
                  u'ファイルローカルな名前 {name} の定義には static ではなく、'\
                  u'匿名名前空間を使用してください。').format(name = node.spelling),
                  source, node.location.line)

class CastChecker(Checker):
    """
    キャスト関連の規約チェックを行います。
    """

    _CAST_PATTERN = re.compile(r'\([\sa-zA-Z0-9_<>&*]+\)\s*[a-zA-Z0-9(\'"&*]')

    def hook_node_enter(self, node, source, skip):
        if skip:
            return
        if node.kind == cindex.CursorKind.CXX_DYNAMIC_CAST_EXPR:
            error('others/cast', ERROR_LEVEL_REQUIRED,
                _(u'実行時型情報 (RTTI) の使用は禁止されています。'),
                  source, node.location.line)
        elif node.kind == cindex.CursorKind.CSTYLE_CAST_EXPR:

            # void へのキャストは特例が多いため、ここでは検証しません。
            typename = node.type.get_canonical().spelling
            if typename in ['void']:
                return

            # マクロ内のキャストを無視するため、キャストらしい表記を探します。
            line = source.simple_lines[node.location.line]
            if not self._CAST_PATTERN.search(line):
                return

            error('others/cast', ERROR_LEVEL_REQUIRED,
                _(u'C スタイルキャストの使用は禁止されています。'),
                  source, node.location.line)

class ExceptionChecker(Checker):
    """
    例外の使用を禁止する規約チェックです。
    """

    def hook_node_enter(self, node, source, skip):
        if skip:
            return
        if node.kind in [cindex.CursorKind.CXX_TRY_STMT,
                         cindex.CursorKind.CXX_THROW_EXPR]:
            error('others/exception', ERROR_LEVEL_REQUIRED,
                _(u'例外の使用は禁止されています。'),
                  source, node.location.line)

class CHeaderChecker(Checker):
    """
    C ヘッダの使用を禁止して、 C++ ヘッダの使用を推奨します。
    """

    def __init__(self, ignores = []):
        self.ignores = ignores

    def hook_include(self, include_file, source):
        dir, header = os.path.split(include_file.include.name)
        if header in C_HEADERS and header not in self.ignores:
            cpp_header = 'c' + header[:-2]
            error('others/c_header', ERROR_LEVEL_REQUIRED, _(
                  u'C 言語の標準ヘッダ {cheader} の使用は禁止されています。'\
                  u'代わりに C++ 形式の {cppheader} を使用してください。').format(
                  cheader = header, cppheader = cpp_header),
                  source, include_file.location.line)

class VoidParamChecker(Checker):
    """
    仮引数として void を指定することを禁止します。
    """

    _PATTERN = re.compile('\(\s*void\s*\)')

    def hook_node_enter(self, node, source, skip):
        if skip:
            return
        if is_function_like(node):
            params = get_children(node, kind = cindex.CursorKind.PARM_DECL)
            if len(params) == 0:
                self._check_impl(node, source)

    def _check_impl(self, node, source):
        line = source.simple_lines[node.location.line]
        if self._PATTERN.search(line):
            error('others/void_param', ERROR_LEVEL_REQUIRED,
                _(u'関数 {name} の仮引数に void を指定しないでください。').format(
                  name = node.spelling), source, node.location.line)

def is_ignored_source(path):
    """規約チェックの対象となるソースファイルであることを検証します。

    Args:
        path: ファイルの絶対パスです。
    Returns:
        規約チェックの対象になるソースファイルであれば True です。
    """

    dir, file = os.path.split(path)
    if root and not dir.startswith(root):
        return True

    for exclude in exclude_files:
        if path.startswith(exclude):
            return True

    return file in C_HEADERS or file in CPP_HEADERS

def filter_clang_errors(input_diagnostics):
    """libclang の解析結果からエラーメッセージだけを取得します。

    Args:
        input_diagnostics:  libclang の解析結果です。
    Returns:
        エラーメッセージのリストです。
    """
    errors = []
    for diag in input_diagnostics:
        threshold = cindex.Diagnostic.Error
        if diag.severity >= threshold and diag.spelling not in ['#pragma once in main file']:
            errors.append(diag)
    return errors

def filter_clang_warnings(input_diagnostics):
    """libclang の解析結果から警告メッセージだけを取得します。

    Args:
        input_diagnostics:  libclang の解析結果です。
    Returns:
        警告メッセージのリストです。
    """
    warnings = []
    for diag in input_diagnostics:
        if diag.severity == cindex.Diagnostic.Warning:
            warnings.append(diag)
    return warnings

def print_tree(ast, depth = 0):
    """デバッグ用に AST をログ出力します。

    Args:
        ast:                規約チェックの対象となる抽象構文木 (AST) です。
        depth:              根からの深さです。
    """
    for i in xrange(depth):
        debug('  ', linefeed = False)
    debug("- {kind} ({name})".format(kind = ast.kind.name, name = ast.spelling))
    for child in ast.get_children():
        print_tree(child, depth + 1)

def check_tree(ast, depth = 0):
    """AST の規約チェックを行います。

    Args:
        ast:                規約チェックの対象となる抽象構文木 (AST) です。
        depth:              根からの深さです。
    """
    file   = ast.location.file
    path   = normalize_path(file.name) if file else None
    source = source_caches[path] if source_caches.has_key(path) else None

    # ソースファイルが規約チェックの対象に含まれない場合はそれ以上辿りません。
    # 本来は辿るべきですが、辿らないことで問題になるケースが非常に少なく、
    # なおかつ規約チェックの実行速度に大きなインパクトがあります。
    if file is not None and source is None:
        return

    for checker in checkers:
        checker.hook_node_enter(ast, source, source is None)
    for child in ast.get_children():
        check_tree(child, depth + 1)
    for checker in checkers:
        checker.hook_node_leave(ast, source, source is None)

def check_source(source, warnings, tu):
    """1 つのソースファイルを入力として規約チェックを行います。

    Args:
        source:             規約チェックの対象となるソースファイルです。
        warnings:           libclang の警告です。
        tu:                 TranslationUnit です。
    """
    # 各ファイルの解析開始時に 1 度だけ実行される規約チェックです。
    for checker in checkers:
        checker.hook_begin(source)

    # libclang の警告に対して実行される規約チェックです。
    for warn in warnings:
        if warn.location.file.name == source.path:
            for checker in checkers:
                checker.hook_warning(warn, source)

    # ヘッダファイルの規約チェックです。
    for include_file in tu.get_includes():
        src_path = normalize_path(include_file.source.name)
        if src_path != source.path:
            continue
        for checker in checkers:
            checker.hook_include(include_file, source)

    # 各トークンに対して実行される規約チェックです。
    for token in tu.get_tokens(extent=source.get_full_extent()):
        for checker in checkers:
            checker.hook_token(token, source)

    # 各行に対して実行される規約チェックです。
    for i in xrange(1, len(source.lines)):
        for checker in checkers:
            raw_line    = source.raw_lines[i]
            line        = source.lines[i]
            simple_line = source.simple_lines[i]
            checker.hook_line(i, raw_line, line, simple_line, source)

    # 各ファイルの解析終了時に 1 度だけ実行される規約チェックです。
    for checker in checkers:
        checker.hook_end(source)

def check_translation_unit(source):
    """1 つの翻訳単位を入力として規約チェックを行います。

    Args:
        source:             規約チェックの対象となるソースコードです。
    """

    # libclang で AST (抽象構文木) を作成します。
    path = source.path
    translation_unit = index.parse(path, build_options, unsaved_files = [(path, source.body)])
    clang_errors     = filter_clang_errors(translation_unit.diagnostics)
    clang_warnings   = filter_clang_warnings(translation_unit.diagnostics)
    for diag in clang_errors:
        message = diag.spelling.decode('utf_8')
        if diag.location.file:
            filename  = diag.location.file.name
            linenum   = diag.location.line
            text      = _(u"SYNTAX: {file} L.{line} {clang_message}").format(
                        file=filename, line=linenum, clang_message=message)
            verbose(text)
        else:
            text      = _(u"SYNTAX: {file} {clang_message}").format(
                        file=path, clang_message=message)
            verbose(text)
    if len(clang_errors) > 0 and not syntax_error_ignored:
        warning(_(u'{file} の解析に失敗したため規約チェックをスキップします。').format(file=path))
        return

    # ヘッダファイルを再帰的に辿って規約チェックの対象になるソースファイルを列挙します。
    files = [ path ]
    for include_file in translation_unit.get_includes():
        src    = include_file.source.name
        header = normalize_path(include_file.include.name)
        if not is_ignored_source(header):
            files.append(header)
    files = set(files)

    # 規約チェックしやすいようにソースファイルを加工します。
    for file in files:
        src = read_source(file)
        src.process(translation_unit)

    # 翻訳単位の解析開始時に 1 度だけ実行される規約チェックです。
    for file in files:
        for checker in checkers:
            checker.hook_setup(read_source(file))

    # AST (抽象構文木) に対して実行される規約チェックです。
    print_tree(translation_unit.cursor)
    check_tree(translation_unit.cursor)

    # ファイル単位の規約チェックです。
    for file in files:
        check_source(read_source(file), clang_warnings, translation_unit)

    # 翻訳単位の解析終了時に 1 度だけ実行される規約チェックです。
    for file in files:
        for checker in checkers:
            checker.hook_cleanup(read_source(file))

def check_file(file):
    """指定されたファイルの規約チェックを実行します。

    Args:
        file:               対象のファイルです。
    Returns:
        規約チェックの結果を表す CheckedResult クラスのインスタンスです。
        ただし、何らかの理由で規約チェックが実施されなかった場合は None です。
    """
    # パスを正規化します。
    path = normalize_path(file)

    # 入力されたファイルが規約チェックの対象となるか確認します。
    if is_ignored_source(path):
        verbose(_(u'{file} は規約チェックの対象から除外します。').format (file = path))
        return None

    # 規約チェックを実施するファイルの名前を出力します。
    verbose(_(u'{file} の規約チェックを開始します。').format(file = path))

    # 入力ファイルの内容を全て読み込みます。
    source = read_source(path, True)
    if source is None:
        warning(_(u'{file} のオープンに失敗しました。').format(file=path))
        return None

    # 解析対象となるファイルを記憶しておきます。
    result = CheckedResult(path)
    checked_results.append(result)

    # ソースファイルの内容に対する規約チェックを実施します。
    check_translation_unit(source)

    return result

def check_files(files):
    """指定された全てのファイルについて規約チェックを実行します。

    Args:
        files:              対象のファイルのリストです。
    """

    global error_count
    global warning_count

    # 規約チェックの開始前に状態を初期化します。
    del checked_results[:]
    error_count = 0
    warning_count = 0

    # 入力ファイルを 1 つずつ取り出して規約チェックを行います。
    for file in files:
        result = check_file(file)
        if result:
            print_errors(result)

def create_exclude_file_list(path):
    """規約チェックから除外するファイルのリストを生成します。

    Args:
        path:               規約チェックから除外するファイルを列挙したリストファイルです。
    Returns:
        規約チェックから除外するファイルのリストです。
    """

    # リストファイルを読み込みます。
    try:
        raw_body = codecs.open(path, 'r', 'utf_8', 'strict').read()
    except:
        warning(_(u'{file} のオープンに失敗しました。').format(file=path))
        return None

    # 行毎に除外するファイルが列挙されているため行で分割します。
    body  = remove_bom(raw_body)
    lines = body.splitlines()

    # 各行を解析して除外すべきファイルを列挙します。
    files = []
    for line in lines:
        files.append(normalize_path(SIGLO_ROOT + '/' + line))

    return files

def parse_arguments(args):
    """コマンドライン引数を解析します。

    Args:
        args:               コマンドライン引数のリストです。コマンド名は含みません。
    Returns:
        規約チェックの対象となるファイルのリストです。
    """

    global encoding
    global error_level
    global language
    global log_level
    global recursive
    global root
    global syntax_error_ignored
    global exclude_files

    # コマンドライン引数を解析するパーサーを生成します。
    parser = argparse.ArgumentParser(description=
                        'This tool detects coding rule violations for C++ developers.  '
                        'Please note that this script does not necessarily ensures that '
                        'C++ code completely conforms to Nintendo coding rules.')
    parser.add_argument('file', nargs='+',
                        help = 'C++ source or header files')
    parser.add_argument('-I', metavar='DIR', action='append', dest='includes',
                        help = 'add DIR to include search path')
    parser.add_argument('-D', metavar='NAME', action='append', dest='macros',
                        help = 'predefine NAME as a macro')
    parser.add_argument('-l', '--level', metavar='NUM', type=int, dest='level',
                        help = 'suppress errors which levels are less than NUM')
    parser.add_argument('-v', '--verbose', action='store_true', dest='verbose',
                        help = 'show verbose logs')
    parser.add_argument('-d', '--debug', action='store_true', dest='debug',
                        help = 'show debug logs')
    parser.add_argument('-f', '--force', action='store_true', dest='error_ignored',
                        help = 'ignore syntax errors and enforce the checks')
    parser.add_argument('--ignore', metavar='CAT', action='append', dest='ignores',
                        help = 'ignore given error categories')
    parser.add_argument('--only', metavar='CAT', action='append', dest='only',
                        help = 'check only given error categories')
    parser.add_argument('--encoding', metavar='NAME', dest='encoding',
                        help = 'show logs in the given encoding')
    parser.add_argument('--compiler', metavar='NAME', dest='compiler',
                        help = 'enable compiler compatibility')
    parser.add_argument('--cpu', metavar='NAME', dest='cpu',
                        help = 'enable cpu compatibility')
    parser.add_argument('--check-header', action='store_true', dest='check_header',
                        help = 'check header files recursively')
    parser.add_argument('--language', metavar='NAME', dest='language',
                        choices=['en', 'ja'],
                        help = 'show logs in the given language')
    parser.add_argument('--check-header-dir', metavar='PATH', dest='root',
                        help = 'specify the root directory for --check-header')
    parser.add_argument('--exclude-list', metavar='PATH', dest='exclude',
                        help = 'specify an exclude list file path')

    # コマンドライン引数を解析します。
    result = parser.parse_args(args)

    # libclang のエラーを無視して規約チェックを継続するオプションの設定です。
    if result.error_ignored:
        syntax_error_ignored = True

    # ログ出力の詳細度を設定します。
    if result.debug:
        log_level = LOG_LEVEL_DEBUG
    elif result.verbose:
        log_level = LOG_LEVEL_VERBOSE

    # ログ出力のエンコーディングを設定します。現時点で値の正当性は確認していません。
    if result.encoding:
        encoding = result.encoding

    # ログメッセージの言語設定です。
    code, enc = locale.getdefaultlocale()
    if result.language:
        language = result.language
    elif code == 'ja_JP':
        language = 'ja'

    # 指定されたレベル以上の規約だけがチェックされるように設定します。
    if result.level:
        error_level = result.level

    # インクルードされたヘッダファイルの規約チェックを再帰的に実行する設定です。
    if result.check_header:
        recursive = True
    if result.root:
        root = normalize_path(result.root)

    # clang に与えるオプションを設定します。
    if result.macros:
        for macro in result.macros:
            build_options.append('-D')
            build_options.append(macro)
    if result.includes:
        for dir in result.includes:
            build_options.append('-I')
            build_options.append(dir)
    if result.compiler:
        if result.compiler == 'vc':
            build_options.append('-fms-compatibility')
            build_options.append('-fms-extensions')
            build_options.append('-D MSC_VER=1700')
    if result.cpu:
        if result.cpu == 'x86':
            build_options.append('-D _M_IX86')

    # 規約チェックから除外するファイルのリストを生成します。
    if result.exclude:
        exclude_files = create_exclude_file_list(result.exclude)

    return result.file

def define_rules():
    """コーディング規約を定義します。

    Returns:
        規約のチェッカのリストです。
    """

    return [

        #==================================================
        # ディレクトリとファイル (file)
        #==================================================

        # ファイルの拡張子, ファイルの名前, パスの構成
        PathChecker(),

        #==================================================
        # 名前 (naming)
        #==================================================

        # 既定外のアンダースコア使用の禁止, 単語の連結, 名前空間と結合, 自動変数以外の変数の名前
        # 自動変数の名前, 名前のスタイル, 列挙子の名前, マクロの名前
        NamingChecker(),

        # 使用禁止の名前
        BlacklistNameChecker(),

        # 真偽値を設定・取得する関数と真偽値変数の名前
        BooleanNamingChecker(),

        # 〇〇の数に対する命名規則
        CountNamingChecker(),

        #==================================================
        # コーディングスタイル (style)
        #==================================================

        # ファイル形式
        Utf8BomChecker(BOM_REQUIRED),
        LinefeedChecker(LINEFEED_CRLF),

        # インデント
        IndentChecker(INDENT_SPACE, 4),

        # スペーシング
        BinaryOperatorSpacingChecker(['+', '-', '*', '/', '%']),

        # 波括弧
        NewlineBraceChecker(),

        # 1 行の長さ
        LineWidthChecker(100),

        # ファイルヘッダ
        CopyrightChecker('Nintendo', 1, 2),

        # インクルードガード
        IncludeGuardChecker(),

        # ファイルの最後は改行
        EofLineChecker(),

        # 変数の宣言
        MultipleVariableDeclarationChecker(),

        # switch 文における default の必須
        SwitchDefaultChecker(),

        # switch 文における fall through の原則禁止
        SwitchFallthroughChecker(),

        # アクセスレベルの記述順序, private の記述
        AccessSpecifierChecker(),

        # class と typename
        TemplateKeywordChecker(),

        # 代替表記の禁止
        AlternativeTokenChecker(),

        #==================================================
        # 型 (type)
        #==================================================

        # 標準型の使用
        ForbiddenTypeChecker(),

        # ビットフィールドの使用禁止
        BitFieldChecker(),

        #==================================================
        # 変数 (variable)
        #==================================================

        # 仮引数の変更禁止
        ParamReassignmentChecker(),

        #==================================================
        # 関数 (function)
        #==================================================

        # 仮引数の型
        ReferenceChecker(),

        # デフォルト引数の使用禁止
        DefaultParamChecker(),

        # 無例外指定の明示
        NoexceptChecker(),

        #==================================================
        # クラス (class)
        #==================================================

        # struct と class の使い分け
        PodChecker(),

        # 一引数コンストラクタの明示指定
        ExplicitConstructorChecker(),

        # メンバ変数のアクセス制御
        PrivateFieldChecker(),

        # メンバ初期化リストの依存性
        MemberInitializerChecker(),

        #==================================================
        # プリプロセッサ (macro)
        #==================================================

        # 整数定数マクロの使用禁止
        ConstIntegerMacroChecker(),

        #==================================================
        # 実装 (impl)
        #==================================================

        # コンパイラ拡張機能の使用禁止
        CompilerExtensionChecker(),

        # 関数行数の制限値
        FunctionLengthChecker(100, 50),

        #==================================================
        # 所属別 (special)
        #==================================================

        # 標準ライブラリの使用禁止
        StdlibChecker(),

        #==================================================
        # C++11 (c++11)
        #==================================================

        # 仕様を許可されていない C++11 以降の機能の禁止
        # C++03 のインタフェース
        Cpp11Checker(),

        # 明示的な変換関数
        ExplicitConversionChecker(),

        #==================================================
        # その他 (others)
        #==================================================

        # using 指令使用禁止
        UsingDirectiveChecker(),

        # ファイルローカルな名前
        FileLocalChecker(),

        # 実行時型情報 (RTTI) の使用禁止, C スタイルキャストの使用禁止
        CastChecker(),

        # 例外の使用禁止
        ExceptionChecker(),

        # 標準ヘッダのインクルード
        CHeaderChecker(ignores = ['stddef.h', 'stdint.h']),

        # 仮引数並びとしての void 使用禁止
        VoidParamChecker()

    ]

def main(args):
    """プログラムのエントリポイントです。

    Args:
        args:               コマンドライン引数のリストです。コマンド名は含みません。
    """

    # コマンドライン引数を解析します。
    files = parse_arguments(args)

    # 翻訳オブジェクトを生成します。
    if language:
        global _
        _ = gettext.translation(
            domain    = 'CppCodingRuleChecker',
            localedir = os.path.join(os.path.dirname(__file__), 'Locale'),
            languages = [language]
        ).ugettext

    # コーディング規約を定義します。
    global checkers
    checkers = define_rules()

    # 規約チェックを実施します。
    check_files(files)

    # 規約チェックの結果を表示します。
    verbose(_(u'結果: ファイル {files} 個  規約違反 {errors} 個  警告 {warnings} 個').format(
            files=len(checked_results), errors=error_count, warnings=warning_count))

if __name__ == '__main__':
    main(sys.argv[1:])
