﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using CodingCheckerUtil;
using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Text;
using System.Threading.Tasks;

namespace PushChecker
{
    internal static class Utils
    {
        private static readonly char[] Separators = new char[] { '\t', ' ' };

        /// <summary>
        /// pre-push フックに与えられる標準入力をパースします。
        /// </summary>
        internal static IReadOnlyCollection<PushInfo> ParsePrePushInput(string input)
        {
            var pushInfos = new List<PushInfo>();
            using (var sr = new StringReader(input))
            {
                var lines = CodingCheckerUtil.GitUtil.ReadAllLines(sr);

                foreach (var line in lines)
                {
                    // フォーマットは次の通り:
                    // <local ref> SP <local sha1> SP <remote ref> SP <remote sha1> LF
                    var cols = line.Split(Separators, options: StringSplitOptions.RemoveEmptyEntries);
                    if (cols.Length != 4)
                    {
                        continue;
                    }
                    pushInfos.Add(new PushInfo(
                        new RefInfo(cols[0], cols[1]),
                        new RefInfo(cols[2], cols[3])));
                }
            }
            return pushInfos;
        }
    }

    internal class PushInfo
    {
        // HACK: 同値比較を Tuple に委譲する
        private Tuple<RefInfo, RefInfo> _value;

        /// <summary>
        /// push 元のローカルの ref
        /// </summary>
        public RefInfo LocalRef => _value.Item1;
        /// <summary>
        /// push 先のリモートの ref
        /// </summary>
        public RefInfo DestinationRef => _value.Item2;

        public PushInfo(RefInfo localRef, RefInfo destinationRef)
        {
            this._value = Tuple.Create(localRef, destinationRef);
        }

        public override bool Equals(object obj)
        {
            if (obj == null || GetType() != obj.GetType())
            {
                return false;
            }

            var other = obj as PushInfo;
            return EqualityComparer<Tuple<RefInfo, RefInfo>>.Default.Equals(this._value, other._value);
        }

        public override string ToString() => _value.ToString();
        public override int GetHashCode() => _value.GetHashCode();
        public static bool operator ==(PushInfo lhs, PushInfo rhs) => EqualityComparer<Tuple<RefInfo, RefInfo>>.Default.Equals(lhs._value, rhs._value);
        public static bool operator !=(PushInfo lhs, PushInfo rhs) => !(lhs == rhs);
    }
}
