﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

//コミットログチェックツール
//
// 使い方
// commit-msgファイル及びビルドして生成されるCommitLogChecker.exeを.git/hooksにコピーしてください。

using Nintendo.ToolFoundation.IO;
using System;
using System.Reflection;
using System.Collections.Generic;
using System.Diagnostics;
using System.IO;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Runtime.CompilerServices;
using System.Diagnostics.Contracts;

[assembly: InternalsVisibleTo("CommitLogCheckerTest")] // TODO: 厳密名の指定を検討する

namespace CommitLogChecker
{
    internal class Program
    {
        //----------------------------------------------
        //メイン関数
        internal static void Main(string[] args)
        {
            // コマンドライン引数を解析
            var parseResult = ParseCommandLine(args);
            if (parseResult.Params == null)
            {
                Environment.Exit(parseResult.ExitCode);
                return;
            }

            var @params = parseResult.Params;

            //--------------------------------
            // コミットログの読み込み
            string commitLog;
            try
            {
                using (StreamReader sr = new StreamReader(@params.CommitLogFile))
                {
                    commitLog = sr.ReadToEnd();
                }
            }
            catch (System.Exception exception)
            {
                Console.Error.WriteLine("Failed to read commit-log file:" + @params.CommitLogFile);
                Console.Error.WriteLine(exception);
                Environment.Exit(2);
                return;
            }

            List<string> jiraKeys = Utils.ReadJiraKeyList(@params.JiraKeyListFile);
            var commitLogChecker = new CommitLogChecker(jiraKeys);
            var commitLogConverter = new CommitLogConverter(@params.BranchName, jiraKeys);

            //--------------------------------
            // コミットログチェックの例外判定
            if (commitLogChecker.CheckExceptionCommit(commitLog))
            {
                Environment.Exit(0);
            }

            //--------------------------------
            // コミットログの置換
            string replacedCommitLog = commitLogConverter.ReplaceJiraKeySubstitute(commitLog);
            if (!string.IsNullOrEmpty(@params.OutputFile))
            {
                File.WriteAllText(@params.OutputFile, replacedCommitLog);
            }

            //--------------------------------
            // 体裁チェック
            if (commitLogChecker.CheckCommitLog(replacedCommitLog))
            {
                Environment.Exit(0);
            }
            else
            {
                Console.Error.WriteLine(@"Commit Log is");
                Console.Error.WriteLine(commitLog);
                Environment.Exit(1);
            }
        }

        private static ParseCommandLineResult ParseCommandLine(string[] args)
        {
            var s = new CommandLineParserSettings()
            {
                ApplicationDescription = "Validates a git-commit log.",
            };

            try
            {
                CommitLogCheckerParams @params;
                if (!new CommandLineParser(s).ParseArgs(args, out @params))
                {
                    return new ParseCommandLineResult(0);
                }
                return new ParseCommandLineResult(@params);
            }
            catch (ArgumentException)
            {
                return new ParseCommandLineResult(1);
            }
        }

        private class ParseCommandLineResult
        {
            public CommitLogCheckerParams Params { get; }
            public int ExitCode { get; }

            public ParseCommandLineResult(CommitLogCheckerParams @params)
            {
                this.Params = @params;
            }

            public ParseCommandLineResult(int exitCode)
            {
                this.Params = null;
                this.ExitCode = exitCode;
            }
        }
    }
}
