﻿using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.IO;
using System.Linq;
using System.Management.Automation;
using System.Text;
using System.Text.RegularExpressions;
using System.Threading.Tasks;
using System.Xml;
using System.Xml.Linq;

namespace Nintendo.ApiReference
{
    /// <summary>
    /// Yaml ファイルの内容を読み込みオブジェクトを生成するコマンドレットクラスです。
    /// </summary>
    [Cmdlet(VerbsCommon.Add, "ApiLinkSpan")]
    public class AddApiLinkSpanCommand : PSCmdlet
    {
        private static readonly Regex RegexTargetTag = new Regex( @"(<(?<tag>p|th|td|li)>)(.*?)(?=</\k<tag>>)", RegexOptions.Compiled);
        private static readonly Regex RegexExcludeTag =
            new Regex(
                @"<"
                    + @"("
                        + @"("
                            + @"(?<tag>a|code|ac:(link|plain-text-body|structured-macro))\s*"
                            + @"([-a-zA-Z0-9:]+\s*=\s*((""[^""]*"")|('[^']*'))\s*)*"
                        + @")"
                    + @"|"
                        + @"((?<tag>span)\s+class\s*=\s*""confluence-link"")"
                    + @")"
                + @">"
                + @".*?"
                + @"</\k<tag>>",
                RegexOptions.Compiled);
        private static readonly Regex RegexLink =
            new Regex(
                @"({{(?i)NoLink(?-i)}}(?<nolink>.*?){{/(?i)NoLink(?-i)}})"
                + @"|"
                + @"({{ApiLink=(?<url>.*?)}}(?<link>.*?){{/ApiLink}})"
                + @"|"
                + @"((?<![a-zA-Z0-9_])(?<nn>nn::[a-zA-Z0-9_:]*|NN_[A-Z0-9_]*))",
                RegexOptions.Compiled);

        private string _inputFile;

        private string _outputFile;

        private string _targetApiReferencePath = @"/Documents/Api/Html[^/""]*/";

        private string _replaceApiReferencePath = "/Api/Html/";

        private string _replaceDependApiReferencePath = "/Api/Html/";

        private Regex _targetUrlRegex;

        private string _replaceUrlStr;

        private string _replaceDependUrlStr;

        private ISet<string> _doxyTagFileSet = null;

        [Parameter(
            Mandatory = true,
            Position = 0)]
        public string InputFile { get; set; }

        [Parameter(
            Position = 1)]
        public string OutputFile { get; set; }

        [Parameter]
        public string TargetApiReferencePath
        {
            get { return _targetApiReferencePath; }
            set { _targetApiReferencePath = value; }
        }

        [Parameter]
        public string ReplaceApiReferencePath
        {
            get { return _replaceApiReferencePath; }
            set { _replaceApiReferencePath = value; }
        }

        [Parameter]
        public string ReplaceDependApiReferencePath
        {
            get { return _replaceDependApiReferencePath; }
            set { _replaceDependApiReferencePath = value; }
        }

        [Parameter]
        public string DoxyTag { get; set; }

        protected override void BeginProcessing()
        {
            // InputFile の引数チェック
            {
                ProviderInfo provider;
                var providerPath = SessionState.Path.GetResolvedProviderPathFromPSPath(InputFile, out provider);
                if (provider.Name != "FileSystem" || providerPath.Count != 1)
                {
                    ThrowTerminatingError(
                        new ErrorRecord(
                            new ArgumentException(
                                string.Format("Input file path is invalid. - [{0}]", InputFile)),
                            "InvalidArgument",
                            ErrorCategory.InvalidArgument,
                            InputFile));
                    return;
                }

                _inputFile = providerPath[0];
            }

            // OutputFile の引数チェック
            if (!string.IsNullOrEmpty(OutputFile))
            {
                PSDriveInfo psDrive;
                ProviderInfo provider;
                var providerPath = SessionState.Path.GetUnresolvedProviderPathFromPSPath(OutputFile, out provider, out psDrive);
                if (provider.Name != "FileSystem")
                {
                    ThrowTerminatingError(
                        new ErrorRecord(
                            new ArgumentException(
                                string.Format("Output file path is invalid. - [{0}]", OutputFile)),
                            "InvalidArgument",
                            ErrorCategory.InvalidArgument,
                            OutputFile));
                    return;
                }
                _outputFile = providerPath;
            }
            else
            {
                var ext = Path.GetExtension(_inputFile);
                _outputFile = Path.ChangeExtension(_inputFile, ".output" + ext);
            }
            if (File.Exists(_outputFile))
            {
                WriteWarning(string.Format("{0} already exists.", _outputFile));
            }

            _targetUrlRegex =
                new Regex(
                    @"\s+href=""https?://[\w\.]+?/TeamCity/[^""]+?" + _targetApiReferencePath + @"([^""]*?)(?="")",
                    RegexOptions.Compiled);

            _replaceUrlStr = @" href=""../../.." + _replaceApiReferencePath;

            if (!string.IsNullOrEmpty(DoxyTag))
            {
                // いずれか一方が指定されていないかどうか
                if (string.IsNullOrEmpty(ReplaceDependApiReferencePath))
                {
                    ThrowTerminatingError(
                        new ErrorRecord(
                            new ArgumentException("DoxyTag と ReplaceDependApiReferencePath は両方ともしていなければなりません。"),
                            "InvalidArgument",
                            ErrorCategory.InvalidArgument,
                            DoxyTag));
                    return;
                }

                _replaceDependUrlStr = @" href=""../../.." + ReplaceDependApiReferencePath;

                ProviderInfo provider;
                var providerPath = SessionState.Path.GetResolvedProviderPathFromPSPath(DoxyTag, out provider);
                if (provider.Name != "FileSystem" || providerPath.Count != 1)
                {
                    ThrowTerminatingError(
                        new ErrorRecord(
                            new ArgumentException(
                                string.Format("DoxyTag file path is invalid. - [{0}]", DoxyTag)),
                            "InvalidArgument",
                            ErrorCategory.InvalidArgument,
                            InputFile));
                    return;
                }

                var tagFileEle = XElement.Load(providerPath[0]);
                var enumFileName =
                    tagFileEle.Elements().SelectMany(
                        compound =>
                            (
                                from filename in compound.Elements("filename")
                                let filenameStr =
                                    Path.HasExtension((string)filename) ?
                                        (string)filename:
                                        ((string)filename + ".html")
                                select filenameStr)
                            .Concat(
                                (from member in compound.Elements("member")
                                from anchorfile in member.Elements("anchorfile")
                                select (string)anchorfile))
                    );
                _doxyTagFileSet = new HashSet<string>(enumFileName, StringComparer.OrdinalIgnoreCase);
            }
        }

        protected override void ProcessRecord()
        {
            var settings =
                new XmlReaderSettings
                {
                    CloseInput = true,
                    IgnoreWhitespace = true,
                    IgnoreProcessingInstructions = true,
                    IgnoreComments = true
                };

            // ページIDごとにparent属性を持つproperty要素を見つけたかどうかを保持する辞書。
            var pageDic = new Dictionary<int, bool>();

            var writerSettings = new XmlWriterSettings
            {
                CloseOutput = true
            };

            using (var xw = XmlWriter.Create(_outputFile, writerSettings))
            using (var xr = XmlReader.Create(_inputFile, settings))
            {
                while (!xr.IsStartElement())
                {
                    xr.Read();
                }

                xw.WriteStartElement(xr.Name);
                xw.WriteAttributes(xr, false);
                xr.ReadStartElement();  // ルート要素の移動
                while (xr.IsStartElement()) // <object>
                {
                    switch (xr["class"])
                    {
                    case "Page":
                        {
                            // <object class="Page"> から、最新のページのIDを取得する。
                            xw.WriteStartElement(xr.Name);
                            xw.WriteAttributes(xr, false);
                            xr.ReadStartElement();

                            if (xr.Name != "id")
                            {
                                ThrowUnknownFormat();
                            }
                            var id = CopyIdElement(xw, xr);

                            bool hasParent = false;
                            while (xr.NodeType != XmlNodeType.EndElement)
                            {
                                if (!hasParent && xr.Name == "property" && xr["name"] == "parent")
                                {
                                    hasParent = true;
                                }
                                xw.WriteNode(xr, false);
                            }

                            pageDic.Add(id, hasParent);

                            xw.WriteEndElement();
                            xr.ReadEndElement();
                        }
                        break;

                    case "BodyContent":
                        {
                            xw.WriteStartElement(xr.Name);
                            xw.WriteAttributes(xr, false);
                            xr.ReadStartElement();

                            if (xr.Name != "id")
                            {
                                ThrowUnknownFormat();
                            }
                            CopyIdElement(xw, xr);

                            if (xr.Name != "property" || xr["name"] != "body")
                            {
                                ThrowUnknownFormat();
                            }

                            // <property name="body" ...> の情報をひとまず保存。
                            var bodyAttrs = GetAttributes(xr);
                            var body = xr.ReadElementContentAsString();

                            if (xr.Name != "property" || xr["name"] != "content")
                            {
                                ThrowUnknownFormat();
                            }

                            if (xr["class"] == "Page")
                            {
                                var propAttrs = GetAttributes(xr);

                                xr.ReadStartElement("property");
                                if (xr.Name != "id")
                                {
                                    ThrowUnknownFormat();
                                }
                                var idAttrs = GetAttributes(xr);
                                var pageId = xr.ReadElementContentAsInt();

                                // ページIDがparent属性を持っているものであれば、変換を行う。
                                // そうでない場合はそのまま。
                                WriteBodyContent(
                                    xw,
                                    bodyAttrs,
                                    pageDic[pageId] ?
                                        ReplaceTeamCityLink(ProcConnectedLine(body), _targetUrlRegex, _replaceUrlStr, _replaceDependUrlStr, _doxyTagFileSet) :
                                        body);

                                xw.WriteStartElement("property");
                                WriteAttributes(xw, propAttrs);
                                    xw.WriteStartElement("id");
                                        WriteAttributes(xw, idAttrs);
                                        xw.WriteValue(pageId);
                                    xw.WriteEndElement();
                                xw.WriteEndElement();

                                xr.ReadEndElement();
                            }
                            else
                            {
                                // Page 以外はそのままコピー。
                                WriteBodyContent(xw, bodyAttrs, body);
                                xw.WriteNode(xr, false);
                            }

                            while (xr.NodeType != XmlNodeType.EndElement)
                            {
                                xw.WriteNode(xr, false);
                            }

                            xw.WriteEndElement();
                            xr.ReadEndElement();
                        }
                        break;
                    default:
                        xw.WriteNode(xr, false);
                        break;
                    }

                    xw.WriteWhitespace(Environment.NewLine);
                }
            }
        }

        /// <summary>
        /// XmlReaderからXmlWriterに対してid要素をコピーし、id要素の値を関数の値として返します。
        /// </summary>
        /// <param name="xw">XmlWriter</param>
        /// <param name="xr">XmlReader</param>
        /// <returns>id要素の値</returns>
        private static int CopyIdElement(XmlWriter xw, XmlReader xr)
        {
            xw.WriteStartElement(xr.Name);
            xw.WriteAttributes(xr, false);
            var id = xr.ReadElementContentAsInt();
            xw.WriteValue(id);
            xw.WriteEndElement();
            return id;
        }

        /// <summary>
        /// 現在の Element の属性をすべて取得します。
        /// </summary>
        /// <param name="xr">XmlReader</param>
        /// <returns>属性の名前と値のペアの配列。</returns>
        private static Tuple<string, string>[] GetAttributes(XmlReader xr)
        {
            var attrs = new Tuple<string, string>[xr.AttributeCount];
            var hasAttr = xr.MoveToFirstAttribute();
            for (var i = 0; hasAttr; ++i, hasAttr = xr.MoveToNextAttribute())
            {
                attrs[i] = Tuple.Create(xr.Name, xr.Value);
            }
            xr.MoveToElement();
            return attrs;
        }

        /// <summary>
        /// BodyContent の内容を書き込みます。
        /// </summary>
        /// <param name="xw">XmlWriter</param>
        /// <param name="attrs">属性の名前と値のペアのIEnumerable。</param>
        /// <param name="body">書き込む内容。</param>
        private static void WriteBodyContent(XmlWriter xw, IEnumerable<Tuple<string, string>> attrs, string body)
        {
            xw.WriteStartElement("property");
            WriteAttributes(xw, attrs);
            xw.WriteCData(body);
            xw.WriteEndElement();
        }

        /// <summary>
        /// 複数の属性を書き込みます。
        /// </summary>
        /// <param name="xw">XmlWriter</param>
        /// <param name="attrs">属性の名前と値のペアのIEnumerable。</param>
        private static void WriteAttributes(XmlWriter xw, IEnumerable<Tuple<string, string>> attrs)
        {
            foreach (var attr in attrs)
            {
                xw.WriteAttributeString(attr.Item1, attr.Item2);
            }
        }

        /// <summary>
        /// p要素の中の置換を行います。
        /// </summary>
        /// <param name="line">置換対象の文字列。</param>
        /// <returns>置換後の文字列。</returns>
        private static string ProcConnectedLine(string line)
        {
            // <p> <th> <td> <li> で囲まれていた部分を解析し、spanタグをつける
            return
                RegexTargetTag.Replace(
                    line,
                    match =>
                        match.Groups[1].Value
                        + AddSpanToApiLink(match.Groups[2].Value));
        }

        /// <summary>
        /// リンク候補を探して span を付ける
        /// </summary>
        /// <param name="line"></param>
        /// <returns></returns>
        private static string AddSpanToApiLink(string line)
        {
            // <a ></a> を除外
            // <code></code> を除外
            // <ac:link></ac:link> を除外
            // <span class="confluence-link"></span> を除外
            // <ac:plain-text-body></ac:plain-text-body> を除外
            // <ac:structured-macro ac:name="xxx"></ac:structured-macro> を除外

            var preIdx = 0;
            StringBuilder sb = new StringBuilder(line.Length * 2);
            var tagMatches = RegexExcludeTag.Matches(line);
            foreach (Match tagMatch in tagMatches)
            {
                sb.Append(ReplaceApiLink(line.Substring(preIdx, tagMatch.Index - preIdx)));
                sb.Append(tagMatch.ToString());
                preIdx = tagMatch.Index + tagMatch.Length;
            }

            if (preIdx < line.Length)
            {
                sb.Append(ReplaceApiLink(line.Substring(preIdx, line.Length - preIdx)));
            }

            return sb.ToString();
        }

        /// <summary>
        /// リンク候補のテキストを正規表現で置換します。
        /// </summary>
        /// <param name="text"></param>
        /// <returns></returns>
        private static string ReplaceApiLink(string text)
        {
            return
                RegexLink.Replace(
                    text,
                    match =>
                        !string.IsNullOrEmpty(match.Groups["nolink"].Value) ?
                            match.Groups["nolink"].Value:   // {{NoLink}}{{/NoLink}} を除外
                            (!string.IsNullOrEmpty(match.Groups["link"].Value) ?
                                    string.Format(
                                        "<span class=\"ApiLink_{0}\">{1}</span>",   // {{ApiLink}} 処理
                                        match.Groups["url"].Value.Replace(":", "_"),
                                        match.Groups["link"].Value):
                                    string.Format(
                                        "<span class=\"ApiLink_{0}\">{1}</span>",   // nn::*, NN_* を処理
                                        match.Groups["nn"].Value.Replace(":", "_"),
                                        match.Groups["nn"].Value)));
        }

        /// <summary>
        /// TeamCity の成果物へのURLを相対パスに置換します。
        /// </summary>
        /// <param name="text">置換対象を含む文字列。</param>
        /// <param name="targetRegex">置換対象のURLのRegex。</param>
        /// <param name="replaceStr">置換後の文字列。</param>
        /// <returns></returns>
        private static string ReplaceTeamCityLink(string text, Regex targetRegex, string replaceStr, string replaceDependStr, ISet<string> doxyTagFileSet)
        {
            return
                targetRegex.Replace(
                    text,
                    match =>
                        (   null != doxyTagFileSet && !doxyTagFileSet.Contains(RemoveUrlFragment(match.Groups[1].Value)) ?
                                replaceDependStr :
                                replaceStr)
                        + match.Groups[1].Value
                );
        }

        /// <summary>
        /// パスから# で始まる文字列を削除します。
        /// </summary>
        /// <param name="path">パス文字列。</param>
        /// <returns>#から最後まで取り除かれた文字列。#が含まれていない場合は元の文字列。</returns>
        private static string RemoveUrlFragment(string path)
        {
            var fileName = Path.GetFileName(path);
            var igeIdx = fileName.LastIndexOf('#');
            return
                igeIdx != -1 ?
                    fileName.Substring(0, igeIdx) :
                    fileName;
        }

        /// <summary>
        /// 想定していないフォーマットであることを示す例外をスローします。
        /// </summary>
        private void ThrowUnknownFormat()
        {
            ThrowTerminatingError(
                new ErrorRecord(
                    new InvalidOperationException("想定していないフォーマットです。"),
                    "UnknownFormat",
                    ErrorCategory.InvalidData,
                    InputFile));
        }
    }
}
