<#
    Copyright (C)Nintendo All rights reserved.

    These coded instructions, statements, and computer programs contain proprietary
    information of Nintendo and/or its licensed developers and are protected by
    national and international copyright laws. They may not be disclosed to third
    parties or copied or duplicated in any form, in whole or in part, without the
    prior written consent of Nintendo.

    The content herein is highly confidential and should be handled accordingly.
#>

<#
    .SYNOPSIS
        Runs USB tests with sleep/wake and dock/undock.  These are stress tests that should run for a long time.

    .DESCRIPTION
        - Runs network connection test with sleep/wake and dock/undock, to verify the USB connection
           - Runs Windows side testNet_HttpServer_Win.exe
           - Runs device side network connection test
        
#>

param
(
    [string]
    # SDK root directory
    $SigloRoot =
    ($env:NINTENDO_SDK_ROOT | ? { $_ } | % { $_.Trim() }),

    [string]
    # Device's IP address
    $TargetAddress =
    ($env:TARGET_ADDRESS_PATTERN | ? { $_ } | % { $_.Trim() }),

    [string]
    # Host's IP address
    $HostAddress =
    ($env:HOSTPRIVATEIPADDR | ? { $_ } | % { $_.Trim() }),

    [string]
    # UUID
    # eg. 93188440-b6d5-14b7-056b-08cd57910008
    $CurrentUuid = 
    ($env:CURRENT_UUID),
    
    [int]
    # Length of time to run in seconds
    $TimeToRunSeconds =
    ($env:TEST_NXAGING_TIME_SECONDS),

    [string]
    # eg. Tests\Outputs\Win32-v140\Tests\testNet_HttpServer_Win\Develop\testNet_HttpServer_Win.exe
    $TestListApp =
    ($env:TESTRUNNER_EXE_LIST)
)


# constants
$NX_BIT_TYPE = "SocketRamp-a64"
$MAX_SOCKETS = 32
$PORT_NUM = 81
$WAIT_TYPE_DEFAULT = "Select"
$TIME_RUN_WINDOWS_APP_SEC = 300


try
{
    # make sure Target Manager is connected
    & Tools\CommandLineTools\ControlTarget.exe connect

    # enable logging on the device
    Write-Host "Enable logging on the device SD card"
    & Tools\CommandLineTools\RunOnTarget.exe ./Programs/Eris/Outputs/NX-NXFP2-a64/TargetTools/DevMenuCommandSystem/Develop/DevMenuCommandSystem.nsp -- debug enable-sd-card-logging

    # apply network connection settings
    & Programs\Eris\Outputs\Win32-v140\TargetTools\SettingsManager\Develop\SettingsManager.exe Import Tests\Net\Resources\NetworkSettingsForCI.txt
    & Tools\CommandLineTools\RunOnTarget.exe ./Programs/Eris/Outputs/NX-NXFP2-a64/TargetTools/SettingsManager/Develop/SettingsManager.nsp -- Import Tests\Net\Resources\NetworkSettingsForCI.txt
    & Tools\CommandLineTools\ControlTarget.exe reset --target $TargetAddress
    & Tools\CommandLineTools\ControlTarget.exe connect

    # time parameters
    $timeToRunTestsSeconds = $TimeToRunSeconds
    $stopWatch = [Diagnostics.Stopwatch]::StartNew()
    $elapsedTime = 0
    $elapsedTimeSeconds = 0
    
    # find the Windows side test app file name
    $windowsAppFileNameNoExt = [System.IO.Path]::GetFileNameWithoutExtension($TestListApp)

    # run device side tests until time has elapsed
    while ($elapsedTimeSeconds -lt $timeToRunTestsSeconds)
    {
        # put device into sleep, and then wake
        & Tools\CommandLineTools\ControlTarget.exe press-power-button
        Start-sleep -s 5
        & Tools\CommandLineTools\ControlTarget.exe press-power-button
        Start-sleep -s 5
        
        # in case device is in the wrong sleep/wake state, wake device up
        & Tools\CommandLineTools\ControlTarget.exe power-on
        & Tools\CommandLineTools\ControlTarget.exe connect

        # run the Windows side of network test for the duration of the time limit
        Start-Process -FilePath $TestListApp -ArgumentList "--DurationSec=$TIME_RUN_WINDOWS_APP_SEC --PortNum=$PORT_NUM --NULL"
        Start-sleep -s 1

        # run device side test
        & Tools\CommandLineTools\RunOnTarget.exe Tests\Outputs\NX-NXFP2-a64\Tests\testNet_SoHttpSocketRamp\Develop\testNet_SoHttpSocketRamp.nsp -- --Name=$NX_BIT_TYPE --NetProfile=$CurrentUuid --MaxSockets=$MAX_SOCKETS --IpAddr=$HostAddress --PortNum=$PORT_NUM --Resource=/kb100 --TimeoutMs=300 """--Md5Hash=44,d0,88,ce,f1,36,d1,78,e9,c8,ba,84,c3,fd,f6,ca""" --WaitType=$WAIT_TYPE_DEFAULT --DoInit=false --NULL
 
        # kill the Windows side test app if it's still running
        $processName = Get-Process $windowsAppFileNameNoExt -ErrorAction SilentlyContinue
        if ($processName)
        {
            $processName | Stop-Process -force
        }
 
        # perform cradle dock/undock
        Start-sleep -s 5
        & Tools\CommandLineTools\ControlTarget.exe disable-cradle
        Start-sleep -s 5
        & Tools\CommandLineTools\ControlTarget.exe enable-cradle
        Start-sleep -s 5
        
        # check time elapsed
        $elapsedTime = $stopWatch.Elapsed
        $elapsedTimeSeconds = $elapsedTime.TotalSeconds
    }
}

finally
{
    # kill the Windows side test app if it's still running
    $processName = Get-Process $windowsAppFileNameNoExt -ErrorAction SilentlyContinue
    if ($processName)
    {
        $processName | Stop-Process -force
    }
}