<#
    Copyright (C)Nintendo All rights reserved.

    These coded instructions, statements, and computer programs contain proprietary
    information of Nintendo and/or its licensed developers and are protected by
    national and international copyright laws. They may not be disclosed to third
    parties or copied or duplicated in any form, in whole or in part, without the
    prior written consent of Nintendo.

    The content herein is highly confidential and should be handled accordingly.
#>

<#
    .SYNOPSIS
        Get BootImageHash from Externals/NxSystemImages/BootLoader
#>
[CmdletBinding()]
Param
(
    [string]
    $OutputName,

    [string]
    $TargetDirectory
)

$scriptPath          = $MyInvocation.MyCommand.Path
$scriptDirectoryPath = [System.IO.Path]::GetDirectoryName($scriptPath)

Import-Module "${scriptDirectoryPath}/../../Modules/Error"
Import-Module "${scriptDirectoryPath}/../../Modules/Path"
Import-Module "${scriptDirectoryPath}/../../Modules/Utility"
Import-Module "${scriptDirectoryPath}/../../Modules/Runner"
Import-Module "${scriptDirectoryPath}/../../Modules/UsbTarget"

$OutputDirectory = "$(Get-NintendoSdkRootPath)/Integrate/Outputs/BootImageHash"
$OutputFile      = "$OutputDirectory/$OutputName.bootimage-hashlist.txt"

trap [Exception]
{
    Write-ErrorRecord $_
    exit 1
}

# 出力ディレクトリの確保
if (!(Test-Path $OutputDirectory)) {
    mkdir $OutputDirectory > $null
}

# 既存ファイルの削除
if (Test-Path $OutputFile) {
    rm $OutputFile
}

function Get-NameAndHash
{
    param
    (
        [Parameter(Mandatory=$true)]
        [string]
        $FileName
    )
    
    Write-Output "$($(Get-Item $FileName).Name) $($(Get-FileHash $FileName).Hash)" 
}

function Get-NameAndHashTruncated
{
    param
    (
        [Parameter(Mandatory=$true)]
        [string]
        $FileName,
        
        [Parameter(Mandatory=$true)]
        [Int]
        $TruncatedSize        
    )
    
    $tempfile = [System.IO.Path]::GetTempFileName()
    New-TruncatedFile -Source $FileName -Output $tempfile -Size $TruncatedSize
    Write-Output "$($(Get-Item $FileName).Name) $($(Get-FileHash $tempfile).Hash)" 
    rm $tempfile 
}

# BCT のハッシュ値の計算と書き出し
Get-NameAndHash "$TargetDirectory/Normal-K1.bct" > $OutputFile 
Get-NameAndHash "$TargetDirectory/Normal-K2.bct" >> $OutputFile
Get-NameAndHash "$TargetDirectory/Normal-K3.bct" >> $OutputFile
Get-NameAndHash "$TargetDirectory/Recovery-K1.bct" >> $OutputFile
Get-NameAndHash "$TargetDirectory/Recovery-K2.bct" >> $OutputFile
Get-NameAndHash "$TargetDirectory/Recovery-K3.bct" >> $OutputFile
Get-NameAndHash "$TargetDirectory/Safe-K1.bct" >> $OutputFile
Get-NameAndHash "$TargetDirectory/Safe-K2.bct" >> $OutputFile
Get-NameAndHash "$TargetDirectory/Safe-K3.bct" >> $OutputFile

# BL の Truncate とハッシュ値の計算と書き出し
Get-NameAndHashTruncated "$TargetDirectory/Normal-K1.bl" -TruncatedSize 102400 >> $OutputFile 
Get-NameAndHashTruncated "$TargetDirectory/Normal-K2.bl" -TruncatedSize 102400 >> $OutputFile
Get-NameAndHashTruncated "$TargetDirectory/Normal-K3.bl" -TruncatedSize 102400 >> $OutputFile
Get-NameAndHashTruncated "$TargetDirectory/Recovery-K1.bl" -TruncatedSize 102400 >> $OutputFile
Get-NameAndHashTruncated "$TargetDirectory/Recovery-K2.bl" -TruncatedSize 102400 >> $OutputFile
Get-NameAndHashTruncated "$TargetDirectory/Recovery-K3.bl" -TruncatedSize 102400 >> $OutputFile

Get-Content $OutputFile