<#
    Copyright (C)Nintendo All rights reserved.

    These coded instructions, statements, and computer programs contain proprietary
    information of Nintendo and/or its licensed developers and are protected by
    national and international copyright laws. They may not be disclosed to third
    parties or copied or duplicated in any form, in whole or in part, without the
    prior written consent of Nintendo.

    The content herein is highly confidential and should be handled accordingly.
#>

<#
    .SYNOPSIS
        Initialize sdev by system updater.

    .DESCRIPTION
        Initialize sdev by system updater.
#>
[CmdletBinding()]
Param
(
    # The host IP address pattern as RegularExpression
    [string]
    $AddressPattern,

    [string]
    $BuildType = 'Develop',

    [string]
    $Platform = 'NX-NXFP2-a64',

    [string]
    $BootConfig = 'Skip', # Skip, <BootConfigFile>

    [string]
    $Target = '',

    [string]
    $KeyType = 'Auto', # K1, K2, Auto

    [string]
    $ConnectionType = 'Hb', # Hb, Usb

    [string]
    $RecoveryConnectionType = 'Auto', # Auto, Hb, Usb

    [string]
    $BootType = 'Nand', # Nand, List

    [string]
    $SignedType = 'Unsigned', # Unsigned, Signed

    [string]
    $StorageSize = '64G', # 32G, 64G

    [string]
    $DeveloperType = 'Internal', # Internal, Public

    [switch]
    $SkipHostBridgeUpdate = $false,

    [switch]
    $SkipStopTargetManager = $false,

    [switch]
    $SkipHelloWorld = $false,

    [switch]
    $AfterReset = $false,

    [switch]
    $SkipStopProcessesForCi = $false,

    [string]
    $Options = ''
)

$scriptPath          = $MyInvocation.MyCommand.Path
$scriptDirectoryPath = [System.IO.Path]::GetDirectoryName($scriptPath)

Import-Module "${scriptDirectoryPath}/../../Modules/Error"
Import-Module "${scriptDirectoryPath}/../../Modules/Path"
Import-Module "${scriptDirectoryPath}/../../Modules/HostBridge"
Import-Module "${scriptDirectoryPath}/../../Modules/Runner"
Import-Module "${scriptDirectoryPath}/../../Modules/Utility"
Import-Module "${scriptDirectoryPath}/../../Modules/InitializeTarget"

$WriteRecovery    = "$(Get-NintendoSdkRootPath)/Integrate/Scripts/NX/Recovery/Write-RecoverySystem.ps1"
$RestartHostBridge  = "$(Get-NintendoSdkRootPath)/Integrate/Scripts/NX/Recovery/Restart-HostBridge.ps1"
$UpdateHostBridge  = "$(Get-NintendoSdkRootPath)/Integrate/Scripts/Update-HostBridge.ps1"
$InitialzeSdev      = "$(Get-NintendoSdkRootPath)/Integrate/Scripts/NX/Initialize/Initialize-Sdev-Raw.ps1"

Test-ParameterIsNotNullOrEmpty -Name "AddressPattern" -Value $AddressPattern

trap [Exception]
{
    Write-ErrorRecord $_
    exit 1
}

if(!$SkipStopProcessesForCi)
{
    if($SkipStopTargetManager)
    {
        Stop-ProcessesForSystemInitialize -SkipStopTargetManager
    }
    else
    {
        Stop-ProcessesForSystemInitialize
    }
}


# Get the address of the HostBridge
$TargetAddress = Get-SigleTargetAddress -AddressPattern $AddressPattern

if(!$SkipHostBridgeUpdate)
{
    # Update HostBridge
    Invoke-CriticalCommand "& `"$UpdateHostBridge`" -AddressPattern `"$AddressPattern`""
}
else
{
    Write-Host "Skipping HostBridge update."
}

if($KeyType -eq 'Auto')
{
    $KeyType = Get-TargetFusedKeyType $TargetAddress
}

if($RecoveryConnectionType -eq 'Auto')
{
    $RecoveryConnectionType=$ConnectionType
}

$RecoveryWriterName = "RecoveryWriter-$KeyType-$RecoveryConnectionType-$SignedType"
$RecoveryWriter = "$(Get-NintendoSdkRootPath)/Programs/Eris/Outputs/$Platform/SystemImages/QspiBootImages/$RecoveryWriterName/$BuildType/$RecoveryWriterName.qspi.img"
$InitialImageName = "NX-$KeyType-$ConnectionType-$SignedType-$BootType-$StorageSize-$DeveloperType"
$InitialImagePath =  "$(Get-NintendoSdkRootPath)/Programs/Eris/Outputs/$Platform/InitialImages/$InitialImageName/$BuildType/$InitialImageName.initimg"

Write-Host "Initialization Parameters:"
Write-Host "  KeyType:        $KeyType"
Write-Host "  ConnectionType: $ConnectionType"
Write-Host "  RecoveryType:   $RecoveryConnectionType"
Write-Host "  BootType:       $BootType"
Write-Host "  SignedType:     $SignedType"
Write-Host "  StorageSize:    $StorageSize"
Write-Host "  DeveloperType:  $DeveloperType"
Write-Host "  BootConfig:     $BootConfig"

Test-EnsureExistPath $RecoveryWriter
Test-EnsureExistPath $InitialImagePath

# Write BootConfig
if($BootConfig -ne "Skip")
{
    $WriteBootConfig    = "$(Get-NintendoSdkRootPath)/Integrate/Scripts/NX/Recovery/Write-BootConfig.ps1"
    Invoke-CriticalCommand "& `"$WriteBootConfig`" -AddressPattern $AddressPattern -BootConfig $BootConfig -KeyType $KeyType"
}

# Write recovery system.
Invoke-CriticalCommand "& `"$WriteRecovery`" -AddressPattern $AddressPattern -BuildType $BuildType -Platform $Platform -RecoveryWriter $RecoveryWriter"

# Write serial number if serial number is not written.
if($KeyType -eq 'K1')
{
    $WriteSerial = "$(Get-NintendoSdkRootPath)/Integrate/Scripts/NX/Recovery/Write-SerialNumber.ps1"
    Invoke-CriticalCommand "& `"$WriteSerial`" -HostBridgeAddress $TargetAddress"
}
elseif(($KeyType -eq 'K5') -or ($KeyType -eq 'M1') -or ($KeyType -eq 'M2'))
{
    $WriteSerial = "$(Get-NintendoSdkRootPath)/Integrate/Scripts/NX/Recovery/Write-SerialNumber.ps1"
    Invoke-CriticalCommand "& `"$WriteSerial`" -HostBridgeAddress $TargetAddress -WriteName"
}

if([String]::IsNullOrEmpty($Target))
{
    $TargetOption = "-Target $TargetAddress"
}
else
{
    $TargetOption = "-Target $Target"
}

if($SkipHelloWorld)
{
    $TargetOption += " -SkipHelloWorld"
}

# Write the InitialImage.
Invoke-CriticalCommand "& `"$InitialzeSdev`" $TargetOption -AddressPattern $AddressPattern -BuildType $BuildType -Platform $Platform -InitialImage `"$InitialImagePath`" -AfterReset:`$$AfterReset -Options `"$Options`""
