<#
    Copyright (C)Nintendo All rights reserved.

    These coded instructions, statements, and computer programs contain proprietary
    information of Nintendo and/or its licensed developers and are protected by
    national and international copyright laws. They may not be disclosed to third
    parties or copied or duplicated in any form, in whole or in part, without the
    prior written consent of Nintendo.

    The content herein is highly confidential and should be handled accordingly.
#>

<#
    .SYNOPSIS
        Initialize edev by system updater.

    .DESCRIPTION
        Initialize edev by system updater.
#>
[CmdletBinding()]
Param
(
    # This parameter is not used. (It is prepared to make a format even with Initialize-Sdev-*.)
    [string]
    $AddressPattern = '',

    [string]
    $BuildType = 'Develop',

    [string]
    $Platform = 'NX-NXFP2-a64',

    [string]
    $BootConfig = 'Skip', # Skip, Auto, <BootConfigFile>

    [string]
    $KeyType = 'K5',

    [string]
    $ConnectionType = "Usb",

    [string]
    $BootType = 'Nand', # Nand, List

    [string]
    $SignedType = 'Unsigned', # Unsigned, Signed

    [string]
    $StorageSize = '64G', # 32G, 64G

    [string]
    $DeveloperType = 'Internal',  # Internal, Public

    [string]
    $CalImageFile = '',

    [string]
    $AdditionalOptions = '',

    [Int]
    $DetectTargetRetryCount = 5,

    [switch]
    $SkipStopTargetManager = $false,

    [switch]
    $SkipStopProcessesForCi = $false
)

Write-Host "Initialization Parameters:"
Write-Host "  BuildType:      $BuildType"
Write-Host "  Platform:       $Platform"
Write-Host "  KeyType:        $KeyType"
Write-Host "  ConnectionType: $ConnectionType"
Write-Host "  BootType:       $BootType"
Write-Host "  SignedType:     $SignedType"
Write-Host "  StorageSize:    $StorageSize"
Write-Host "  DeveloperType:  $DeveloperType"

$scriptPath          = $MyInvocation.MyCommand.Path
$scriptDirectoryPath = [System.IO.Path]::GetDirectoryName($scriptPath)

Import-Module "${scriptDirectoryPath}/../../Modules/Error"
Import-Module "${scriptDirectoryPath}/../../Modules/Path"
Import-Module "${scriptDirectoryPath}/../../Modules/Utility"
Import-Module "${scriptDirectoryPath}/../../Modules/Runner"
Import-Module "${scriptDirectoryPath}/../../Modules/UsbTarget"

$RcmWriter          = "$(Get-NintendoSdkRootPath)/Externals/NxSystemImages/RcmWriter/RcmWriter.exe"
$RcmWriterRcm       = "$(Get-NintendoSdkRootPath)/Externals/NxSystemImages/RcmWriter/RcmWriter.$KeyType.rcm"
$RcmWriterRcmOld    = "$(Get-NintendoSdkRootPath)/Externals/NxSystemImages/RcmWriter/RcmWriter.$KeyType.old.rcm"
$RcmWriterAdditionalOptions = ""

if($KeyType -eq 'M1')
{
    $RcmWriterXmlWoBc   = "$(Get-NintendoSdkRootPath)/Integrate/Sources/Configs/RcmWriter/RecoverySystem-Generic-M1.xml"
    $RcmWriterXmlWithBc = "$(Get-NintendoSdkRootPath)/Integrate/Sources/Configs/RcmWriter/RecoverySystem-Generic-WithBootConfig-M1.xml"
}
elseif($KeyType -eq 'M2')
{
    $RcmWriterXmlWoBc   = "$(Get-NintendoSdkRootPath)/Integrate/Sources/Configs/RcmWriter/RecoverySystem-Generic-M2.xml"
    $RcmWriterXmlWithBc = "$(Get-NintendoSdkRootPath)/Integrate/Sources/Configs/RcmWriter/RecoverySystem-Generic-WithBootConfig-M2.xml"
    $RcmWriterAdditionalOptions = "--old_rcm `"$RcmWriterRcmOld`""
}
else
{
    $RcmWriterXmlWoBc   = "$(Get-NintendoSdkRootPath)/Integrate/Sources/Configs/RcmWriter/RecoverySystem-Generic.xml"
    $RcmWriterXmlWithBc = "$(Get-NintendoSdkRootPath)/Integrate/Sources/Configs/RcmWriter/RecoverySystem-Generic-WithBootConfig.xml"
}
$BootConfigInRepo   = "$(Get-NintendoSdkRootPath)/Externals/NxBootConfigs/bc-$BootConfig.bin"
$BootConfigDummy    = "$(Get-NintendoSdkRootPath)/Externals/NxBootConfigs/bc-dummy.bin"
$Bct                = "$(Get-NintendoSdkRootPath)/Externals/NxSystemImages/BootLoader/Recovery-$KeyType.bct"
$Bl                 = "$(Get-NintendoSdkRootPath)/Externals/NxSystemImages/BootLoader/Recovery-$KeyType.bl"
$Gpt                = "$(Get-NintendoSdkRootPath)/Externals/NxSystemImages/Recovery/gpt-recovery.img"
$Package2Unsigned   = "$(Get-NintendoSdkRootPath)/Programs/Eris/Outputs/$Platform/SystemImages/Package2Images/DefaultPackage2ForInitialize/$BuildType/DefaultPackage2ForInitialize.package2.img"
$Package2Signed     = "$(Get-NintendoSdkRootPath)/Externals\NxSystemImages\Package2Signed\$BuildType\p2initialize.bin"
$SystemPartitionP2Unsigned = "$(Get-NintendoSdkRootPath)/Programs/Eris/Outputs/$Platform/SystemImages/SystemPartition/RecoverySystemPartitionWithUsb-P2Unsigned/$BuildType/RecoverySystemPartitionWithUsb-P2Unsigned.fatimg"
$SystemPartitionP2Signed   = "$(Get-NintendoSdkRootPath)/Programs/Eris/Outputs/$Platform/SystemImages/SystemPartition/RecoverySystemPartitionWithUsb-P2Signed/$BuildType/RecoverySystemPartitionWithUsb-P2Signed.fatimg"
$RunOnTarget        = "$(Get-NintendoSdkRootPath)/Tools/CommandLineTools/RunOnTargetPrivate.exe"
$ControlTarget      = "$(Get-NintendoSdkRootPath)/Tools/CommandLineTools/ControlTarget.exe"
$InitialImageName   = "NX-$KeyType-$ConnectionType-$SignedType-$BootType-$StorageSize-$DeveloperType"
$InitialImagePath   = "$(Get-NintendoSdkRootPath)/Programs/Eris/Outputs/$Platform/InitialImages/$InitialImageName/$BuildType/$InitialImageName.initimg"
$SystemUpdater      = "$(Get-NintendoSdkRootPath)/Programs/Chris/Outputs/$Platform/TargetTools/SystemUpdaterHostFs/$BuildType/SystemUpdaterHostFs.nsp"

$OutputDirectory    = "$(Get-NintendoSdkRootPath)/Integrate/Outputs/RcmWriter"
$OutputConfig       = "$OutputDirectory/RecoverySystem-Generic.xml"
$OutputBct          = "$OutputDirectory/Bct.img"
$OutputBl           = "$OutputDirectory/Bl.img"
$OutputBootConfig   = "$OutputDirectory/BootConfig.img"
$OutputGpt          = "$OutputDirectory/Gpt.img"
$OutputPackage2     = "$OutputDirectory/Package2.img"
$OutputSystemPartition = "$OutputDirectory/SystemPartition.fatimg"

trap [Exception]
{
    Write-ErrorRecord $_
    exit 1
}

if (!(Test-Path $OutputDirectory)) {
    mkdir $OutputDirectory > $null
}

if($BootConfig -ne "Skip")
{
    $RcmWriterXml = $RcmWriterXmlWithBc

    if($SignedType -ne "Signed")
    {
        if(Test-Path $BootConfig)
        {
            $BootConfigFile = $BootConfig
        }
        elseif(Test-Path $BootConfigInRepo)
        {
            $BootConfigFile = $BootConfigInRepo
        }
        else
        {
            throw "Not found the boot config file: $BootConfig"
        }
    }
    else
    {
        if(Test-Path $BootConfig)
        {
            $BootConfigFile = $BootConfig
        }
        else
        {
            $BootConfigFile = $BootConfigDummy
        }
    }

    cp $BootConfigFile $OutputBootConfig
    Write-Host "  BootConfigFile:  $BootConfigFile"
}
else
{
    if($SignedType -ne "Signed")
    {
        $RcmWriterXml = $RcmWriterXmlWoBc
    }
    else
    {
        # Enforce to write BootConfig on Signed system
        $RcmWriterXml = $RcmWriterXmlWithBc
        $BootConfigFile = $BootConfigDummy
        cp $BootConfigFile $OutputBootConfig
        Write-Host "  BootConfigFile:  $BootConfigFile"
    }
}

if($SignedType -eq "Signed")
{
    $Package2 = $Package2Signed
    $SystemPartition = $SystemPartitionP2Signed
}
else
{
    $Package2 = $Package2Unsigned
    $SystemPartition = $SystemPartitionP2Unsigned
}

$CalImageOption = ""
if(($CalImageFile -ne "") -and (Test-Path $CalImageFile))
{
    Write-Host "Write Cal Image."
    $CalImageOption = "--cal-image $CalImageFile"
}

# Test existing files
Test-EnsureExistPath $RcmWriter
Test-EnsureExistPath $RcmWriterRcm
Test-EnsureExistPath $RcmWriterXml
Test-EnsureExistPath $SystemUpdater
Test-EnsureExistPath $RunOnTarget
Test-EnsureExistPath $ControlTarget
Test-EnsureExistPath $InitialImagePath
Test-EnsureExistPath $Bct
Test-EnsureExistPath $Bl
Test-EnsureExistPath $Gpt
Test-EnsureExistPath $Package2
Test-EnsureExistPath $SystemPartition

if(!$SkipStopProcessesForCi)
{
    if($SkipStopTargetManager)
    {
        Stop-ProcessesForSystemInitialize -SkipStopTargetManager
    }
    else
    {
        Stop-ProcessesForSystemInitialize
        Unregister-AllTargets
    }
}

# Copy files

cp $RcmWriterXml $OutputConfig
cp $Bct $OutputBct
cp $Bl $OutputBl
cp $Gpt $OutputGpt
cp $Package2 $OutputPackage2
cp $SystemPartition $OutputSystemPartition

# RcmWriter
Invoke-CriticalCommand "& `"$RcmWriter`" --rcm `"$RcmWriterRcm`" --list `"$OutputConfig`" $RcmWriterAdditionalOptions"

# Detect target
Write-Host "Find USB target device..."
Get-SigleUsbTargetName -RetryCount $DetectTargetRetryCount  | Out-Null

# Write
Invoke-CriticalCommand "& `"$RunOnTarget`" run --target usb --verbose --failure-timeout 600 --pattern-success-exit `"Succeeded initializing the system.`" `"$SystemUpdater`" '--' --input `"$InitialImagePath`" $CalImageOption $AdditionalOptions"
