﻿<#
    Copyright (C)Nintendo All rights reserved.

    These coded instructions, statements, and computer programs contain proprietary
    information of Nintendo and/or its licensed developers and are protected by
    national and international copyright laws. They may not be disclosed to third
    parties or copied or duplicated in any form, in whole or in part, without the
    prior written consent of Nintendo.

    The content herein is highly confidential and should be handled accordingly.
#>

<#
    .SYNOPSIS
        Yaml Module

    .DESCRIPTION
        This file defines functions for processing YAML
#>


$moduleFilePath = $MyInvocation.MyCommand.Path
$modulePath     = [System.IO.Path]::GetDirectoryName($moduleFilePath)
$moduleRootPath = [System.IO.Path]::GetDirectoryName($modulePath)

Import-Module "${moduleRootPath}\Path"

<#
    .SYNOPSIS
        Converts a YAML-formatted string to an XML object

    .OUTPUTS
        XmlDocument instance
#>
function ConvertFrom-Yaml
{
    param
    (
        [Parameter(Mandatory=$true)]
        [string]
        # YAML-formatted string
        $InputObject
    )

    $str = $InputObject

    $result = & {
        try
        {
            $enc = $OutputEncoding
            $OutputEncoding = [Text.Encoding]::GetEncoding('SJIS')
            return ,($str | & (Get-YamlConverterPath) --yaml-to-xml 2>&1)
        }
        finally
        {
            $OutputEncoding = $enc
        }
    }

    if ($LastExitCode -ne 0)
    {
        $result | %{
            $_ -csplit [System.Environment]::NewLine | %{
                if ("" -cne $_) { $Host.UI.WriteErrorLine("> $_") }
            }
        }

        throw "format conversion failed"
    }

    return ,[xml]($result -join [System.Environment]::NewLine)
}
Export-ModuleMember -Function ConvertFrom-Yaml

<#
    .SYNOPSIS
        Converts an XML object to a YAML-formatted string

    .OUTPUTS
        String instance
#>
function ConvertTo-Yaml
{
    param
    (
        [Parameter(Mandatory=$true)]
        [xml]
        # XML object
        $InputObject
    )

    $str = &{
        try
        {
            $writer = New-Object "System.IO.StringWriter"
            $InputObject.Save($writer)
            return $writer.ToString()
        }
        finally
        {
            $writer.Close()
        }
    }

    $result = & {
        try
        {
            $enc = $OutputEncoding
            $OutputEncoding = [Text.Encoding]::GetEncoding('SJIS')
            return ,($str | & (Get-YamlConverterPath) --xml-to-yaml 2>&1)
        }
        finally
        {
            $OutputEncoding = $enc
        }
    }

    if ($LastExitCode -ne 0)
    {
        $result | %{
            $_ -csplit [System.Environment]::NewLine | %{
                if ("" -cne $_) { $Host.UI.WriteErrorLine("> $_") }
            }
        }

        throw "format conversion failed"
    }

    return ,($result -join [System.Environment]::NewLine)
}
Export-ModuleMember -Function ConvertTo-Yaml
