﻿<#
    Copyright (C)Nintendo. All rights reserved.

    These coded instructions, statements, and computer programs contain proprietary
    information of Nintendo and/or its licensed developers and are protected by
    national and international copyright laws. They may not be disclosed to third
    parties or copied or duplicated in any form, in whole or in part, without the
    prior written consent of Nintendo.
 
    The content herein is highly confidential and should be handled accordingly.
#>

Import-Module $PSScriptRoot/../Path

<#
    .SYNOPSIS
    複数のパスを結合します。

    .DESCRIPTION
    複数のパスを結合します。
    引数で指定された配列の全ての組み合わせのパスを生成します。
    最大 5つまで引数を指定できます。
    ただし、最初のパス要素はパイプラインで指定します。

    .INPUTS
    1つめのパス要素。

    .OUTPUTS
    組み合わせたパス文字列。
#>
function Join-ManyPath
{
    [CmdletBinding()]
    Param(
        # 1つめのパス要素を指定します。
        [parameter(Mandatory = $true,
                   ValueFromPipeline = $true)]
        [string[]] $Path1,

        # 2つめのパス要素を指定します。
        [parameter(Mandatory = $true,
                   Position = 0)]
        [string[]] $Path2,

        # 3つめのパス要素を指定します。
        [parameter(Position = 1)]
        [string[]] $Path3,

        # 4つめのパス要素を指定します。
        [parameter(Position = 2)]
        [string[]] $Path4,

        # 5つめのパス要素を指定します。
        [parameter(Position = 3)]
        [string[]] $Path5
    )

    process {
        foreach ($p1 in $Path1) {
            foreach ($p2 in $Path2) {
                $p12 = Join-Path $p1 $p2
                if (!$Path3) {
                    $p12
                } else {
                    foreach ($p3 in $Path3) {
                        $p123 = Join-Path $p12 $p3
                        if (! $Path4) {
                            $p123
                        } else {
                            foreach ($p4 in $Path4) {
                                $p1234 = Join-Path $p123 $p4
                                if (! $Path5) {
                                    $p1234
                                } else {
                                    foreach ($p5 in $Path5) {
                                        Join-Path $p1234 $p5
                                    }
                                }
                            }
                        }
                    }
                }
            }
        }
    }
}

Export-ModuleMember Join-ManyPath

<#
    .SYNOPSIS
    zipファイルを解凍します。

    .DESCRIPTION
    zipファイルを解凍します。
    出力ディレクトリにzipファイル名を付けて解凍します。

    .INPUTS
    なし。

    .OUTPUTS
    なし。
#>
function Expand-Zip
{
    [CmdletBinding()]
    Param(
        # zip ファイルのパスを指定します。
        # このパラメータは Get-ChildItem コマンドレットに渡します。
        [parameter(Mandatory = $true)]
        [string[]] $Path,

        # 解凍先のディレクトリを指定します。
        [parameter(Mandatory = $true)]
        [string] $Output
    )

    $7Zip = Get-7ZipPath

    Get-ChildItem $Path |
    ForEach-Object {
        & $7Zip x $_.FullName ("-o" + $Output) | Where-Object { ! $_.StartsWith("Extracting  ") }
    }
}

Export-ModuleMember Expand-Zip

<#
    .SYNOPSIS
    zip ファイルの名前(拡張子を除く)を取得します。

    .DESCRIPTION
    zip ファイルの名前(拡張子を除く)を取得します。

    .INPUTS
    なし。

    .OUTPUTS
    なし。
#>
function Get-ZipName
{
    [CmdletBinding()]
    Param(
        # zip ファイルのパスを指定します。
        # このパラメータは Get-ChildItem コマンドレットに渡します。
        [parameter(Mandatory = $true)]
        [string[]] $Path
    )

    Get-ChildItem $Path -Name -Recurse -Include *.zip |
    ForEach-Object {
        $_ -replace '\.zip$',''
    }
}

Export-ModuleMember Get-ZipName

<#
    .SYNOPSIS
    存在しているパスのみ選択します。

    .DESCRIPTION
    存在しているパスのみ選択します。
    存在していないパスは破棄されます。

    .INPUTS
    パイプを使用してパスを含む文字列をコマンドレットに渡すことができます。

    .OUTPUTS
    存在しているパス文字列。
#>
function Select-ExistPath
{
    [CmdletBinding()]
    Param(
        # パスを指定します。
        [parameter(Mandatory = $true,
                   ValueFromPipeline = $true)]
        [string[]] $Path
    )
    
    process {
        $Path |
        Where-Object { Test-Path $_ }
    }
}

Export-ModuleMember Select-ExistPath

<#
    .SYNOPSIS
    相対パスを取得します。

    .DESCRIPTION
    相対パスを取得します。

    .INPUTS
    パイプを使用してパスを含む文字列をコマンドレットに渡すことができます。

    .OUTPUTS
    パス文字列。
#>
function Get-RelativePath
{
    [CmdletBinding()]
    Param(
        # 相対パスの起点となるディレクトリへのパスを指定します。
        [parameter(Mandatory = $true)]
        [string] $Root,

        # パスを指定します。
        [parameter(Mandatory = $true,
                   ValueFromPipeline = $true,
                   ValueFromPipelineByPropertyName = $true)]
        [alias("PSPath")]
        [string[]] $Path
    )

    begin {
        $rootPathLen = (Resolve-Path $Root).Path.Length + 1
    }
    process {
        foreach ($p in $Path) {
            if (Split-Path $p -IsAbsolute) {
                $p = Split-Path $p -NoQualifier
            } else {
                $p = Join-Path $PWD $p
            }

            $fullPath = [IO.Path]::GetFullPath($p)
            $fullPath.Substring($rootPathLen)
        }
    }
}

Export-ModuleMember Get-RelativePath

<#
    .SYNOPSIS
    exe, ell のパスからサテライトアセンブリのパスを取得します。

    .DESCRIPTION
    exe, ell のパスからサテライトアセンブリのパスを取得します。

    .INPUTS
    パイプを使用してパスを含む文字列をコマンドレットに渡すことができます。

    .OUTPUTS
    パス文字列。
#>
function Get-ResourceDllName
{
    [CmdletBinding()]
    Param(
        # exeまたはdllのパスを指定します。
        [parameter(Mandatory = $true,
                   ValueFromPipeline = $true,
                   Position = 0)]
        [string[]] $Path,

        # 言語ディレクトリを指定します。既定値はenです。
        [string] $Lang = 'en'
    )

    process {
        foreach ($p in $Path) {
            $fileName = [IO.Path]::GetFileName($p)

            Join-Path ([IO.Path]::GetDirectoryName($p)) en |
            Join-Path -ChildPath ([IO.Path]::ChangeExtension($fileName, ".resources.dll"))
        }
    }
}

Export-ModuleMember Get-ResourceDllName

<#
    .SYNOPSIS
    ディレクトリツリーをコピーします。

    .DESCRIPTION
    ディレクトリツリーをコピーします。

    .INPUTS
    パイプを使用してパスを含む文字列をコマンドレットに渡すことができます。

    .OUTPUTS
    パス文字列。
#>
function Copy-TreeFile
{
    [CmdletBinding()]
    Param(
        # 新しい場所のパスを指定します。
        [parameter(Mandatory = $true)]
        [string] $Source,

        # 新しい場所のパスを指定します。
        [parameter(Mandatory = $true)]
        [string] $Destination,

        # コピー対象の項目のパスを指定します。
        [parameter(Mandatory = $true,
                   ValueFromPipeline = $true)]
        [string[]] $Path
    )

    process {
        foreach ($p in $Path) {
            $src = Join-Path $Source $p
            $dst = Join-Path $Destination $p

            $dstDir = Split-Path $dst -Parent
            if (!(Test-Path $dstDir)) {
                mkdir $dstDir > $null
            }
            Copy-Item $src $dst
        }
    }
}

Export-ModuleMember Copy-TreeFile

<#
    .SYNOPSIS
    変換リストファイルをインポートします。

    .DESCRIPTION
    変換リストファイルをインポートします。

    .INPUTS
    なし。

    .OUTPUTS
    変換対象と変換後のパスを含むPSObject。
#>
function Import-TranslateList
{
    [CmdletBinding()]
    Param (
        # 変換リストファイルのパスを指定します。
        [parameter(
            Mandatory = $true)]
        [string[]]$Path
    )

    Get-Content $Path |
    ForEach-Object {
        $transPair = ($_ -replace '/','\') -split ','
        New-Object PSObject -Property @{ Target = $transPair[0]; Replace = $transPair[1] }
    }
}

Export-ModuleMember Import-TranslateList

<#
    .SYNOPSIS
    パスを置換します。

    .DESCRIPTION
    パスを置換します。

    .INPUTS
    なし。

    .OUTPUTS
    置換されたパス文字列。
#>
function Get-ReplacePath
{
    [CmdletBinding()]
    Param (
        # 変換対象のパスと変換後のパスを含むPSObjectを指定します。
        [parameter(
            Mandatory = $true)]
        $Translate,

        # 置換するパス文字列を指定します。
        [parameter(
            Mandatory = $true,
            ValueFromPipeline = $true)]
        [string[]]$Path
    )

    process {
        foreach ($pathStr in $Path) {
            $incPath = $pathStr -replace '/','\'
            foreach ($rep in $Translate) {
                if ($incPath.StartsWith($rep.Target, [StringComparison]::OrdinalIgnoreCase)) {
                    $incPath = $rep.Replace + $incPath.Substring($rep.Target.Length)
                    break
                }
            }

            $incPath
        }
    }
}

Export-ModuleMember Get-ReplacePath

<#
    .SYNOPSIS
    ファイルを置換します。

    .DESCRIPTION
    ファイルを置換します。
    置換先ディレクトリに存在しないファイルは何も行われません。
    パスの変換ルールが指定された場合は、変換されたパスでファイルを置換します。

    .INPUTS
    なし。

    .OUTPUTS
    なし。
#>
function Set-ExistFile
{
    Param(
        # 置き換えたいファイルがあるディレクトリのルートのパスを指定します。
        [parameter(Mandatory = $true,
                   Position = 0)]
        [string] $Path,

        # 置換先のパスを指定します。
        [parameter(Mandatory = $true,
                   Position = 1)]
        [string] $Destination,

        # パス変換リストを指定します。
        $TranslateList
    )

    $srcRootLen = (Resolve-Path $Path).Path.Length + 1

    Get-ChildItem $Path -Recurse |
    Where-Object { ! $_.PSIsContainer } |
    ForEach-Object {
        $srcRelPath = $_.FullName.Substring($srcRootLen)
        if ($TranslateList -and !$srcRelPath.StartsWith('Include\', [StringComparison]::OrdinalIgnoreCase)) {
            $repPath =
                $srcRelPath.Replace('\','/') `
                -replace '^Externals/[^/]+/Programs/([^/]+)/Include/','Programs/$1/Outputs/Include/'
            $dstRelPath = $repPath | Get-ReplacePath $TranslateList
        } else {
            $dstRelPath = $srcRelPath
        }
        $dst = Join-Path $Destination $dstRelPath

        # 上書き側に存在するファイルのみ上書き
        if (Test-Path $dst) {
            # New-Object PSObject -Property @{ Path = $_.FullName; Destination = $dst }
            Copy-Item $_.FullName $dst
        }
    }
}

Export-ModuleMember Set-ExistFile

<#
    .SYNOPSIS
    zip ファイルを作成します。

    .DESCRIPTION
    zip ファイルを作成します。

    .INPUTS
    なし。

    .OUTPUTS
    なし。
#>
function New-Zip
{
    [CmdletBinding()]
    Param(
        # zipファイルのパスを指定します。
        [parameter(Mandatory = $true)]
        [string] $Path,

        # zipに含めるファイルのルートディレクトリを指定します。
        [parameter(Mandatory = $true)]
        [string] $InputDir
    )

    $7Zip = Get-7ZipPath

    $zipDir = Split-Path $Path -Parent
    if (! (Test-Path $zipDir)) {
        mkdir $zipDir > $null
    }
    $fullPath = Resolve-Path $zipDir | Join-Path -ChildPath (Split-Path $Path -Leaf)

    Push-Location $InputDir
    & $7Zip a $fullPath | Where-Object { ! $_.StartsWith("Compressing  ") }
    Pop-Location
}

Export-ModuleMember New-Zip

<#
    .SYNOPSIS
    SHA1 ハッシュを取得します。

    .DESCRIPTION
    SHA1 ハッシュを取得します。

    .INPUTS
    パイプを使用してパスを含む文字列をコマンドレットに渡すことができます。

    .OUTPUTS
    SHA1ハッシュの16進文字列。
#>
function Get-Sha1String
{
    [CmdletBinding()]
    Param(
        # Security.Cryptography.SHA1CryptoServiceProvider のインスタンスを指定します。
        [parameter(Mandatory = $true)]
        $hasher,

        # ハッシュを取得するファイルの FileInfo オブジェクトを指定します。
        [parameter(Mandatory = $true,
                   ValueFromPipeline = $true)]
        [IO.FileInfo[]]$File
    )

    begin {
        $OFS = ""
    }
    process {
        foreach ($fi in $File) {
            try {
                $fs = $fi.OpenRead()

                $hashStrs =
                    $hasher.ComputeHash($fs) |
                    ForEach-Object { $_.ToString("x2") }

                [string]$hashStrs
            } finally {
                if ($fs) { $fs.Dispose() }
            }
        }
    }
}

<#
    .SYNOPSIS
    Zarf シグネチャファイルを作成します。

    .DESCRIPTION
    Zarf シグネチャファイルを作成します。

    .INPUTS
    なし。

    .OUTPUTS
    なし。
#>
function New-ZarfSignature
{
    [CmdletBinding()]
    Param(
        # Zarfシグネチャファイルへのパスを指定します。
        [parameter(Mandatory = $true)]
        [string] $Path,

        # Zarfに含めるファイルのルートディレクトリを指定します。
        [parameter(Mandatory = $true)]
        [string] $InputDir
    )

    try {
        $hasher = New-Object Security.Cryptography.SHA1CryptoServiceProvider
        $InputDirStrLen = (Resolve-Path $InputDir).Path.Length + 1

        Get-ChildItem $InputDir -Recurse |
        Where-Object { !$_.PSIsContainer } |
        ForEach-Object {
            $path = $_.FullName.Substring($InputDirStrLen)
            "{0}`t{1}.{2}" -f $path, (Get-Sha1String $hasher $_), $_.Length
        } |
        Set-Content $Path -Encoding String
    } finally {
        if ($hasher) {
            # .NET 4 以降
            # $hasher.Dispose()
        }
    }
}

<#
    .SYNOPSIS
    Zarf ファイルを作成します。

    .DESCRIPTION
    Zarf ファイルを作成します。

    .INPUTS
    なし。

    .OUTPUTS
    なし。
#>
function New-Zarf
{
    [CmdletBinding()]
    Param(
        # Zarf に含むアーカイブを指定します。
        [parameter(Mandatory = $true)]
        [string] $Archive,

        # Zarf 定義ファイルを指定します。
        [parameter(Mandatory = $true)]
        [string] $Source,

        # Zarf ファイルの出力先を指定します。
        [parameter(Mandatory = $true)]
        [string] $Output,

        # Zarf に含むアーカイブのファイルがあるルートディレクトリを指定します。
        [parameter(Mandatory = $true)]
        [string] $Root
    )

    $sigloRoot = (Resolve-Path $PSScriptRoot/../../../..).Path

    # .signature の作成
    $signaturePath = [IO.Path]::ChangeExtension($Archive, ".signature")
    New-ZarfSignature $signaturePath $Root

    if (!(Test-Path $Output)) {
        mkdir $Output > $null
    }

    & $sigloRoot/Integrate/Outputs/AnyCPU/Tools/ZarfCreator/ZarfCreator/Release/ZarfCreator `
        --root $sigloRoot `
        --output-dir $Output `
        --source-file $Source `
        --archive-path $Archive
}

Export-ModuleMember New-Zarf
