﻿<#
    Copyright (C)Nintendo All rights reserved.

    These coded instructions, statements, and computer programs contain proprietary
    information of Nintendo and/or its licensed developers and are protected by
    national and international copyright laws. They may not be disclosed to third
    parties or copied or duplicated in any form, in whole or in part, without the
    prior written consent of Nintendo.

    The content herein is highly confidential and should be handled accordingly.
#>

<#
    .SYNOPSIS
        Invoke TestRunner

    .DESCRIPTION
        Run TestRunner for performing automated tests
#>

param
(
    [switch]
    # Write test execution logs to the display
    $Verbose,

    [switch]
    # Return exit code 0 except for internal error
    $QuietExit,

    [switch]
    # List files that are mentioned in the testlists
    $FileListing,

    [int]
    # The maximum number of jobs
    $Jobs = $env:NNTEST_JOBS,

    [string[]]
    # Path to the directories of testlists
    $Path =
    ($env:NNTEST_PATH | ? { $_ } | % { $_.Split(",") } | % { $_.Trim() }),

    [string[]]
    # Path to the testlists
    $File =
    ($env:NNTEST_FILE | ? { $_ } | % { $_.Split(",") } | % { $_.Trim() }),

    [string[]]
    # Qualify the platforms
    $Platform =
    ($env:NNTEST_PLATFORM | ? { $_ } | % { $_.Split(",") } | % { $_.Trim() }),

    [string[]]
    # Qualify the build types
    $BuildType =
    ($env:NNTEST_BUILD_TYPE | ? { $_ } | % { $_.Split(",") } | % { $_.Trim() }),

    [string[]]
    # Qualify the modules
    $Module =
    ($env:NNTEST_MODULE | ? { $_ } | % { $_.Split(",") } | % { $_.Trim() }),

    [string[]]
    # Qualify the test categories
    $Category =
    ($env:NNTEST_CATEGORY | ? { $_ } | % { $_.Split(",") } | % { $_.Trim() }),

    [string]
    # Path to the program name database
    $ProgramNameDatabase = $env:NNTEST_PROGRAM_NAME_DATABASE,

    [string]
    # Pattern that target names must match
    $TargetNamePattern = $env:NNTEST_TARGET_NAME_PATTERN,

    [string]
    # Pattern that target interfaces must match
    $TargetInterfacePattern = $env:NNTEST_TARGET_INTERFACE_PATTERN,

    [string]
    # Pattern that target addresses must match
    $TargetAddressPattern = $env:NNTEST_TARGET_ADDRESS_PATTERN
)

$scriptPath          = $MyInvocation.MyCommand.Path
$scriptDirectoryPath = [System.IO.Path]::GetDirectoryName($scriptPath)

Import-Module "${scriptDirectoryPath}\Modules\Error"
Import-Module "${scriptDirectoryPath}\Modules\Path"
Import-Module "${scriptDirectoryPath}\Modules\Git"

trap [Exception]
{
    Write-ErrorRecord $_
    exit 1
}

if ( Test-Path "env:\NNTEST_IS_CANCELLED" )
{
    Write-Host "This build has been cancelled"
    exit 0
}

$env:NNTEST_BUILD_SYSTEM_ROOT = Get-NintendoSdkRootPath

$env:PATH = "$(Get-NintendoSdkCommandLineToolsRootPath);${env:PATH}"
$env:PATH = "$(Get-NintendoSdkRootPath)\Externals\Binaries\Ninja;${env:PATH}"

$env:PLATFORM = ""

if ( -not (Test-Path "env:\NINTENDO_SDK_ROOT") )
{
    $env:NINTENDO_SDK_ROOT = Get-NintendoSdkRootPath
}

if ( -not (Test-Path "env:\NNTEST_RUNNER_TOOLS_PATH") )
{
    $env:NNTEST_RUNNER_TOOLS_PATH =
        "${env:NNTEST_BUILD_SYSTEM_ROOT}\Tools\CommandLineTools"
}

$option = New-Object "System.Collections.Generic.List[string]"

$outputPath = "$(Get-NintendoSdkIntegrateOutputRootPath)\TestRunner"

if ( -not (Test-Path $outputPath) )
{
    New-Item $outputPath -itemType Directory | Out-Null
}

$option.Add("--output `"${outputPath}`"")

try
{
    $option.Add("--branch-name $(Get-GitNintendoSdkBranch)")
}
catch [System.Management.Automation.RuntimeException]
{
    $option.Add("--branch-name N/A")
}

try
{
    $option.Add("--commit-hash $(Get-GitNintendoSdkCommit)")
}
catch [System.Management.Automation.RuntimeException]
{
    $option.Add("--commit-hash N/A")
}

if ( $Verbose )
{
    $option.Add("--verbose")
}

if ( $QuietExit )
{
    $option.Add("--quiet-exit")
}

if ( $FileListing )
{
    $option.Add("--file-listing")
}

if ( 0 -lt $Jobs )
{
    $option.Add("--jobs ${Jobs}")
}

if ( $null -ne $Path )
{
    $Path | % { $option.Add("--path `"$_`"") }
}

if ( $null -ne $File )
{
    $File | % { $option.Add("--file `"$_`"") }
}

if ( $null -ne $Platform )
{
    $Platform | % { $option.Add("--platform $_") }
}

if ( $null -ne $BuildType )
{
    $BuildType | % { $option.Add("--build-type $_") }
}

if ( $null -ne $Module )
{
    $Module | % { $option.Add("--module $_") }
}

if ( $null -ne $Category )
{
    $Category | % { $option.Add("--category $_") }
}

$epiDirectory = "${env:NNTEST_BUILD_SYSTEM_ROOT}\Outputs\ExecutableProductInfo"

if ( Test-Path $epiDirectory -PathType Container )
{
    $option.Add("--executable-product-info `"$epiDirectory`"")
}

if ( -not [string]::IsNullOrEmpty($TargetNamePattern) )
{
    $option.Add("--target-name `"$TargetNamePattern`"")
}

if ( -not [string]::IsNullOrEmpty($TargetInterfacePattern) )
{
    $option.Add("--target-interface `"$TargetInterfacePattern`"")
}

if ( -not [string]::IsNullOrEmpty($TargetAddressPattern) )
{
    $option.Add("--target-address `"$TargetAddressPattern`"")
}

Invoke-Expression "& `"$(Get-TestRunnerPath)`" $($option -join ' ')"

exit $LastExitCode
