﻿<#
    Copyright (C)Nintendo All rights reserved.

    These coded instructions, statements, and computer programs contain proprietary
    information of Nintendo and/or its licensed developers and are protected by
    national and international copyright laws. They may not be disclosed to third
    parties or copied or duplicated in any form, in whole or in part, without the
    prior written consent of Nintendo.

    The content herein is highly confidential and should be handled accordingly.
#>

<#
    .SYNOPSIS
        Confluence の特定のスペースをエクスポートします。

    .DESCRIPTION
        指定した Confluence のスペースをエクスポートします。
        エクスポートしたファイルは ${ArtifactsRoot} へ置かれます。
        エクスポートしたファイルは ${DocumentsRoot}\${SpaceKey} へ展開されます。
#>

Param(
  # 認証情報が書かれた設定ファイルを指定します
  # 以下、設定ファイルのサンプルです
  # <?xml version="1.0"?>
  # <Configuration>
  #   <User>ユーザ名</User>
  #   <Password>パスワード</Password>
  # </Configuration>
  [string]$AuthenticationFile,

  # エクスポートするスペースキーを指定します
  [string]$SpaceKey,

  # エクスポートタイプを指定します
  [string]$ExportType = "TYPE_XML",

  # エクスポート時のタイムアウト値(秒単位)を指定します。
  [int]$TimeoutSec = 600,

  # エクスポート対象の Confluence の URL を指定します。
  [string]$ConfluenceUrl = "http://spdlybra.nintendo.co.jp/confluence",

  # ファイル名を指定します。
  [string]$FileBaseName = "Confluence-space"
)

$ScriptPath = $MyInvocation.MyCommand.Path
$ScriptName = [System.IO.Path]::GetFileNameWithoutExtension($ScriptPath)
$ScriptDirectoryPath = [System.IO.Path]::GetDirectoryName($ScriptPath)

Import-Module "${ScriptDirectoryPath}\Modules\Error"
Import-Module "${ScriptDirectoryPath}\Modules\Path"
Import-Module "${ScriptDirectoryPath}\Modules\Rpc"


# HTTP リクエスト用の Authroization ヘッダを返します
function Get-HTTPAuthorizationHeader()
{
    return "Basic " + [System.Convert]::ToBase64String(
        [System.Text.Encoding]::UTF8.GetBytes($User + ":" + $Password))
}

# Confluence リクエスト用の Url を取得
function Get-ConfluenceUrl($Path)
{
    return $ConfluenceUrl.TrimEnd('/') + '/' + $Path.TrimStart('/')
}

# Confluence に HTTP リクエストを送信して、結果のレスポンスオブジェクトを返却
function Invoke-ConfluenceRequest($Url, $Method)
{
    $request = [System.Net.WebRequest]::Create($Url)
    $request.Method = $Method
    $request.Timeout = $TimeoutSec * 1000 # 秒をミリ秒に変換
    $request.Headers["Authorization"] = Get-HTTPAuthorizationHeader
    $request.Headers["X-Atlassian-Token"] = "nocheck"
    return $request.GetResponse()
}

# Confuence の HTTP レスポンスの結果をテキストとして返却
function Get-ConfluenceResponseText($Url, $Method)
{
    $response = Invoke-ConfluenceRequest -Url $Url -Method $Method
    return (New-Object System.IO.StreamReader $response.GetResponseStream()).ReadToEnd()
}

# Confluence へのリクエスト結果を指定したパスにダウンロード
function Invoke-ConfluenceDownloadFile($Path, $SaveAsPath)
{
    $u = New-Object System.UriBuilder $ConfluenceUrl
    $u.Path = ""
    $u.Fragment = ""
    $u.Query = ""
    $downloadUrl = $u.Uri.ToString().TrimEnd("/") + "/" + $Path.TrimStart("/")
    $auth = Get-HTTPAuthorizationHeader
    
    $webClient = New-Object System.Net.WebClient
    $webClient.Headers.Add("Authorization", $auth)
    $webClient.DownloadFile($downloadUrl, $SaveAsPath)
}

trap [Exception]
{
    Write-Output $_
    exit 1
}

if ([string]::IsNullOrEmpty($AuthenticationFile))
{
    throw "AuthenticationFile が指定されていません"
}

if ([string]::IsNullOrEmpty($SpaceKey))
{
    throw "SpaceKey が指定されていません"
}

if ([string]::IsNullOrEmpty($ExportType))
{
    throw "ExportType が指定されていません"
}

if ("TYPE_XML" -ne $ExportType)
{
    throw "ExportType には TYPE_XML のみ指定可能です"
}

# JSON パースライブラリの初期化
[void][System.Reflection.Assembly]::LoadWithPartialName("System.Web.Extensions")
$JsonSerializer = New-Object System.Web.Script.Serialization.JavaScriptSerializer

# 設定ファイルの読み込み
[Xml]$Config = Get-Content $AuthenticationFile
$User = $Config.Configuration.User
$Password = $Config.Configuration.Password

# エクスポートしたファイルの保存先を準備
$DocumentsRoot = Get-NintendoSdkDocumentsRootPath
$documentsOutputs = Join-Path $DocumentsRoot Outputs
$filePath = Join-Path $documentsOutputs "$FileBaseName-$SpaceKey.zip"
# ダウンロード先のフォルダが無い場合は作成
if (!(Test-Path $documentsOutputs)) {
    mkdir $documentsOutputs > $null
}

# ページのエクスポートジョブを実行
$exportUrl = Get-ConfluenceUrl ("/rest/space-xml/1.0/export/{0}" -F $SpaceKey)
$exportStatus = $JsonSerializer.DeserializeObject((Get-ConfluenceResponseText $exportUrl -Method "POST"))

if ($exportStatus["downloadPath"] -ne $null)
{
    # レスポンスの返却とともにエクスポートが終わっている場合、即座にダウンロードを開始
    Invoke-ConfluenceDownloadFile -Path $exportStatus["downloadPath"] -SaveAsPath $filePath
}
elseif ($exportStatus["taskId"] -ne $null)
{
    # レスポンスの返却後、エクスポートジョブの完了をポーリングして待機
    $taskStatusUrl = Get-ConfluenceUrl ("/longrunningtaskxml.action?taskId={0}" -F $exportStatus["taskId"])
    while ($true) {
        $xml = [xml](Get-ConfluenceResponseText -Url $taskStatusUrl -Method "GET")
        $downloadPath = $null
        # currentStatus 要素からダウンロード用のリンクを抽出
        if ($xml.task.currentStatus -match 'href="([^"]*?)"')
        {
            $downloadPath = $matches[1]
        }
        $r = New-Object -TypeName PSObject -Property @{
            Complete = ([int]$xml.task.percentComplete -eq 100);
            Successful = $xml.task.isSuccessful -eq "true";
            DownloadPath = $downloadPath
        }
        if ($r.Complete)
        {
            # percentComplete が 100% になっても、Successful が $true になるか、ステータスにエラーメッセージが入るまではリクエストをリトライする
            if ($r.Successful -and ($r.DownloadPath -ne $null))
            {
                Invoke-ConfluenceDownloadFile -Path $r.DownloadPath -SaveAsPath $filePath
                break
            }
            elseif ($xml.task.currentStatus -match 'error in the export')
            {
                throw ("スペースのエクスポートに失敗しました: {0}" -F $xml.task.currentStatus)
            }
        }
        Start-Sleep -Seconds 5
    }
}

exit 0
