﻿<#
    Copyright (C)Nintendo All rights reserved.

    These coded instructions, statements, and computer programs contain proprietary
    information of Nintendo and/or its licensed developers and are protected by
    national and international copyright laws. They may not be disclosed to third
    parties or copied or duplicated in any form, in whole or in part, without the
    prior written consent of Nintendo.

    The content herein is highly confidential and should be handled accordingly.
#>

<#
    .SYNOPSIS
        公開ドキュメントを作成します。

    .DESCRIPTION
        作成された公開ドキュメントは ${DocumentsRoot} 以下へ配置されます。
#>

[CmdletBinding()]
Param(
    # FairlyWrites のプロジェクトファイルのテンプレートを指定します
    [Parameter(Mandatory=$true)]
    [string]$ProjectFilePath,

    # Confuence から XML 形式でエクスポートしたデータを展開したディレクトリを指定します
    [Parameter(Mandatory=$true)]
    [string]$SourceDirectoryPath,

    # API リファレンスへのパスを指定します。
    [Parameter()]
    [string]$ApiPath = "Api/Html",

    # TagInfo.xml へのパスを指定します。(複数のxml が "," で区切って指定される)
    [Parameter()]
    [string]$TagInfoXml,

    # Confluence の出力フォルダ名を指定します。
    [Parameter()]
    [string]$ConfluenceOutputName = "Package",

    # TagFile.xml へのパスを指定します。
    [Parameter()]
    [string]$DoxyTag,

    [Parameter()]
    [switch]$ConfluenceTranslated,

    # 言語を指定します。ja or en
    [Parameter()]
    [string]$Lang = "ja",

    [Parameter()]
    [string]$ReplaceDependApiReferencePath = "Api/HtmlNX"
)

$ErrorActionPreference = "Stop"

$ScriptPath = $MyInvocation.MyCommand.Path
$ScriptDirectoryPath = [System.IO.Path]::GetDirectoryName($ScriptPath)

Import-Module "${ScriptDirectoryPath}\Modules\Error"
Import-Module "${ScriptDirectoryPath}\Modules\Path"

# GraphViz がインストールされていなければ警告を出力する。
if ( $Env:GRAPHVIZ_DOT -eq $null )
{
    Write-Host "warning: GRAPHVIZ_DOT is not found."
}
elseif (-not (Test-Path $Env:GRAPHVIZ_DOT) )
{
    Write-Host "warning: GraphViz is not installed, so figures by confluence macro (Flowchart etc.) may collapse."
}

# 出力先ディレクトリ
$sigloRoot = (Resolve-Path $ScriptDirectoryPath/../..).Path

$DocumentsRoot = Get-NintendoSdkDocumentsRootPath
$documentsOutputs = Join-Path $DocumentsRoot Outputs
$documentsIntermediates = Join-Path $documentsRoot Intermediates

Import-Module $sigloRoot/Integrate/Outputs/AnyCPU/Tools/AddApiLinkSpan/AddApiLinkSpan/Release/AddApiLinkSpan.psd1

# $ApiPath が相対パスのときに絶対パスにする(絶対パスのときは変化なし)。
$apiFullPath = [IO.Path]::GetFullPath([IO.Path]::Combine($documentsOutputs, $ApiPath)).TrimEnd(@("\"))

$replaceApiReferencePath = ([uri](
    Join-Path $documentsOutputs $ConfluenceOutputName |
    Join-Path -ChildPath "../some.html")).MakeRelative([uri]$apiFullPath)

# FairlyLinguist へ渡す入力フォルダ引数の格納用配列
$fairlyLinguistParamAry = @()

# 入力フォルダ名を含んだ文字列を分割
$InputArray = $SourceDirectoryPath -Split ","

# 各入力データのタイムスタンプ格納用配列
$DateDic = @{}

if (Test-Path $documentsOutputs/$ConfluenceOutputName/contents/Pages) {
    Remove-Item $documentsOutputs/$ConfluenceOutputName/contents/Pages -Recurse -Force
}

[string]$doxyTagParam = $null
if ($DoxyTag) {
    # $DoxyTag が相対パスのときに絶対パスにする(絶対パスのときは変化なし)。
    $doxyTagParam = [IO.Path]::Combine($documentsOutputs, $DoxyTag)
}

$confluenceIdx = 0
foreach($SourcePath in $InputArray) {
    if (Test-Path $SourcePath/exportDescriptor.properties) {
        $exportDateStr =
            (Get-Content $SourcePath/exportDescriptor.properties |
             Select-Object -First 1).TrimStart("#")
        $exportDateFormat = 'ddd MMM dd HH:mm:ss JST yyyy'

        $exportDate =
            [DateTime]::SpecifyKind(
                [DateTime]::ParseExact(
                    $exportDateStr,
                    $exportDateFormat,
                    [Globalization.CultureInfo]::InvariantCulture).Subtract((New-Object TimeSpan 9,0,0)),
                [DateTimeKind]::Utc)

        $exportDateOffsStr = (New-Object DateTimeOffset ($exportDate.ToLocalTime())).ToString()
        $DateDic.Add("Export date $confluenceIdx", $exportDateOffsStr)
    }

    $targetSpace = Join-Path $documentsIntermediates\$ConfluenceOutputName\Confluence (Split-Path $SourcePath -Leaf)

    mkdir $targetSpace -Force > $null

    # attachments があったら削除しておきます。
    if (Test-Path $targetSpace\attachments) {
        Remove-Item $targetSpace\attachments -Recurse -Force
    }

    # attachments をジャンクションとして作成
    New-Item -ItemType Junction -Path $targetSpace\attachments -Value $SourcePath\attachments > $null

    # リンクを置換
    Add-ApiLinkSpan `
        -InputFile $SourcePath/entities.xml `
        -OutputFile $targetSpace/entities.xml `
        -ReplaceApiReferencePath "/$replaceApiReferencePath/" `
        -ReplaceDependApiReferencePath "/$ReplaceDependApiReferencePath/" `
        -DoxyTag $doxyTagParam

    $fairlyLinguistParamAry += "-src", $targetSpace
    $confluenceIdx++
}

if (Test-Path $documentsOutputs/$ConfluenceOutputName/contents/ChangeLog.html) {
    $gatherDate = (Get-Item $documentsOutputs/$ConfluenceOutputName/contents/ChangeLog.html).LastWriteTimeUtc

    $gatherDateOffsStr = (New-Object DateTimeOffset ($gatherDate.ToLocalTime())).ToString()
    $DateDic.Add("ChangeLog gather date", $gatherDateOffsStr)
}

$DateDic.Add("API Reference commit", (& (Get-GitPath) log -1 '--pretty=format:%H'))

$exportDateHtmlName = 'buildInfo.html'
$exportHtmlPath = "$documentsOutputs/$ConfluenceOutputName/contents/$exportDateHtmlName"

mkdir (Split-Path $exportHtmlPath -Parent) -Force > $null

New-Object PSObject -Property $DateDic |
ConvertTo-Html -As LIST |
Set-Content $exportHtmlPath

# 言語依存の単語データの読み込み
$words =
    [xml](Get-Content `
        -Path (Join-Path (Split-Path $ProjectFilePath -Parent) common_${Lang}.xml) `
        -Encoding UTF8)

# バージョンの設定
mkdir $documentsIntermediates/FairlyLinguist -Force > $null
$outProjectFilePath = [IO.Path]::ChangeExtension("$documentsIntermediates/FairlyLinguist/" + (Split-Path $ProjectFilePath -Leaf), ".xml")
$ProjectFileContents = [xml](Get-Content $ProjectFilePath -Encoding UTF8)
$ProjectFileContents.ProjectList.ProjectInfo.TargetLanguage = $Lang
$ProjectFileContents.ProjectList.ProjectInfo.Title =
    $ProjectFileContents.ProjectList.ProjectInfo.Title -replace '\${Documents}', $words.Words.Documents
$ProjectFileContents.ProjectList.ProjectInfo.Author =
    $ProjectFileContents.ProjectList.ProjectInfo.Author -replace '\${Author}', $words.Words.Author
$ProjectFileContents.ProjectList.ProjectInfo.ExternalJsPath =
    $ProjectFileContents.ProjectList.ProjectInfo.ExternalJsPath -replace '\${Lang}', $Lang
$ProjectFileContents.ProjectList.ProjectInfo.HtmlTitleTemplate =
    $ProjectFileContents.ProjectList.ProjectInfo.HtmlTitleTemplate -replace '\${Lang}', $Lang
$ProjectFileContents.ProjectList.ProjectInfo.Version = (Get-Date).ToString("yyyy.MM.dd.HHmm")
$ProjectFileContents.ProjectList.ProjectInfo.CreationYear = (Get-Date).Year.ToString()
$ProjectFileContents.Save($outProjectFilePath)

if ( $ConfluenceTranslated )
{
    # 出力対象言語の翻訳が存在しないページを出力しない
    $fairlyLinguistParamAry += "-omitNotTranslated"
}
else {
    # 出力対象言語への翻訳が未完了の場合は、原文を出力する。
    $fairlyLinguistParamAry += "-limitToTranslated"
}

# FairlyLinguist で変換
& (Get-FairlyLinguistCuiPath) $outProjectFilePath @fairlyLinguistParamAry -dst $DocumentsRoot -log $documentsIntermediates/FairlyLinguist

switch ($LastExitCode) {
    0 {} # 正常に終了
    1 {} # ドキュメントの生成で警告が発生
    2 {} # リンク、添付画像でエラーが発生
    13 {} # 絵文字の変換でエラーが発生

    # 上記以外の値は終了エラーとする
    default { throw "FairlyLinguistCui failed. - $LastExitCode" }
}

if ($TagInfoXml) {
    # $TagInfoXml の各メンバが相対パスのときに絶対パスにする(絶対パスのときは変化なし)。
    $TagInfoXmls = $TagInfoXml.Split(",")
    $TagInfoArray=@()
    foreach( $Info in $TagInfoXmls )
    {
        $Info = [IO.Path]::Combine($documentsOutputs, $Info)
        $TagInfoArray += $Info
    }

    # タグファイル情報オプション
    $TagInfoOption = ""
    if ( $TagInfoArray.Length -gt 0 )
    {
        $TagInfoOption = [string]($TagInfoArray -join ",")
    }

    $htmlDir = "$documentsOutputs/$ConfluenceOutputName/contents/Pages"
    $relApi = "../../../"

    # 変換済みhtmlに対し、タグファイルと突き合わせて span 書き変えを追加
    $exePath = "$sigloRoot/Integrate/Outputs/AnyCPU/Tools/ApiLinkRewriter/ApiLinkRewriter/Release/ApiLinkRewriter.exe"
    & $exePath --html-dir $htmlDir --pre-api-url $relApi --tag-info-xml $TagInfoOption
    if ($LastExitCode) {
        throw "ApiLinkRewriter failed. - $LastExitCode"
    }
}
