﻿/*
 *  Copyright 2005-2014 Acer Cloud Technology, Inc.
 *  All Rights Reserved.
 *
 *  This software contains confidential information and
 *  trade secrets of Acer Cloud Technology, Inc.
 *  Use, disclosure or reproduction is prohibited without
 *  the prior express written permission of Acer Cloud
 *  Technology, Inc.
 */

/*
 *               Copyright (C) 2010, BroadOn Communications Corp.
 *
 *  These coded instructions, statements, and computer programs contain
 *  unpublished  proprietary information of BroadOn Communications Corp.,
 *  and  are protected by Federal copyright law. They may not be disclosed
 *  to  third  parties or copied or duplicated in any form, in whole or in
 *  part, without the prior written consent of BroadOn Communications Corp.
 *
 */


/*
 * Implementation of class for publishing V1 ETickets
 *
 * This interface does not support concurrent operations, since it
 * uses global state to build the new ticket.
 */
#include <nn/nn_Macro.h>
#include <nn/escore/estypes.h>

USING_ES_NAMESPACE

#include "pub_utils.h"
#include <nn/publish/publish.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_Macro.h>

#ifdef _GVM_DEFINE_INTTYPES_H_
#include <inttypes.h>
#endif

PUBLISH_NAMESPACE_START


/*
 * State data required for each instance of a ticket being generated
 */
class PublishTicketPrivData
{
    NN_DISALLOW_COPY(PublishTicketPrivData);

    public:
        PublishTicketPrivData(u32 ticketLen) :
            __tp(NULL),
            __devCertLoaded(false),
            __rightsIdSet(false),
            __numContItemRights(0),
            __contItemRights(NULL),
            __compressedItemRights(false),
            __setItemRightsCalled(false)
        {
            //
            // Clear device cert and Content Item Rights
            //
            memset((void *)&__devCert, 0, sizeof(__devCert));

            __ticketLen = ticketLen;
            __tp = (ESV2Ticket *) malloc(__ticketLen);
            NN_SDK_ASSERT(__tp != NULL, "Failed to allocate memory (size %d)\n", __ticketLen);
        }

        ~PublishTicketPrivData()
        {
            if (__tp != NULL) {
                free(__tp);
                __tp = NULL;    // pro forma
            }
            if (__contItemRights != NULL) {
                free(__contItemRights);
                __contItemRights = NULL;
            }
        }

        ESV2Ticket *__tp;
        u32 __ticketLen;
        IOSCEccEccCert __devCert;
        bool __devCertLoaded;
        bool __rightsIdSet;
        u32 __numContItemRights;
        ESV1ContentRecord *__contItemRights;
        bool __compressedItemRights;
        bool __setItemRightsCalled;
};

#define PRIVDATA    ((PublishTicketPrivData *)privData)


//
// Initialization shared between the constructor and the Reset() method
//
ESError
PublishTicket::Initialize(u8 *tikTemplate, u32 tikTemplateLen)
{
    ESError rv = ES_ERR_OK;

    NN_SDK_ASSERT(privData == NULL);

    if (tikTemplate != NULL && tikTemplateLen < sizeof(ESV2Ticket)) {
        pubLog(PUB_DEBUG_ERROR, "Invalid template size %d < %" PRIuSizeT "\n", tikTemplateLen, sizeof(ESV2Ticket));
        rv = ES_ERR_INVALID;
        goto end;
    }

    if (tikTemplateLen == 0) {
        tikTemplateLen = sizeof(ESV2Ticket);
    }
    privData = new PublishTicketPrivData(tikTemplateLen);

    if (tikTemplate == NULL) {
        memset(PRIVDATA->__tp, 0, PRIVDATA->__ticketLen);
    } else {
        memcpy(PRIVDATA->__tp, tikTemplate, PRIVDATA->__ticketLen);
    }

    PRIVDATA->__tp->formatVersion = 2;     // Only generate V2 tickets

    //
    // If there is no template or the template only includes V2 template, then initialize
    // the section header
    //
    if (tikTemplate == NULL || tikTemplateLen == sizeof(ESV2Ticket)) {
        PRIVDATA->__tp->sectTotalSize = 0;
        PRIVDATA->__tp->sectHdrOfst = sizeof(ESV2Ticket);
        PRIVDATA->__tp->nSectHdrs = 0;
        PRIVDATA->__tp->sectHdrEntrySize = 0;
    } else {
        //
        // If the input template has pre-existing ItemRights, then SignTicket
        // does not need to call SetItemRights.
        //
        PRIVDATA->__setItemRightsCalled = true;
    }

    if (tikTemplate != NULL && PRIVDATA->__tp->rightsId != 0ULL) {
        PRIVDATA->__rightsIdSet = true;
    }

end:
    return rv;
}


//
// Constructor takes an optional ticket template
//
// For simplicity, the template only supports the base V0+V1 header.  It is
// too complicated to deal with updating pre-existing V1 sections.
//
PublishTicket::PublishTicket(u8 *tikTemplate, u32 tikTemplateLen) :
    privData(NULL)
{
    ESError rv = ES_ERR_OK;

    IOSC_Initialize();

    rv = PublishTicket::Initialize(tikTemplate, tikTemplateLen);
    if (rv != ES_ERR_OK) goto err;

    return;

err:
    if (privData != NULL) {
        delete (PublishTicketPrivData *) privData;
        privData = NULL;
    }
    NN_SDK_ASSERT(rv != ES_ERR_OK);
}


//
// PublishTicket destructor invokes the private data destructor
//
PublishTicket::~PublishTicket()
{
    if (privData != NULL) {
        delete (PublishTicketPrivData *) privData;
        privData = NULL;
    }
}


//
// Reset method reuses an existing PublishTicket object for a new ticket
//
// For simplicity, the template only supports the base V0+V1 header.  It is
// too complicated to deal with updating pre-existing V1 sections.
//
ESError
PublishTicket::Reset(u8 *tikTemplate, u32 tikTemplateLen)
{
    ESError rv = ES_ERR_OK;

    //
    // If this object has already been used, free the private data so that
    // Initialize() can allocate a fresh one.
    //
    if (privData != NULL) {
        delete (PublishTicketPrivData *) privData;
        privData = NULL;
    }

    rv = PublishTicket::Initialize(tikTemplate, tikTemplateLen);

    return rv;
}


ESError
PublishTicket::SetTicketId(ESTicketId inTicketId)
{
    ESError rv = ES_ERR_OK;

    PRIVDATA->__tp->ticketId = inTicketId;

    return rv;
}


ESError
PublishTicket::GetTicketId(ESTicketId *outTicketId)
{
    ESError rv = ES_ERR_OK;

    *outTicketId = PRIVDATA->__tp->ticketId;

    return rv;
}


ESError
PublishTicket::SetTicketVersion(ESTicketVersion inTicketVersion)
{
    ESError rv = ES_ERR_OK;

    PRIVDATA->__tp->ticketVersion = inTicketVersion;

    return rv;
}


ESError
PublishTicket::GetTicketVersion(ESTicketVersion *outTicketVersion)
{
    ESError rv = ES_ERR_OK;

    *outTicketVersion = PRIVDATA->__tp->ticketVersion;

    return rv;
}


ESError
PublishTicket::SetRightsId(ESRightsId inRightsId)
{
    ESError rv = ES_ERR_OK;

    PRIVDATA->__rightsIdSet = true;
    memcpy( PRIVDATA->__tp->rightsId, inRightsId, sizeof(ESRightsId));

    return rv;
}


ESError
PublishTicket::GetRightsId(ESRightsId *outRightsId)
{
    ESError rv = ES_ERR_OK;

    memcpy(outRightsId, PRIVDATA->__tp->rightsId,  sizeof(ESRightsId));

    return rv;
}


ESError
PublishTicket::SetDeviceCert(u8 *inDeviceCert, u32 inDeviceCertLen)
{
    ESError rv = ES_ERR_OK;
    u8 *name;
    ESDeviceId devId;

    if (inDeviceCert == NULL || inDeviceCertLen != sizeof(IOSCEccEccCert)) {
        pubLog(PUB_DEBUG_ERROR, "Invalid parameters (certLen %d)\n", inDeviceCertLen);
        rv = ES_ERR_INVALID;
        goto end;
    }

    memcpy((u8 *)&PRIVDATA->__devCert, inDeviceCert, inDeviceCertLen);
    PRIVDATA->__devCertLoaded = true;

    /*
     * Parse the common name of the device certificate to determine the device ID
     */
    name = PRIVDATA->__devCert.head.name.deviceId;
    if (memcmp(name, "CT", 2) == 0 ||
        memcmp(name, "TW", 2) == 0 ||
        memcmp(name, "NG", 2) == 0 ||
        memcmp(name, "VP", 2) == 0 ||
        memcmp(name, "PC", 2) == 0) {

        sscanf((const char *)(name+2), "%08X", (int*)&devId);
    } else {
        pubLog(PUB_DEBUG_ERROR, "Device Certificate type not valid (device name %s)\n", name);
        rv = ES_ERR_INVALID;
        goto end;
    }

    PRIVDATA->__tp->deviceId = devId;
end:
    return rv;
}


/*
 * This call is optional, since the common name string in the device cert
 * specifies the device ID
 */
ESError
PublishTicket::SetDeviceId(ESDeviceId inDeviceId)
{
    ESError rv = ES_ERR_OK;

    PRIVDATA->__tp->deviceId = inDeviceId;

    return rv;
}


ESError
PublishTicket::GetDeviceId(ESDeviceId *outDeviceId)
{
    ESError rv = ES_ERR_OK;

    *outDeviceId = PRIVDATA->__tp->deviceId;

    return rv;
}


ESError
PublishTicket::SetLicenseType(ESLicenseType licenseType)
{
    ESError rv = ES_ERR_OK;

    PRIVDATA->__tp->licenseType = licenseType;

    return rv;
}


ESError
PublishTicket::SetCommonKeyId(u8 commonKeyId)
{
    ESError rv = ES_ERR_OK;

    PRIVDATA->__tp->commonKeyId = commonKeyId;

    return rv;
}


ESError
PublishTicket::GetCommonKeyId(u8 *commonKeyId)
{
    ESError rv = ES_ERR_OK;

    *commonKeyId = PRIVDATA->__tp->commonKeyId;

    return rv;
}


//
// Only turns on the flag, but an overloaded version is provided to
// turn it on or off.  Ditto for the other propertyMask bits.
//
ESError
PublishTicket::SetPreInstallationFlag()
{
    ESError rv = ES_ERR_OK;

    PRIVDATA->__tp->propertyMask |= ES_TICKET_PRE_INSTALL_FLAG;

    return rv;
}


//
// Give the publishing code a means to turn off this flag as well.
//
ESError
PublishTicket::SetPreInstallationFlag(u8 preInstallationFlag)
{
    ESError rv = ES_ERR_OK;

    if (preInstallationFlag) {
        PRIVDATA->__tp->propertyMask |= ES_TICKET_PRE_INSTALL_FLAG;
    } else {
        PRIVDATA->__tp->propertyMask &= ~(ES_TICKET_PRE_INSTALL_FLAG);
    }

    return rv;
}


ESError
PublishTicket::GetPreInstallationFlag(u8 *preInstallationFlag)
{
    ESError rv = ES_ERR_OK;

    *preInstallationFlag = ((PRIVDATA->__tp->propertyMask & ES_TICKET_PRE_INSTALL_FLAG) != 0);

    return rv;
}


ESError
PublishTicket::SetSharedTitleFlag()
{
    ESError rv = ES_ERR_OK;

    PRIVDATA->__tp->propertyMask |= ES_TICKET_SHARED_TITLE_FLAG;

    return rv;
}


ESError
PublishTicket::SetSharedTitleFlag(u8 sharedTitleFlag)
{
    ESError rv = ES_ERR_OK;

    if (sharedTitleFlag) {
        PRIVDATA->__tp->propertyMask |= ES_TICKET_SHARED_TITLE_FLAG;
    } else {
        PRIVDATA->__tp->propertyMask &= ~(ES_TICKET_SHARED_TITLE_FLAG);
    }

    return rv;
}


ESError
PublishTicket::GetSharedTitleFlag(u8 *sharedTitleFlag)
{
    ESError rv = ES_ERR_OK;

    *sharedTitleFlag = ((PRIVDATA->__tp->propertyMask & ES_TICKET_SHARED_TITLE_FLAG) != 0);

    return rv;
}


ESError
PublishTicket::SetAllContentsFlag()
{
    ESError rv = ES_ERR_OK;

    PRIVDATA->__tp->propertyMask |= ES_TICKET_ALL_CONTENTS_FLAG;

    return rv;
}


ESError
PublishTicket::SetAllContentsFlag(u8 allContentsFlag)
{
    ESError rv = ES_ERR_OK;

    if (allContentsFlag) {
        PRIVDATA->__tp->propertyMask |= ES_TICKET_ALL_CONTENTS_FLAG;
    } else {
        PRIVDATA->__tp->propertyMask &= ~(ES_TICKET_ALL_CONTENTS_FLAG);
    }

    return rv;
}


ESError
PublishTicket::GetAllContentsFlag(u8 *allContentsFlag)
{
    ESError rv = ES_ERR_OK;

    *allContentsFlag = ((PRIVDATA->__tp->propertyMask & ES_TICKET_ALL_CONTENTS_FLAG) != 0);

    return rv;
}


#define VALID_LIMIT_CODE(lim)   (ES_LC_NO_LIMITS <= (lim) && (lim) <= ES_LC_ELAPSED_TIME)
static bool
__maskNotEmpty(u8 *contAccessMask)
{
    u32 i;

    for (i = 0; i < ES_CONTENT_ITEM_ACCESS_MASK_LEN; i++) {
        if (contAccessMask[i] != 0) {
            return true;
        }
    }
    return false;
}


/*
 * SetContentMask will cause empty V1 ContentItemRights records
 * to be allocated in the case that the bit mask is sufficiently
 * sparse.  This routine removes the empty records, but can only
 * be called once per PublishTicket instance to avoid complexity.
 */
static void
__compressContItemRights(PublishTicketPrivData *privData)
{
    u32 recordIndex;

    NN_SDK_ASSERT(PRIVDATA->__contItemRights != NULL);

    if (PRIVDATA->__compressedItemRights) {
        pubLog(PUB_DEBUG_ERROR, "Duplicate call to compressContentItemRights\n");
        return;
    }

    for (recordIndex = 0; recordIndex < PRIVDATA->__numContItemRights; ) {
        if (__maskNotEmpty(PRIVDATA->__contItemRights[recordIndex].accessMask)) {
            recordIndex++;
            continue;
        }

        /* Found an empty record, so slide everything down by one */
        for (u32 loopIndex = recordIndex; loopIndex + 1 < PRIVDATA->__numContItemRights; loopIndex++) {
            PRIVDATA->__contItemRights[loopIndex] = PRIVDATA->__contItemRights[loopIndex + 1];  // structure assignment
        }

        /*
         * No need to realloc, since the only operation that's valid is to free the pointer after this
         */
        PRIVDATA->__numContItemRights -= 1;
    }

    PRIVDATA->__compressedItemRights = true;
    return;
}


/*
 * This method is the simpler way to set the V1 ContentAccessMask records.
 * The caller can also set up their own ItemRights records and call SetItemRights.
 */
ESError
PublishTicket::SetContentMask(u32 startIndex, u32 numContents)
{
    ESError rv = ES_ERR_OK;
    u32 bitNum;
    u32 maxIndex = startIndex + numContents - 1;
    u32 numRecords;
    u32 recordIndex;

    if (PRIVDATA->__setItemRightsCalled || PRIVDATA->__compressedItemRights) {
        pubLog(PUB_DEBUG_ERROR, "SetContentMask cannot be called after SetItemRights\n");
        rv = ES_ERR_INVALID;
        goto end;
    }

    if (maxIndex > ES_CONTENT_INDEX_MAX) {
        pubLog(PUB_DEBUG_ERROR, "Invalid Content Index %d\n", maxIndex);
        rv = ES_ERR_INVALID;
        goto end;
    }

    if (numContents == 0) {
        goto end;
    }

    numRecords = 1 + (maxIndex / (ES_CONTENT_ITEM_ACCESS_MASK_LEN * 8));

    /*
     * Allocate more ItemRights records if necessary
     */
    if (numRecords > PRIVDATA->__numContItemRights) {
        ESV1ContentRecord *newItemRights;

        if ((newItemRights = (ESV1ContentRecord *) malloc(numRecords * sizeof(ESV1ContentRecord))) == NULL) {
            pubLog(PUB_DEBUG_ERROR, "Malloc failed for %" PRIuSizeT " bytes\n", numRecords * sizeof(ESV1ContentRecord));
            rv = ES_ERR_INVALID;
            goto end;
        }

        /*
         * If there were some already, copy the contents and free the old version
         */
        if (PRIVDATA->__numContItemRights > 0) {
            memcpy((u8 *)newItemRights, (u8 *) PRIVDATA->__contItemRights, PRIVDATA->__numContItemRights * sizeof(ESV1ContentRecord));
            free((u8 *)PRIVDATA->__contItemRights);
        }
        PRIVDATA->__contItemRights = newItemRights;

        /*
         * Initialize the added records.  Note that these are in the internal format, as opposed to the
         * ESItemRights supplied by the caller to SetItemRights.  The only real difference is that the
         * starting offset field is in network order.
         */
        for (recordIndex = PRIVDATA->__numContItemRights; recordIndex < numRecords; recordIndex++) {
            PRIVDATA->__contItemRights[recordIndex].offset = ntohl(recordIndex * ES_CONTENT_ITEM_ACCESS_MASK_LEN * 8);
            memset((u8 *)PRIVDATA->__contItemRights[recordIndex].accessMask, 0, sizeof(ESV1ContentRecordAccessMask));
        }
        PRIVDATA->__numContItemRights = numRecords;
    }

    /*
     * Set the requested bits in the appropriate masks
     */
    for (bitNum = startIndex; bitNum < startIndex + numContents; bitNum++) {
        recordIndex = bitNum / (ES_CONTENT_ITEM_ACCESS_MASK_LEN * 8);
        __bitSet(PRIVDATA->__contItemRights[recordIndex].accessMask, bitNum % (ES_CONTENT_ITEM_ACCESS_MASK_LEN * 8));
    }

end:
    return rv;
}


/*
 * The TitleKey passed in already encrypted with the common key, using
 * the titleId as the IV
 */
ESError
PublishTicket::SetTitleKey(u8 *titleKey)
{
    ESError rv = ES_ERR_OK;

    memcpy((u8 *)&PRIVDATA->__tp->titleKey, titleKey, sizeof(PRIVDATA->__tp->titleKey.aesKey));

    return rv;
}


/*
 * Just return the titleKey field in whatever state it is (personalized or not)
 */
ESError
PublishTicket::GetTitleKey(u8 *outTitleKey)
{
    ESError rv = ES_ERR_OK;

    memcpy(outTitleKey, (u8 *)&PRIVDATA->__tp->titleKey, sizeof(PRIVDATA->__tp->titleKey.aesKey));

    return rv;
}


/*
 * The TitleKey passed in cleartext, so it needs to be encrypted with the common key
 * using the titleId as the IV
 */
ESError
PublishTicket::SetTitleKeyClear(u8 *titleKeyClr, u8 *commonKey)
{
    ESError rv = ES_ERR_OK;
    IOSCAesIv aesIv;
    IOSCAesKey titleKeyEnc;
    IOSCSecretKeyHandle hCommonKey;

    if (!PRIVDATA->__rightsIdSet) {
        pubLog(PUB_DEBUG_ERROR, "SetTitleId must be called before SetTitleKeyClear\n");
        rv = ES_ERR_INVALID;
        goto end;
    }

    rv = IOSC_CreateObject(&hCommonKey, IOSC_SECRETKEY_TYPE, IOSC_ENC_SUBTYPE);
    if (rv != IOSC_ERROR_OK) {
        pubLog(PUB_DEBUG_ERROR, "Failed to create secret key object, rv=%d\n", rv);
        rv = ES_ERR_CRYPTO;
        goto end;
    }

    rv = IOSC_ImportSecretKey(hCommonKey, 0, 0, IOSC_NOSIGN_NOENC,
            NULL, NULL, commonKey);
    if (rv != IOSC_ERROR_OK) {
        pubLog(PUB_DEBUG_ERROR, "Failed to import common key, rv=%d\n", rv);
        rv = ES_ERR_CRYPTO;
        goto end;
    }

    /*
     * Encrypt the titleKey with the common key
     *
     * Initialization Vector is the Rights ID (network byte order)
     */
    memset(aesIv, 0, sizeof(aesIv));
    memcpy(aesIv, &PRIVDATA->__tp->rightsId, sizeof(PRIVDATA->__tp->rightsId));
    rv = IOSC_Encrypt(hCommonKey, aesIv, titleKeyClr, sizeof(IOSCAesKey), titleKeyEnc);
    if (rv != IOSC_ERROR_OK) {
        pubLog(PUB_DEBUG_ERROR, "IOSC_Encrypt failed (rv %d)\n", rv);
        goto end;
    }
    (void) IOSC_DeleteObject(hCommonKey);

    memcpy((u8 *)&PRIVDATA->__tp->titleKey.aesKey, titleKeyEnc, sizeof(PRIVDATA->__tp->titleKey.aesKey));

end:
    return rv;
}


ESError
PublishTicket::SetReserved(u8 *reserved, u32 reservedLen)
{
    ESError rv = ES_ERR_OK;

    if (reservedLen > sizeof(PRIVDATA->__tp->reserved)) {
        pubLog(PUB_DEBUG_ERROR, "SetReserved with length %d (max %" PRIuSizeT ")\n",
            reservedLen, sizeof(PRIVDATA->__tp->reserved));
        rv = ES_ERR_INVALID;
        goto end;
    }

    memset(PRIVDATA->__tp->reserved, 0, sizeof(PRIVDATA->__tp->reserved));
    memcpy(PRIVDATA->__tp->reserved, reserved, reservedLen);
end:
    return rv;
}


ESError
PublishTicket::GetReserved(u8 *reserved, u32 *reservedLen)
{
    ESError rv = ES_ERR_OK;

    if (*reservedLen < sizeof(PRIVDATA->__tp->reserved)) {
        pubLog(PUB_DEBUG_ERROR, "GetReserved with length %d (need %" PRIuSizeT ")\n",
            *reservedLen, sizeof(PRIVDATA->__tp->reserved));
        rv = ES_ERR_INVALID;
        goto end;
    }

    memcpy(reserved, PRIVDATA->__tp->reserved, sizeof(PRIVDATA->__tp->reserved));
    *reservedLen = sizeof(PRIVDATA->__tp->reserved);
end:
    return rv;
}


ESError
PublishTicket::SetAccountId(u32 accountId)
{
    ESError rv = ES_ERR_OK;

    PRIVDATA->__tp->accountId = accountId;

    return rv;
}


ESError
PublishTicket::GetAccountId(u32 *outAccountId)
{
    ESError rv = ES_ERR_OK;

    *outAccountId = PRIVDATA->__tp->accountId;

    return rv;
}

ESError
PublishTicket::SetIssuer(const u8 *issuerName)
{
    ESError rv = ES_ERR_OK;

    strncpy((char *)&PRIVDATA->__tp->sig.issuer, (char *)issuerName, sizeof(PRIVDATA->__tp->sig.issuer));

    return rv;
}

ESError
PublishTicket::GetIssuer(u8 *outIssuerName)
{
    ESError rv = ES_ERR_OK;

    strncpy((char *)outIssuerName, (char *)&PRIVDATA->__tp->sig.issuer, sizeof(PRIVDATA->__tp->sig.issuer));

    return rv;
}

ESError
PublishTicket::SetSignData(const u8 *signData)
{
    ESError rv = ES_ERR_OK;

    PRIVDATA->__tp->sig.sigType = IOSC_SIG_RSA2048_H256;

    memcpy((u8 *)&PRIVDATA->__tp->sig.sig, signData, sizeof(CSLOSRsaPublicKey2048));

    return rv;
}

ESError
PublishTicket::GetSignData(u8 *outSignData)
{
    ESError rv = ES_ERR_OK;

    memcpy(outSignData, (u8 *)&PRIVDATA->__tp->sig.sig, sizeof(CSLOSRsaPublicKey2048));

    return rv;
}
ESError
PublishTicket::GetTicketDataForSign(u8 *outTicketBuf, u32 outTicketBufLen)
{
    ESError rv = ES_ERR_OK;

    int headerLen = (int)((u8*)&PRIVDATA->__tp->sig.issuer - (u8*)PRIVDATA->__tp);

    if (outTicketBuf == NULL || outTicketBufLen < PRIVDATA->__ticketLen - headerLen) {
        pubLog(PUB_DEBUG_ERROR, "Invalid parameters\n");
        rv = ES_ERR_INVALID;
        goto end;
    }
    memcpy(outTicketBuf, PRIVDATA->__tp->sig.issuer, PRIVDATA->__ticketLen - headerLen);

end:
    return rv;
}

ESError
PublishTicket::GetTicketDataForSignLen(u32 *outTicketLen)
{
    ESError rv = ES_ERR_OK;

    int headerLen = (int)((u8*)&PRIVDATA->__tp->sig.issuer - (u8*)PRIVDATA->__tp);

    *outTicketLen = PRIVDATA->__ticketLen - headerLen;

    return rv;
}

ESError
PublishTicket::SetItemRights(ESItemRight *itemRights, u32 numItemRights)
{
    ESError rv = ES_ERR_OK;
    u32 i, size;
    u16 nSections = 0;
    u32 numPermRights = 0, numSubRights = 0;
    u32 numCntRights = 0, numCntLmtRights = 0;
    u32 numAccTitleRights = 0, numLimResRights = 0;
    u8 *buf;
    ESV1SectionHeader sectHdr;
    ESV1PermanentRecord permRec;
    ESV1SubscriptionRecord subRec;
    ESV1ContentRecord cntRec;
    ESV1ContentConsumptionRecord cntLmtRec;
    ESV1AccessTitleRecord accTitleRec;
    ESV1LimitedResourceRecord limResRec;
#ifdef __EVL_API__
    u32 numGenRights = 0;
    ESV1GenericRecord genRec;
#endif

    /*
     * This call can only be made once for a given ticket, which also means
     * that ItemRights can not be added to a ticket template with pre-existing
     * V1 section entries.
     */
    if (PRIVDATA->__ticketLen > sizeof(ESV2Ticket)) {
        pubLog(PUB_DEBUG_ERROR, "Ticket already contains ItemRights\n");
        rv = ES_ERR_INVALID;
        goto end;
    }

    /*
     * Get the total number of rights for memory allocation
     */
    for (i = 0; i < numItemRights; i++) {
        if (itemRights[i].type == ES_ITEM_RIGHT_PERMANENT) {
            numPermRights++;
        } else if (itemRights[i].type == ES_ITEM_RIGHT_SUBSCRIPTION) {
            numSubRights++;
        } else if (itemRights[i].type == ES_ITEM_RIGHT_CONTENT) {
            numCntRights++;
        } else if (itemRights[i].type == ES_ITEM_RIGHT_CONTENT_CONSUMPTION) {
            numCntLmtRights++;
        } else if (itemRights[i].type == ES_ITEM_RIGHT_ACCESS_TITLE) {
            numAccTitleRights++;
        } else if (itemRights[i].type == ES_ITEM_RIGHT_LIMITED_RESOURCE) {
            numLimResRights++;
#ifdef __EVL_API__
        } else if (itemRights[i].type == ES_ITEM_RIGHT_GENERIC) {
            numGenRights++;
#endif
        }
    }

    /*
     * Note that it is not required for a ticket to have any ContentItemRights
     */
    if (numCntRights > 0) {
        if (PRIVDATA->__numContItemRights > 0) {
            pubLog(PUB_DEBUG_WARN, "Redundant content masks:  SetContentMask ignored\n");
            PRIVDATA->__numContItemRights = 0;
            free((u8 *)PRIVDATA->__contItemRights);
            PRIVDATA->__contItemRights = NULL;
        }
    } else {
        if (PRIVDATA->__numContItemRights > 0) {
            __compressContItemRights(PRIVDATA);
            numCntRights = PRIVDATA->__numContItemRights;
        }
    }

    size = sizeof(ESV2Ticket);
    if (numPermRights > 0) {
        size += sizeof(ESV1SectionHeader) + (numPermRights * sizeof(ESV1PermanentRecord));
        nSections++;
    }
    if (numSubRights > 0) {
        size += sizeof(ESV1SectionHeader) + (numSubRights * sizeof(ESV1SubscriptionRecord));
        nSections++;
    }
    if (numCntRights > 0) {
        size += sizeof(ESV1SectionHeader) + (numCntRights * sizeof(ESV1ContentRecord));
        nSections++;
    }
    if (numCntLmtRights > 0) {
        size += sizeof(ESV1SectionHeader) + (numCntLmtRights * sizeof(ESV1ContentConsumptionRecord));
        nSections++;
    }
    if (numAccTitleRights > 0) {
        size += sizeof(ESV1SectionHeader) + (numAccTitleRights * sizeof(ESV1AccessTitleRecord));
        nSections++;
    }
    if (numLimResRights > 0) {
        size += sizeof(ESV1SectionHeader) + (numLimResRights * sizeof(ESV1LimitedResourceRecord));
        nSections++;
    }
#ifdef __EVL_API__
    if (numGenRights > 0) {
        size += sizeof(ESV1SectionHeader) + (numGenRights * sizeof(ESV1GenericRecord));
        nSections++;
    }
#endif

    buf = (u8 *) realloc(PRIVDATA->__tp, size);
    if (buf == NULL) {
        pubLog(PUB_DEBUG_ERROR, "Failed to re-allocate memory (size %d)\n", size);
        rv = ES_ERR_FAIL;
        goto end;
    }

    memset(buf + PRIVDATA->__ticketLen, 0, size - PRIVDATA->__ticketLen);
    PRIVDATA->__tp = (ESV2Ticket *) buf;
    PRIVDATA->__ticketLen = size;

    PRIVDATA->__tp->sectTotalSize = size - sizeof(ESV2Ticket);

    /*
     * Add section headers
     */
    PRIVDATA->__tp->nSectHdrs = nSections;
    if (nSections > 0)
    {
        PRIVDATA->__tp->sectHdrEntrySize = sizeof(ESV2SectionHeader);
    }
    buf = (u8 *)&PRIVDATA->__tp[1];

    if (numPermRights > 0) {
        sectHdr.sectOfst = ntohl(ntohl(PRIVDATA->__tp->sectHdrOfst) + (nSections * ntohs(PRIVDATA->__tp->sectHdrEntrySize)));
        sectHdr.nRecords = ntohl(numPermRights);
        sectHdr.recordSize = ntohl(sizeof(ESV1PermanentRecord));
        sectHdr.sectionSize = ntohl(numPermRights * sizeof(ESV1PermanentRecord));
        sectHdr.sectionType = ntohs(ES_ITEM_RIGHT_PERMANENT);
        sectHdr.flags = 0;

        memcpy(buf, &sectHdr, sizeof(ESV1SectionHeader));
        buf += sizeof(ESV1SectionHeader);
    }

    if (numSubRights > 0) {
        sectHdr.sectOfst = ntohl(ntohl(PRIVDATA->__tp->sectHdrOfst) + (nSections * ntohs(PRIVDATA->__tp->sectHdrEntrySize)) + (numPermRights * sizeof(ESV1PermanentRecord)));
        sectHdr.nRecords = ntohl(numSubRights);
        sectHdr.recordSize = ntohl(sizeof(ESV1SubscriptionRecord));
        sectHdr.sectionSize = ntohl(numSubRights * sizeof(ESV1SubscriptionRecord));
        sectHdr.sectionType = ntohs(ES_ITEM_RIGHT_SUBSCRIPTION);
        sectHdr.flags = 0;

        memcpy(buf, &sectHdr, sizeof(ESV1SectionHeader));
        buf += sizeof(ESV1SectionHeader);
    }

    if (numCntRights > 0) {
        sectHdr.sectOfst = ntohl(ntohl(PRIVDATA->__tp->sectHdrOfst) + (nSections * ntohs(PRIVDATA->__tp->sectHdrEntrySize)) + (numPermRights * sizeof(ESV1PermanentRecord)) + (numSubRights * sizeof(ESV1SubscriptionRecord)));
        sectHdr.nRecords = ntohl(numCntRights);
        sectHdr.recordSize = ntohl(sizeof(ESV1ContentRecord));
        sectHdr.sectionSize = ntohl(numCntRights * sizeof(ESV1ContentRecord));
        sectHdr.sectionType = ntohs(ES_ITEM_RIGHT_CONTENT);
        sectHdr.flags = 0;

        memcpy(buf, &sectHdr, sizeof(ESV1SectionHeader));
        buf += sizeof(ESV1SectionHeader);
    }

    if (numCntLmtRights > 0) {
        sectHdr.sectOfst = ntohl(ntohl(PRIVDATA->__tp->sectHdrOfst) + (nSections * ntohs(PRIVDATA->__tp->sectHdrEntrySize)) + (numPermRights * sizeof(ESV1PermanentRecord)) + (numSubRights * sizeof(ESV1SubscriptionRecord)) + (numCntRights * sizeof(ESV1ContentRecord)));
        sectHdr.nRecords = ntohl(numCntLmtRights);
        sectHdr.recordSize = ntohl(sizeof(ESV1ContentConsumptionRecord));
        sectHdr.sectionSize = ntohl(numCntLmtRights * sizeof(ESV1ContentConsumptionRecord));
        sectHdr.sectionType = ntohs(ES_ITEM_RIGHT_CONTENT_CONSUMPTION);
        sectHdr.flags = 0;

        memcpy(buf, &sectHdr, sizeof(ESV1SectionHeader));
        buf += sizeof(ESV1SectionHeader);
    }

    if (numAccTitleRights > 0) {
        sectHdr.sectOfst = ntohl(ntohl(PRIVDATA->__tp->sectHdrOfst) + (nSections * ntohs(PRIVDATA->__tp->sectHdrEntrySize)) + (numPermRights * sizeof(ESV1PermanentRecord)) + (numSubRights * sizeof(ESV1SubscriptionRecord)) + (numCntRights * sizeof(ESV1ContentRecord)) + (numCntLmtRights * sizeof(ESV1ContentConsumptionRecord)));
        sectHdr.nRecords = ntohl(numAccTitleRights);
        sectHdr.recordSize = ntohl(sizeof(ESV1AccessTitleRecord));
        sectHdr.sectionSize = ntohl(numAccTitleRights * sizeof(ESV1AccessTitleRecord));
        sectHdr.sectionType = ntohs(ES_ITEM_RIGHT_ACCESS_TITLE);
        sectHdr.flags = 0;

        memcpy(buf, &sectHdr, sizeof(ESV1SectionHeader));
        buf += sizeof(ESV1SectionHeader);
    }

    if (numLimResRights > 0) {
        sectHdr.sectOfst = ntohl(ntohl(PRIVDATA->__tp->sectHdrOfst) + (nSections * ntohs(PRIVDATA->__tp->sectHdrEntrySize)) + (numPermRights * sizeof(ESV1PermanentRecord)) + (numSubRights * sizeof(ESV1SubscriptionRecord)) + (numCntRights * sizeof(ESV1ContentRecord)) + (numCntLmtRights * sizeof(ESV1ContentConsumptionRecord)) + (numAccTitleRights *sizeof(ESV1AccessTitleRecord)));
        sectHdr.nRecords = ntohl(numLimResRights);
        sectHdr.recordSize = ntohl(sizeof(ESV1LimitedResourceRecord));
        sectHdr.sectionSize = ntohl(numLimResRights * sizeof(ESV1LimitedResourceRecord));
        sectHdr.sectionType = ntohs(ES_ITEM_RIGHT_LIMITED_RESOURCE);
        sectHdr.flags = 0;

        memcpy(buf, &sectHdr, sizeof(ESV1SectionHeader));
        buf += sizeof(ESV1SectionHeader);
    }

#ifdef __EVL_API__
    if (numGenRights > 0) {
        sectHdr.sectOfst = ntohl(ntohl(PRIVDATA->__tp->sectHdrOfst) + (nSections * ntohs(PRIVDATA->__tp->sectHdrEntrySize)) + (numPermRights * sizeof(ESV1PermanentRecord)) + (numSubRights * sizeof(ESV1SubscriptionRecord)) + (numCntRights * sizeof(ESV1ContentRecord)) + (numCntLmtRights * sizeof(ESV1ContentConsumptionRecord)) + (numAccTitleRights *sizeof(ESV1AccessTitleRecord)) + (numLimResRights * sizeof(ESV1LimitedResourceRecord)));
        sectHdr.nRecords = ntohl(numGenRights);
        sectHdr.recordSize = ntohl(sizeof(ESV1GenericRecord));
        sectHdr.sectionSize = ntohl(numGenRights * sizeof(ESV1GenericRecord));
        sectHdr.sectionType = ntohs(ES_ITEM_RIGHT_GENERIC);
        sectHdr.flags = 0;

        memcpy(buf, &sectHdr, sizeof(ESV1SectionHeader));
        buf += sizeof(ESV1SectionHeader);
    }
#endif

    /*
     * Add item right records
     */
    for (i = 0; i < numItemRights; i++) {
        if (itemRights[i].type == ES_ITEM_RIGHT_PERMANENT) {
            memcpy(permRec.referenceId, itemRights[i].right.perm.referenceId, ES_REFERENCE_ID_LEN);
            permRec.referenceIdAttr = ntohl(itemRights[i].right.perm.referenceIdAttr);
            memcpy(buf, &permRec, sizeof(ESV1PermanentRecord));
            buf += sizeof(ESV1PermanentRecord);
        }
    }

    for (i = 0; i < numItemRights; i++) {
        if (itemRights[i].type == ES_ITEM_RIGHT_SUBSCRIPTION) {
            subRec.limit = ntohl(itemRights[i].right.sub.limit);
            memcpy(subRec.referenceId, itemRights[i].right.sub.referenceId, ES_REFERENCE_ID_LEN);
            subRec.referenceIdAttr = ntohl(itemRights[i].right.sub.referenceIdAttr);
            memcpy(buf, &subRec, sizeof(ESV1SubscriptionRecord));
            buf += sizeof(ESV1SubscriptionRecord);
        }
    }

    /*
     * Figure out whether to use the item rights set with SetContentMask or
     * what has been supplied on the SetItemRights call
     */
    if (PRIVDATA->__numContItemRights > 0) {
        memcpy(buf, PRIVDATA->__contItemRights, PRIVDATA->__numContItemRights * sizeof(ESV1ContentRecord));
        buf += PRIVDATA->__numContItemRights * sizeof(ESV1ContentRecord);
    } else {
        for (i = 0; i < numItemRights; i++) {
            if (itemRights[i].type == ES_ITEM_RIGHT_CONTENT) {
                cntRec.offset = ntohl((u32) itemRights[i].right.cnt.offset);
                memcpy(cntRec.accessMask, itemRights[i].right.cnt.accessMask, ES_CONTENT_ITEM_ACCESS_MASK_LEN);
                memcpy(buf, &cntRec, sizeof(ESV1ContentRecord));
                buf += sizeof(ESV1ContentRecord);
            }
        }
    }

    for (i = 0; i < numItemRights; i++) {
        if (itemRights[i].type == ES_ITEM_RIGHT_CONTENT_CONSUMPTION) {
            if (!VALID_LIMIT_CODE(itemRights[i].right.cntLmt.code)) {
                pubLog(PUB_DEBUG_ERROR, "Invalid limit type %d\n",
                    itemRights[i].right.cntLmt.code);
                rv = ES_ERR_INVALID;
            }
            cntLmtRec.index = ntohs(itemRights[i].right.cntLmt.index);
            cntLmtRec.code = ntohs((u16) itemRights[i].right.cntLmt.code);
            cntLmtRec.limit = ntohl(itemRights[i].right.cntLmt.limit);
            memcpy(buf, &cntLmtRec, sizeof(ESV1ContentConsumptionRecord));
            buf += sizeof(ESV1ContentConsumptionRecord);
        }
    }

    for (i = 0; i < numItemRights; i++) {
        if (itemRights[i].type == ES_ITEM_RIGHT_ACCESS_TITLE) {
            accTitleRec.accessTitleId = ntohll(itemRights[i].right.accTitle.accessTitleId);
            accTitleRec.accessTitleMask = ntohll(itemRights[i].right.accTitle.accessTitleMask);
            memcpy(buf, &accTitleRec, sizeof(ESV1AccessTitleRecord));
            buf += sizeof(ESV1AccessTitleRecord);
        }
    }

    for (i = 0; i < numItemRights; i++) {
        if (itemRights[i].type == ES_ITEM_RIGHT_LIMITED_RESOURCE) {
            limResRec.limit = ntohl(itemRights[i].right.sub.limit);
            memcpy(limResRec.referenceId, itemRights[i].right.sub.referenceId, ES_REFERENCE_ID_LEN);
            limResRec.referenceIdAttr = ntohl(itemRights[i].right.sub.referenceIdAttr);
            memcpy(buf, &limResRec, sizeof(ESV1LimitedResourceRecord));
            buf += sizeof(ESV1LimitedResourceRecord);
        }
    }

#ifdef __EVL_API__
    for (i = 0; i < numItemRights; i++) {
        if (itemRights[i].type == ES_ITEM_RIGHT_GENERIC) {
            genRec.type = ntohl(itemRights[i].right.gen.type);
            memcpy(genRec.data, itemRights[i].right.gen.data, ES_GENERIC_ITEM_DATA_LEN);
            memcpy(buf, &genRec, sizeof(ESV1GenericRecord));
            buf += sizeof(ESV1GenericRecord);
        }
    }
#endif

    PRIVDATA->__setItemRightsCalled = true;

end:
    return rv;
}


/*
 * Sign the ticket using the RSA private key of the XS (2048 bits)
 */
ESError
PublishTicket::SignTicket(const u8 *issuerName, u8 *rsaPubMod, u8 *rsaPrivExp)
{
    ESError rv = ES_ERR_OK;
    int signLen;
    int headerLen;
    NN_ALIGNAS(4) IOSCHashContext hashCtx;
    int hashType = IOSC_HASH_SHA256;
    u8 hashVal[32];
    int hashLen = 32;

    /*
     * If SetItemRights has not been called, do that now to make sure content mask
     * is set in the ticket
     */
    if (!PRIVDATA->__setItemRightsCalled) {
        PublishTicket::SetItemRights(NULL, 0);
    }

    memset(&PRIVDATA->__tp->sig.issuer, 0, sizeof(PRIVDATA->__tp->sig.issuer));
    strncpy((char *)&PRIVDATA->__tp->sig.issuer, (char *)issuerName, sizeof(PRIVDATA->__tp->sig.issuer));

    /*
     * The signature covers everything from the "issuer" field to the end
     * of the ticket (including all the V1 headers and sections)
     */
    headerLen = (int)((u8*)&PRIVDATA->__tp->sig.issuer - (u8*)PRIVDATA->__tp);
    signLen = PRIVDATA->__ticketLen - headerLen;

    rv = IOSC_GenerateHash(hashCtx, NULL, 0, hashType|IOSC_HASH_FIRST, NULL);
    if (rv != IOSC_ERROR_OK) {
        pubLog(PUB_DEBUG_ERROR, "Initialize hash context (rv %d)\n", rv);
        goto end;
    }

    rv = IOSC_GenerateHash(hashCtx, (u8 *)&PRIVDATA->__tp->sig.issuer, signLen,
            hashType|IOSC_HASH_LAST, hashVal);
    if (rv != IOSC_ERROR_OK) {
        pubLog(PUB_DEBUG_ERROR, "Hash finalize failed (rv %d)\n", rv);
        goto end;
    }

    /*
     * Compute the RSA signature of the message digest
     *
     * In the real ticket server, this would be done using an HSM
     */
    CSL_RsaSignData(hashVal, hashLen,
        rsaPubMod, rsaPrivExp, sizeof(CSLOSRsaPublicKey2048),
        (u8 *)&PRIVDATA->__tp->sig.sig);

    PRIVDATA->__tp->sig.sigType = IOSC_SIG_RSA2048_H256;

end:
    return rv;
}


ESError
PublishTicket::PersonalizeTicket(u8 *privEccKey, u32 privEccKeyLen)
{
    ESError rv = ES_ERR_OK;
    IOSCSecretKeyHandle hSharedKey = 0;
    IOSCAesKey sharedKey;
    IOSCAesKey outKey;
    IOSCAesIv iv;

    if (privEccKey == NULL || privEccKeyLen != sizeof(IOSCEccPrivateKey)) {
        pubLog(PUB_DEBUG_ERROR, "Invalid parameters (keysize %d)\n", privEccKeyLen);
        rv = ES_ERR_INVALID;
        goto end;
    }

    if (!PRIVDATA->__devCertLoaded) {
        pubLog(PUB_DEBUG_ERROR, "No device cert (required for public key)\n");
        rv = ES_ERR_INVALID;
        goto end;
    }

    //
    // Using ECDH to calculate the shared secret for personalizing the ticket
    //
    rv = CSL_GenerateEccSharedKey(privEccKey, (u8 *)PRIVDATA->__devCert.pubKey, sharedKey, sizeof(sharedKey));
    if (rv != CSL_OK) {
        pubLog(PUB_DEBUG_ERROR, "Failed to generate ECDH shared key, rv=%d\n", rv);
        rv = ES_ERR_CRYPTO;
        goto end;
    }

    rv = IOSC_CreateObject(&hSharedKey, IOSC_SECRETKEY_TYPE, IOSC_ENC_SUBTYPE);
    if (rv != IOSC_ERROR_OK) {
        pubLog(PUB_DEBUG_ERROR, "Failed to create secret key object, rv=%d\n", rv);
        rv = ES_ERR_CRYPTO;
        goto end;
    }

    rv = IOSC_ImportSecretKey(hSharedKey, 0, 0, IOSC_NOSIGN_NOENC,
            NULL, NULL, sharedKey);
    if (rv != IOSC_ERROR_OK) {
        pubLog(PUB_DEBUG_ERROR, "Failed to import shared key, rv=%d\n", rv);
        rv = ES_ERR_CRYPTO;
        goto end;
    }

    //
    // Encrypt title key with shared key
    // IV is the ticketId in network byte order
    //
    memset(iv, 0, sizeof(IOSCAesIv));
    memcpy(iv, (u8 *) &PRIVDATA->__tp->ticketId, sizeof(ESTicketId));

    rv = IOSC_Encrypt(hSharedKey, iv, PRIVDATA->__tp->titleKey.aesKey, sizeof(IOSCAesKey), outKey);
    if (rv != IOSC_ERROR_OK) {
            pubLog(PUB_DEBUG_ERROR, "Failed to encrypt title key, rv=%d\n", rv);
            rv = ES_ERR_CRYPTO;
            goto end;
    }

    memcpy(PRIVDATA->__tp->titleKey.aesKey, outKey, sizeof(PRIVDATA->__tp->titleKey.aesKey));

end:
    if (hSharedKey) IOSC_DeleteObject(hSharedKey);

    return rv;
}


/*
 * For debugging
 */
ESError
PublishTicket::DumpTicket()
{
    ESError rv = ES_ERR_OK;
    u32 i, j, k;
    u32 nSectHdrs;
    ESV2SectionHeader *shp;
    ESV1SubscriptionRecord *subRecP;
    ESV1PermanentRecord *permRecP;
    ESV1ContentRecord *cntRecP;
    ESV1ContentConsumptionRecord *cntLmtRecP;
    ESV1AccessTitleRecord *accTitleRecP;
    ESV1LimitedResourceRecord *limResRecP;
#ifdef __EVL_API__
    ESV1GenericRecord *genRecP;
#endif

    printf("Contents of ticket at 0x%0*" PRIxPTR ":\n", (int)sizeof(PRIVDATA->__tp)*2, (uintptr_t)PRIVDATA->__tp);
    printf("  Ticket ID = 0x%" PRIx64 "\n", (uint64_t)PRIVDATA->__tp->ticketId);
    printf("  Version   = %" PRIi16 "\n", PRIVDATA->__tp->ticketVersion);
    printf("  Rights ID\n");
    for (i = 0; i < sizeof(ESRightsId); i++) {
        printf(" %02" PRIx8, PRIVDATA->__tp->rightsId[i]);
    }
    printf("\n");

    printf("  Device ID = 0x%" PRIx64 "\n", (uint64_t)PRIVDATA->__tp->deviceId);
    printf("  Issuer    = %s\n", PRIVDATA->__tp->sig.issuer);
    printf("  SigType   = 0x%0" PRIx32 "\n", PRIVDATA->__tp->sig.sigType);
    printf("  AccountId = 0x%" PRIx32 "\n", PRIVDATA->__tp->accountId);
    printf("  LicenseType = %" PRIi8 "\n", PRIVDATA->__tp->licenseType);
    printf("  CommonKeyId = %" PRIi8 "\n", PRIVDATA->__tp->commonKeyId);
    printf("  PropertyMask = 0x%" PRIx16 "\n", PRIVDATA->__tp->propertyMask);
    printf("  Title Key (encrypted and may be personalized):\n   ");
    for (i = 0; i < sizeof(ESTitleKey); i++) {
        printf(" %02" PRIx8, PRIVDATA->__tp->titleKey.rsaKey[i]);
    }
    printf("\n");

    printf("  sectTotalSize %" PRIi32 "\n", PRIVDATA->__tp->sectTotalSize);
    printf("  sectHdrOffset %" PRIi32 "\n", PRIVDATA->__tp->sectHdrOfst);
    printf("  nSectHdrs %" PRIi16 "\n", PRIVDATA->__tp->nSectHdrs);
    printf("  nSectHdrEntrySize %" PRIi16 "\n", PRIVDATA->__tp->sectHdrEntrySize);

    //
    // V1 section headers
    //
    nSectHdrs = PRIVDATA->__tp->nSectHdrs;
    shp = (ESV2SectionHeader *)((u8 *)&PRIVDATA->__tp[0] + PRIVDATA->__tp->sectHdrOfst);
    for (i = 0; i < nSectHdrs; i++) {
        u16 secType = shp->sectionType;
        u32 nRecords = shp->nRecords;
        printf("V1 Section Header %d:\n", i);
        printf("  sectOffset %" PRIi32 "\n", shp->sectOfst);
        printf("  nRecords %" PRIi32 "\n", nRecords);
        printf("  recordSize %" PRIi32 "\n", shp->recordSize);
        printf("  sectionSize %" PRIi32 "\n", shp->sectionSize);
        printf("  sectionType %s\n",
            (secType == ES_ITEM_RIGHT_PERMANENT) ? "PERMANENT" :
           ((secType == ES_ITEM_RIGHT_SUBSCRIPTION) ? "SUBSCRIPTION" :
           ((secType == ES_ITEM_RIGHT_CONTENT) ? "CONTENT" :
           ((secType == ES_ITEM_RIGHT_CONTENT_CONSUMPTION) ? "CONTENT CONSUMPTION" :
           ((secType == ES_ITEM_RIGHT_ACCESS_TITLE) ? "ACCESS TITLE" :
           ((secType == ES_ITEM_RIGHT_LIMITED_RESOURCE) ? "LIMITED RESOURCE" :
#ifdef __EVL_API__
           ((secType == ES_ITEM_RIGHT_GENERIC) ? "GENERIC" :
            "Unknown")))))));
#else
            "Unknown"))))));
#endif
        printf("  V1 Section Records:\n");

        //
        // Dump the records in the section
        //
        subRecP = (ESV1SubscriptionRecord *)((u8 *)&PRIVDATA->__tp[0] + shp->sectOfst);
        permRecP = (ESV1PermanentRecord *)subRecP;
        cntRecP = (ESV1ContentRecord *)subRecP;
        cntLmtRecP = (ESV1ContentConsumptionRecord *)subRecP;
        accTitleRecP = (ESV1AccessTitleRecord *)subRecP;
        limResRecP = (ESV1LimitedResourceRecord *)subRecP;
#ifdef __EVL_API__
        genRecP = (ESV1GenericRecord *)subRecP;
#endif

        for (j = 0; j < nRecords; j++) {
            if (secType == ES_ITEM_RIGHT_PERMANENT) {
                printf("    refId ");
                for (k = 0; k < ES_REFERENCE_ID_LEN; k++) {
                    printf("%02" PRIx8 " ", permRecP->referenceId[k] & 0xff);
                }
                printf("\n    refIdAttr %" PRIx32 "\n", permRecP->referenceIdAttr);
                permRecP++;
            } else if (secType == ES_ITEM_RIGHT_SUBSCRIPTION) {
                printf("    limit %" PRIx32 "\n", subRecP->limit);
                printf("    refId ");
                for (k = 0; k < ES_REFERENCE_ID_LEN; k++) {
                    printf("%02" PRIx8 " ", subRecP->referenceId[k] & 0xff);
                }
                printf("\n    refIdAttr %" PRIx32 "\n", subRecP->referenceIdAttr);
                subRecP++;
            } else if (secType == ES_ITEM_RIGHT_CONTENT) {
                printf("    offset %" PRIx32 "\n", cntRecP->offset);
                printf("    accessMask:\n");
                for (k = 0; k < ES_CONTENT_ITEM_ACCESS_MASK_LEN; k++) {
                    if ((k & 0x1f) == 0) printf("    ");
                    printf("%02" PRIx8 " ", cntRecP->accessMask[k] & 0xff);
                    if ((k & 0x1f) == 0x1f) printf("\n");
                }
                cntRecP++;
            } else if (secType == ES_ITEM_RIGHT_CONTENT_CONSUMPTION) {
                printf("    index %" PRIx16 "\n", cntLmtRecP->index);
                printf("    code %" PRIx16 "\n", cntLmtRecP->code);
                printf("    limit %" PRIx32 "\n", cntLmtRecP->limit);
                cntLmtRecP++;
            } else if (secType == ES_ITEM_RIGHT_ACCESS_TITLE) {
                printf("    accessTitleId %" PRIx64 "\n", (uint64_t)accTitleRecP->accessTitleId);
                printf("    accessTitleMask %" PRIx64 "\n", (uint64_t)accTitleRecP->accessTitleMask);
                accTitleRecP++;
            } else if (secType == ES_ITEM_RIGHT_LIMITED_RESOURCE) {
                printf("    limit %" PRIx32 "\n", limResRecP->limit);
                printf("    refId ");
                for (k = 0; k < ES_REFERENCE_ID_LEN; k++) {
                    printf("%02" PRIx8 " ", limResRecP->referenceId[k] & 0xff);
                }
                printf("\n    refIdAttr %" PRIx32 "\n", limResRecP->referenceIdAttr);
                limResRecP++;
#ifdef __EVL_API__
            } else if (secType == ES_ITEM_RIGHT_GENERIC) {
                printf("    type %" PRIx32 ", data:\n", ntohl(genRecP->type));
                for (k = 0; k < ES_GENERIC_ITEM_DATA_LEN; k++) {
                    if ((k & 0x1f) == 0) printf("    ");
                    printf("%02" PRIx8 " ", genRecP->data[k] & 0xff);
                    if ((k & 0x1f) == 0x1f) printf("\n");
                }
                printf("\n");
                genRecP++;
#endif
            } else {
                printf("  Unknown Record Type %" PRIx16 "\n", secType);
            }
        }
        shp++;
    }

    return rv;
}


/*
 * Return the current length of the ticket
 */
ESError
PublishTicket::GetTicketLen(u32 *ticketLen)
{
    ESError rv = ES_ERR_OK;

    *ticketLen = PRIVDATA->__ticketLen;

    return rv;
}


/*
 * Return the contents of the ticket to the caller
 */
ESError
PublishTicket::OutputTicket(u8 *ticketBuf, u32 ticketBufLen)
{
    ESError rv = ES_ERR_OK;

    if (ticketBuf == NULL || ticketBufLen < PRIVDATA->__ticketLen) {
        pubLog(PUB_DEBUG_ERROR, "Invalid parameters\n");
        rv = ES_ERR_INVALID;
        goto end;
    }
    memcpy(ticketBuf, PRIVDATA->__tp, PRIVDATA->__ticketLen);

end:
    return rv;
}

PUBLISH_NAMESPACE_END
