﻿// 文字コード:UTF-8
/// @file

#include <cctype>
#include <string>
#if PLATFORM_IS_WINDOWS
    #include <nn/nn_Windows.h>
#endif
#include "lib/CString.hpp"
#include "lib/U16String.hpp"
#include "lib/U16CString.hpp"
#include "lib/StringConvert.hpp"
#include "lib/StringUtil.hpp"

namespace lib {

//------------------------------------------------------------------------------
U16String U16String::GetEmpty()
{
    return U16String(1, U16(""));
}

//------------------------------------------------------------------------------
U16String U16String::FromUtf8(const String& aStr)
{
    U16String str(aStr.capacity());
    str.setFromUtf8(aStr);
    return str;
}

//------------------------------------------------------------------------------
U16String U16String::FromUtf8(int aCap, const char* aStr)
{
    SYS_ASSERT(0 < aCap);
    U16String retStr(aCap);
    retStr.setFromUtf8(aStr);
    return retStr;
}

//------------------------------------------------------------------------------
U16String U16String::FromUtf8AsPossible(int aCap, const char* aStr)
{
    SYS_ASSERT(0 < aCap);
    U16String retStr(aCap);
    retStr.setFromUtf8AsPossible(aStr);
    return retStr;
}

//------------------------------------------------------------------------------
U16String U16String::FromCharacter(char16_t aChar)
{
    const char16_t str[] = {aChar, 0};
    return U16String(str);
}

#if PLATFORM_IS_WINDOWS
//------------------------------------------------------------------------------
U16String U16String::FromSjis(const String& aStr)
{
    U16String str(aStr.capacity());
    str.setFromSjisAsPossible(aStr.buffer());
    return str;
}

//------------------------------------------------------------------------------
U16String U16String::FromSjis(int aCap, const char* aStr)
{
    SYS_ASSERT(0 < aCap);
    U16String retStr(aCap);
    retStr.setFromSjisAsPossible(aStr);
    return retStr;
}

//------------------------------------------------------------------------------
U16String U16String::FromSjisAsPossible(int aCap, const char* aStr)
{
    SYS_ASSERT(0 < aCap);
    U16String retStr(aCap);
    retStr.setFromSjisAsPossible(aStr);
    return retStr;
}
#endif

//------------------------------------------------------------------------------
U16String U16String::FromLiteral(const String& aStr)
{
#if PLATFORM_IS_WINDOWS
    return FromSjis(aStr);
#else
    return FromUtf8(aStr);
#endif
}

//------------------------------------------------------------------------------
U16String U16String::FromLiteral(int aCap, const char* aStr)
{
#if PLATFORM_IS_WINDOWS
    return FromSjis(aCap, aStr);
#else
    return FromUtf8(aCap, aStr);
#endif
}

//------------------------------------------------------------------------------
U16String U16String::FromLiteralAsPossible(int aCap, const char* aStr)
{
#if PLATFORM_IS_WINDOWS
    return FromSjisAsPossible(aCap, aStr);
#else
    return FromUtf8AsPossible(aCap, aStr);
#endif
}

//------------------------------------------------------------------------------
U16String U16String::FromFormat(int aCap, const char16_t* aStr, ...)
{
    ::std::va_list vaList = {};
    va_start(vaList, aStr);
    auto string = FromFormatV(aCap, aStr, vaList);
    va_end(vaList);
    return string;
}

//------------------------------------------------------------------------------
U16String U16String::FromFormatV(int aCap, const char16_t* aStr, ::std::va_list aVaList)
{
    auto string = U16String(aCap);
    string.formatV(aStr, aVaList);
    return string;
}

//------------------------------------------------------------------------------
U16String U16String::FromFormatAsPossible(int aCap, const char16_t* aStr, ...)
{
    ::std::va_list vaList = {};
    va_start(vaList, aStr);
    auto string = FromFormatVAsPossible(aCap, aStr, vaList);
    va_end(vaList);
    return string;
}

//------------------------------------------------------------------------------
U16String U16String::FromFormatVAsPossible(int aCap, const char16_t* aStr, ::std::va_list aVaList)
{
    auto string = U16String(aCap);
    string.formatVAsPossible(aStr, aVaList);
    return string;
}

//------------------------------------------------------------------------------
U16String::U16String()
: mBuf(0)
, mCap(0)
{}
//------------------------------------------------------------------------------
U16String::U16String(int aCap)
: U16String()
{
    SYS_ASSERT(0 < aCap);
    reset(aCap);
}
//------------------------------------------------------------------------------
U16String::U16String(const char16_t* aStr)
: U16String()
{
    reset(aStr);
}
//------------------------------------------------------------------------------
U16String::U16String(int aCap, const char16_t* aStr)
: U16String()
{
    SYS_ASSERT(0 < aCap);
    reset(aCap, aStr);
}

//------------------------------------------------------------------------------
U16String::U16String(int aCap, const char* aStr)
: U16String()
{
    SYS_ASSERT(0 < aCap);
    reset(aCap, aStr);
}
//------------------------------------------------------------------------------
U16String::U16String(const U16String& aString)
: U16String()
{
    reset(aString);
}

//------------------------------------------------------------------------------
U16String::~U16String()
{
    reset();
}
//------------------------------------------------------------------------------
void U16String::clear()
{
    if (mBuf) {
        SYS_ASSERT(mCap > 0);
        mBuf[0] = 0;
    }
}
//------------------------------------------------------------------------------
void U16String::reset()
{
    if (mBuf) {
        delete [] mBuf;
        mBuf = 0;
        mCap = 0;
    }
}
//------------------------------------------------------------------------------
void U16String::reset(int aCap)
{
    SYS_ASSERT(0 < aCap);
    reset();
    mCap = aCap;
    mBuf = new char16_t[mCap];
    mBuf[0] = 0;
}
//------------------------------------------------------------------------------
void U16String::reset(const U16String& aStr)
{
    if (aStr.isValid()) {
        reset(aStr.mCap);
        ::std::memcpy(mBuf, aStr.buffer(), sizeof(char16_t) * mCap);
    } else {
        reset();
    }
}
//------------------------------------------------------------------------------
void U16String::reset(const char16_t* aStr)
{
    int cap = U16CString::GetLength(aStr) + 1;
    reset(cap);
    set(aStr);
}
//------------------------------------------------------------------------------
void U16String::reset(int aCap, const char16_t* aStr)
{
    SYS_ASSERT(0 < aCap);
    reset(aCap);
    set(aStr);
}
//------------------------------------------------------------------------------
void U16String::reset(int aCap, const char* aStr)
{
    SYS_ASSERT(0 < aCap);
    reset(aCap);
    set(aStr);
}
//------------------------------------------------------------------------------
void U16String::set(const U16String& aStr)
{
    SYS_ASSERT(aStr.mCap <= mCap);
    if (aStr.isValid()) {
        if (isValid()) {
            ::std::memcpy(mBuf, aStr.buffer(), sizeof(char16_t) * aStr.mCap);
        } else {
            // 事前のmCap比較のAssertで止まるのでここまには到達しない
            SYS_DEBUG_ASSERT_NOT_REACHED_MSG(
                "this->capacity():%d, aStr.capacity():%d\n",
                capacity(),
                aStr.capacity()
                );
        }
    } else {
        // 引数aStrは空文字
        if (isValid()) {
            // 自身に空Stringが代入されたので自身をクリア
            clear();
            return;
        } else {
            // 空Stringから空Stringへのset呼び出しは代入済み扱いにする
            return;
        }
    }
}
//------------------------------------------------------------------------------
U16String& U16String::operator=(const U16String& aStr)
{
    set(aStr);
    return(*this);
}
//------------------------------------------------------------------------------
void U16String::setAsPossible(const U16String& aStr)
{
    SYS_ASSERT(isValid());
    SYS_ASSERT(aStr.isValid());
    SYS_ASSERT_POINTER(mBuf);
    ::std::memcpy(mBuf, aStr.buffer(), sizeof(char16_t) * mCap);
    if (mCap > 0) {
        mBuf[mCap - 1] = 0;
    }
}
//------------------------------------------------------------------------------
void U16String::set(const char16_t* aStr)
{
    int reqCap = U16CString::GetLength(aStr) + 1;
    SYS_ASSERT(reqCap <= mCap);
    ::std::memcpy(mBuf, aStr, sizeof(char16_t) * reqCap);
}
//------------------------------------------------------------------------------
void U16String::setAsPossible(const char16_t* aStr)
{
    int reqCap = U16CString::GetLength(aStr) + 1;
    int copyLen = (reqCap < mCap)? reqCap: mCap;
    ::std::memcpy(mBuf, aStr, sizeof(char16_t) * copyLen);
    mBuf[mCap - 1] = 0;
}
//------------------------------------------------------------------------------
U16String& U16String::operator=(const char16_t* aStr)
{
    set(aStr);
    return(*this);
}
//------------------------------------------------------------------------------
void U16String::set(const char16_t* aStr, int aLen)
{
    SYS_ASSERT(aLen < mCap);
    ::std::memcpy(mBuf, aStr, sizeof(char16_t) * aLen);
    mBuf[aLen] = 0;
}
//------------------------------------------------------------------------------
void U16String::setAsPossible(const char16_t* aStr, int aLen)
{
    int copyLen = (aLen < mCap)? aLen: mCap;
    ::std::memcpy(mBuf, aStr, sizeof(char16_t) * copyLen);
    if (copyLen < mCap) {
        mBuf[copyLen] = 0;
    } else {
        mBuf[mCap - 1] = 0;
    }
}
//------------------------------------------------------------------------------
void U16String::set(const char* aStr)
{
    int len = CString::GetLength(aStr, mCap);
    SYS_ASSERT(len < mCap);
    for (int i = 0; i < len; ++i) {
        SYS_ASSERT((char16_t)aStr[i] < 0x80);
        mBuf[i] = (char16_t)aStr[i];
    }
    mBuf[len] = 0;
}

//------------------------------------------------------------------------------
void U16String::setFromUtf8(const String& aStr)
{
    SYS_ASSERT(mCap >= aStr.capacity());
    setFromUtf8(aStr.buffer());
}

//------------------------------------------------------------------------------
void U16String::setFromUtf8(const char* aStr)
{
    // UTF8/16は1対1対応するわけではない為、最悪値でアサート
    SYS_ASSERT(CString::GetLength(aStr, mCap) < mCap);
    StringConvert::ConvertUtf8ToUtf16(mBuf, mCap, aStr);
}

//------------------------------------------------------------------------------
void U16String::setFromUtf8AsPossible(const String& aStr)
{
    SYS_ASSERT(isValid());
    setFromUtf8AsPossible(aStr.buffer());
}

//------------------------------------------------------------------------------
void U16String::setFromUtf8AsPossible(const char* aStr)
{
    StringConvert::ConvertUtf8ToUtf16(mBuf, mCap, aStr);
}

#if PLATFORM_IS_WINDOWS
//------------------------------------------------------------------------------
void U16String::setFromSjisAsPossible(const char* aStr)
{
    MultiByteToWideChar(CP_ACP, 0, aStr, -1, (wchar_t*)mBuf, mCap);
}
#endif

//------------------------------------------------------------------------------
void U16String::setFromLiteralAsPossible(const char* aStr)
{
#if PLATFORM_IS_WINDOWS
    setFromSjisAsPossible(aStr);
#else
    setFromUtf8AsPossible(aStr);
#endif
}

//------------------------------------------------------------------------------
int U16String::replace(int aIndex, int aRemoveLen, const char16_t* aInsertStr, int aInsertLen)
{
    remove(aIndex, aRemoveLen);
    return insert(aIndex, aInsertStr, aInsertLen);
}
//------------------------------------------------------------------------------
void U16String::replaceWord(char16_t aRemoveWord, char16_t aInsertWord)
{
    const int len = getLength();
    for (int i = 0; i < len; ++i) {
        if (mBuf[i] == aRemoveWord) {
            // aRemoveWord を aInsertWord に変換
            mBuf[i] = aInsertWord;
        }
    }
}
//------------------------------------------------------------------------------
void U16String::remove(int aIndex, int aLen)
{
    SYS_ASSERT(aIndex < getLength());
    SYS_ASSERT(0 < aLen);

    const int thisLen = getLength();

    // aLenを範囲内に納める
    if (thisLen < aIndex + aLen) {
        aLen = thisLen - aIndex;
    }

    const int maxIndex = thisLen - (aIndex + aLen);
    // 文字コピー（削除する分を詰める）
    for (int i = 0; i < maxIndex; ++i) {
        mBuf[aIndex + i] = mBuf[aIndex + aLen + i];
    }

    // 終端設定
    mBuf[maxIndex + aIndex] = 0;

    SYS_DEBUG_ASSERT(getLength() == thisLen - aLen);
}
//------------------------------------------------------------------------------
int U16String::insert(int aIndex, const char16_t* aStr, int aLen)
{
    SYS_ASSERT(aIndex <= getLength()); // 最後尾に挿入する場合もあるので、==を許容
    SYS_ASSERT_POINTER(aStr);

    const int thisLen = getLength();
    int addLen = aLen;

    {
        const int insertStrLen = U16CString::GetLength(aStr);
        // 長さ指定がない場合は、終端までの長さを設定
        if (addLen == -1) {
            addLen = insertStrLen;
        }

        // 指定の長さが追加文字列の長さを超えていたら、丸める
        if (insertStrLen < addLen) {
            addLen = insertStrLen;
        }
    }

    // 長さ0なら何もしない
    if (addLen == 0) {
        return 0;
    }

    // 後ろにくっつける文字をコピー
    const int tailLen = thisLen - aIndex;
    // 実際にコピーするtailの文字数
    int tailAddLen = tailLen;
    if (capacity() - 1 < thisLen + addLen) {
        // キャパオーバーなら、入れられるだけ入れる
        tailAddLen = capacity() - 1 - aIndex - addLen;
    }

    // 後ろに移動させる
    if (0 <= tailAddLen) {
        // 後ろからコピー
        const int start = tailLen - tailAddLen;
        for (int i = start; i < tailLen; ++i) {
            const int copyTo = thisLen - 1 + addLen - i;
            SYS_ASSERT(copyTo < capacity() - 1);
            mBuf[copyTo] = mBuf[thisLen - 1 - i];
        }
    } else {
        // 追加文字自体がキャパオーバー
        // 追加文字がはみ出さないように補正
        if (capacity() - 1 < aIndex + addLen) {
            addLen = capacity() - 1 - aIndex;
        }
    }

    SYS_ASSERT(0 <= addLen);

    // 追加文字コピー
    for (int i = 0; i < addLen; ++i) {
        mBuf[aIndex + i] = aStr[i];
    }

    // 終端設定
    {
        int termIndex = thisLen + addLen;
        if (capacity() <= termIndex) {
            termIndex = capacity() - 1;
        }
        mBuf[termIndex] = 0;
    }

    return addLen;
}
//------------------------------------------------------------------------------
U16String U16String::getSubstringByLength(int aStartIdx, int aLen) const
{
    SYS_ASSERT(isValid());
    SYS_ASSERT(aLen >= 0);
    SYS_ASSERT(aStartIdx + aLen <= getLength());
    U16String str(aLen + 1);
    StringUtil::GetSubstringByLength(str.mBuf, mBuf, aStartIdx, aLen);
    return str;
}
//------------------------------------------------------------------------------
int U16String::getLength() const
{
    SYS_ASSERT_POINTER(mBuf);
    return(U16CString::GetLength(mBuf));
}
//------------------------------------------------------------------------------
bool U16String::isASCII() const
{
    for (int i = 0; i < getLength(); ++i) {
        if (0x80 <= mBuf[i]) {
            // ASCII文字以外があった
            return false;
        }
    }
    return true;
}
//------------------------------------------------------------------------------
void U16String::correctAsUtf16IfPossible()
{
    // サロゲートペアの後ろが見つかった
    bool foundTail = false;

    // 末尾から検索した方が処理コストが僅かに安い
    for (int i = getLength() - 1; 0 <= i; --i) {
        const char16_t c = mBuf[i];

        if (foundTail) {
            // ペア(後)が見つかっている
            if (0xD800 <= c && c <= 0xDBFF) {
                // ペア(前)が見つかった。つまり、正しいサロゲートペアである
                foundTail = false;
            } else {
                // ペアが見つからなかったので、後ろを破棄する
                SYS_ASSERT(i + 1 < getLength());
                remove(i + 1, 1);
                // ペア(後)が連続して見つかったら、フラグは継続する
                foundTail = (0xDC00 <= c && c <= 0xDFFF);
            }

        } else {
            if (0xD800 <= c && c <= 0xDBFF) {
                // ペア(後)が見つかっていないのに、ペア(前)が見つかった
                // 破棄する
                remove(i, 1);
            } else if (0xDC00 <= c && c <= 0xDFFF) {
                // ペア(後)が見つかった
                foundTail = true;
            }
        }
    }
}

//------------------------------------------------------------------------------
void U16String::strip()
{
    const char16_t space(' ');
    const int length = getLength();
    // 先頭からの空白の数
    int startIdx = 0;
    {
        for (int i=0 ; i < length ; ++i) {
            if (space == at(i)) {
                ++startIdx;
            } else {
                break;
            }
        }
    }
    // strip後の文字長さ
    int strippedLength = length - startIdx;

    // 末尾からの空白の数
    {
        for (int i=length-1 ; i > startIdx ; --i) {
            if (space == at(i)) {
                --strippedLength;
            } else {
                break;
            }
        }
    }
    if (strippedLength < length) {
        SYS_ASSERT(0 <= strippedLength && strippedLength < length);
        setAsPossible(getSubstringByLength(startIdx, strippedLength));
    }
}

//------------------------------------------------------------------------------
bool U16String::isEmpty() const
{
    if (mBuf == nullptr) {
        return true;
    }
    if (mCap <= 0) {
        return true;
    }
    return mBuf[0] == 0x0000;
}
//------------------------------------------------------------------------------
U16String& U16String::operator+=(const U16String& aStr)
{
    SYS_ASSERT(getLength() + aStr.getLength() < mCap);
    U16CString::Catenate(mBuf, aStr.mBuf);
    return(*this);
}

//------------------------------------------------------------------------------
U16String U16String::operator+(const U16String& aStr) const
{
    auto string = U16String(capacity() + aStr.capacity());
    string.set(*this);
    string += aStr;
    return string;
}

//------------------------------------------------------------------------------
bool U16String::operator==(const char16_t* aStr) const
{
    return(equals(aStr));
}

//------------------------------------------------------------------------------
bool U16String::operator==(const U16String& aStr) const
{
    if (aStr.isValid()) {
        return operator==(aStr.buffer());
    } else {
        // 空文字同士ならtrue
        return !isValid();
    }
}

//------------------------------------------------------------------------------
bool U16String::equals(const char16_t* aStr, bool aIgnoresCase) const
{
    SYS_ASSERT_POINTER(aStr);
    if (!isValid()) {
        return(false);
    }
    int strLen = U16CString::GetLength(aStr);
    if (getLength() != strLen) {
        return(false);
    }
    if (aIgnoresCase) {
        for (int i = 0; i < strLen; ++i) {
            if (::std::tolower(mBuf[i]) != ::std::tolower(aStr[i])) {
                return(false);
            }
        }
    } else {
        for (int i = 0; i < strLen; ++i) {
            if (mBuf[i] != aStr[i]) {
                return(false);
            }
        }
    }
    return(true);
}

//------------------------------------------------------------------------------
bool U16String::startsWith(const char16_t* aStr, bool aWillIgnoresCase) const
{
    if (!isValid()) {
        return false;
    }

    int strLen = U16String(U16CString::GetLength(aStr) + 1, aStr).getLength();
    if (strLen > getLength()) {
        return false;
    }
    return getSubstringByLength(0, strLen).equals(aStr, aWillIgnoresCase);
}

//------------------------------------------------------------------------------
bool U16String::contains(char16_t aCh) const
{
    for (int i = 0; i < mCap; ++i) {
        if (mBuf[i] == 0) {
            break;
        }
        if (mBuf[i] == aCh) {
            return true;
        }
    }
    return false;
}

//------------------------------------------------------------------------------
void U16String::format(const char16_t* aFmt, ...)
{
    ::std::va_list vaList = {};
    va_start(vaList, aFmt);
    formatV(aFmt, vaList);
    va_end(vaList);
}

//------------------------------------------------------------------------------
void U16String::formatV(const char16_t* aFmt, ::std::va_list aVaList)
{
    U16CString::FormatV(mBuf, mCap, aFmt, aVaList);
}

//------------------------------------------------------------------------------
void U16String::formatAsPossible(const char16_t* aFmt, ...)
{
    ::std::va_list vaList = {};
    va_start(vaList, aFmt);
    formatVAsPossible(aFmt, vaList);
    va_end(vaList);
}

//------------------------------------------------------------------------------
void U16String::formatVAsPossible(const char16_t* aFmt, va_list aVaList)
{
    U16CString::FormatVAsPossible(mBuf, mCap, aFmt, aVaList);
}

//------------------------------------------------------------------------------
#if DEBUG_IS_ENABLED
void U16String::dump(const char* aMsg) const
{
    LIB_UNUSED(aMsg);
    LIB_PRINTF("%s:", aMsg == nullptr ? "U16String::dump" : aMsg);
    if (isValid()) {
        const String str = String::FromUtf16(*this);
        LIB_PRINTF("%s\n", str.buffer());
    } else {
        LIB_PRINTF("invalid\n");
    }
}
#endif

} // namespace
// EOF
