﻿// 文字コード:UTF-8
/// @file
#include "lib/Rect.hpp"

#include "lib/Math.hpp"

//------------------------------------------------------------------------------
namespace lib {

//------------------------------------------------------------------------------
Rect::Rect()
: mPosition()
, mSize()
{
}

//------------------------------------------------------------------------------
Rect::Rect(const ::lib::Vector2& aPosition, const ::lib::Vector2& aSize)
: mPosition(aPosition)
, mSize(aSize)
{
}

//------------------------------------------------------------------------------
Rect::Rect(float aPosX, float aPosY, float aWidth, float aHeight)
: mPosition(aPosX, aPosY)
, mSize(aWidth, aHeight)
{
}

//------------------------------------------------------------------------------
bool Rect::operator==(const Rect& aRect) const
{
    return mPosition == aRect.position() && mSize == aRect.size();
}

//------------------------------------------------------------------------------
void Rect::set(const ::lib::Vector2& aPosition, const ::lib::Vector2& aSize)
{
    mPosition = aPosition;
    mSize = aSize;
}

//------------------------------------------------------------------------------
bool Rect::isEmpty() const
{
    return (size().x <= 0) || (size().y <= 0);
}

//------------------------------------------------------------------------------
float Rect::getArea() const
{
    return mSize.x * mSize.y;
}

//------------------------------------------------------------------------------
bool Rect::contains(const ::lib::Vector2& aPos) const
{
    return
        position().x <= aPos.x &&
        position().x + size().x > aPos.x &&
        position().y <= aPos.y &&
        position().y + size().y >= aPos.y;
}

//------------------------------------------------------------------------------
bool Rect::contains(const Rect& aRect) const
{
    return
        position().x <= aRect.position().x &&
        position().x + size().x >= aRect.position().x + aRect.size().x &&
        position().y <= aRect.position().y &&
        position().y + size().y >= aRect.position().y + aRect.size().y;
}

//------------------------------------------------------------------------------
bool Rect::intersects(const Rect& aRect) const
{
    return
        position().x + size().x > aRect.position().x &&
        position().x < aRect.position().x + aRect.size().x &&
        position().y + size().y > aRect.position().y &&
        position().y < aRect.position().y + aRect.size().y;
}

//------------------------------------------------------------------------------
void Rect::intersect(const Rect& aRect)
{
    if (aRect.position().x > position().x) {
        size().x -= aRect.position().x - position().x;
        position().x = aRect.position().x;
    }
    if (aRect.position().y > position().y) {
        size().y -= aRect.position().y - position().y;
        position().y = aRect.position().y;
    }
    if (aRect.position().x + aRect.size().x < position().x + size().x) {
        size().x -= (position().x + size().x) - (aRect.position().x + aRect.size().x);
    }
    if (aRect.position().y + aRect.size().y < position().y + size().y) {
        size().y -= (position().y + size().y) - (aRect.position().y + aRect.size().y);
    }
}

//------------------------------------------------------------------------------
Rect Rect::getIntersected(const Rect& aRect) const
{
    Rect r = *this;
    r.intersect(aRect);
    return r;
}

//------------------------------------------------------------------------------
void Rect::unite(const Rect& aRect)
{
    if (aRect.isEmpty()) {
        return;
    }
    if (isEmpty()) {
        *this = aRect;
        return;
    }
    if (aRect.position().x < position().x) {
        size().x += position().x - aRect.position().x;
        position().x = aRect.position().x;
    }
    if (aRect.position().y < position().y) {
        size().y += position().y - aRect.position().y;
        position().y = aRect.position().y;
    }
    if (aRect.position().x + aRect.size().x > position().x + size().x) {
        size().x += (aRect.position().x + aRect.size().x) - (position().x + size().x);
    }
    if (aRect.position().y + aRect.size().y > position().y + size().y) {
        size().y += (aRect.position().y + aRect.size().y) - (position().y + size().y);
    }
}

//------------------------------------------------------------------------------
Rect Rect::getUnited(const Rect& aRect) const
{
    Rect r = *this;
    r.unite(aRect);
    return r;
}

//------------------------------------------------------------------------------
void Rect::clamp(const Rect& aRect)
{
    ::lib::Vector2 maxPos = aRect.getOppositePosition() - size();
    position().x = ::lib::Math::LimitMinMax(position().x, aRect.position().x, maxPos.x);
    position().y = ::lib::Math::LimitMinMax(position().y, aRect.position().y, maxPos.y);
}

//------------------------------------------------------------------------------
Rect Rect::getClamped(const Rect& aRect) const
{
    Rect r = *this;
    r.clamp(aRect);
    return r;
}

//------------------------------------------------------------------------------
void Rect::inflate(float aLen)
{
    inflate(::lib::Vector2(aLen, aLen));
}

//------------------------------------------------------------------------------
Rect Rect::getInflated(float aLen) const
{
    Rect r = *this;
    r.inflate(aLen);
    return r;
}

//------------------------------------------------------------------------------
void Rect::inflate(const ::lib::Vector2& aSize)
{
    position().x -= aSize.x;
    position().y -= aSize.y;
    size().x += 2 * aSize.x;
    size().y += 2 * aSize.y;
}

//------------------------------------------------------------------------------
Rect Rect::getInflated(const ::lib::Vector2& aSize) const
{
    Rect r = *this;
    r.inflate(aSize);
    return r;
}

//------------------------------------------------------------------------------
::lib::Vector2 Rect::getOppositePosition() const
{
    return ::lib::Vector2(mPosition + mSize);
}

//------------------------------------------------------------------------------
::lib::Vector2 Rect::getCenterPosition() const
{
    return ::lib::Vector2(
        position().x + 0.5f * size().x,
        position().y + 0.5f * size().y);
}

//------------------------------------------------------------------------------
::lib::Vector2 Rect::getNearestPosition(const ::lib::Vector2& aPos) const
{
    SYS_ASSERT(!isEmpty());
    return ::lib::Vector2(
        Math::LimitMinMax(aPos.x, mPosition.x, mPosition.x + mSize.x),
        Math::LimitMinMax(aPos.y, mPosition.y, mPosition.y + mSize.y)
        );
}

#if DEBUG_IS_ENABLED
//------------------------------------------------------------------------------
String Rect::toString() const
{
    return String::FromFormat(128, "((%f, %f), (%f, %f))",
        position().x, position().y,
        size().x, size().y);
}

#endif
} // namespace
// EOF
