﻿// 文字コード:UTF-8
/// @file
#pragma once

#include <cmath>
#include <lib/Type.hpp>
#include <lib/debug/Assert.hpp>
#include <lib/TypeTraitsEnableIf.hpp>
#include <lib/TypeTraitsIsUnsigned.hpp>

//------------------------------------------------------------------------------
/// @addtogroup LIB-Math
//@{
/// @name 各種マクロ
//@{
// 単位変換
#define LIB_MATH_DEG_TO_RAD(aDegree) ((aDegree) * 0.017453293f) ///< 度からラジアンへ
#define LIB_MATH_RAD_TO_DEG(aRadian) ((aRadian) * 57.295780f) ///< ラジアンから度へ
//@}

/// @name 各種定数。
/// 使用頻度の高さから、コンパイル時の置換を期待して、特別にdefineでfloatの定数を定義。
//@{
#define LIB_MATH_PI (3.1415927f) ///< pi
#define LIB_MATH_EPSILON (0.0001f) ///< よく使用する許容誤差値
#define LIB_MATH_FLOAT_MAX (3.4028235e38f) ///< floatの最大値
//@}
//@}

namespace lib {
/// @addtogroup LIB-Math
//@{
/// 数学ライブラリ
class Math
    : private ::lib::NonCreatable
{
public:
    /// @name 定数
    //@{
    static const int8_t   S8Max  = 0x7f; ///< int8_tでの最大値
    static const uint8_t  U8Max  = 0xff; ///< uint8_tでの最大値
    static const int16_t  S16Max = 0x7fff; ///< int16_tでの最大値
    static const uint16_t U16Max = 0xffff; ///< uint16_tでの最大値
    static const int32_t  S32Max = 0x7fffffff;  ///< int32_tでの最大値
    static const uint32_t U32Max = 0xffffffffU; ///< uint32_tでの最大値
    static const int64_t  S64Max = 0x7fffffffffffffffLL;  ///< int64_tでの最大値
    static const uint64_t U64Max = 0xffffffffffffffffULL; ///< uint64_tでの最大値
    //@}

    /// @name 三角関数
    //@{
    static inline float SinRad(float aRadian);
    static inline float CosRad(float aRadian);
    static inline float TanRad(float aRadian);
    static inline float SinDeg(float aDegree);
    static inline float CosDeg(float aDegree);
    static inline float TanDeg(float aDegree);
    /// アークサイン : [-1, 1] -> [-Pi/2, Pi/2]ラジアン
    static inline float ArcSinRad(float aValueMinus1toPlus1);
    /// アークコサイン : [-1, 1] -> [0, Pi]ラジアン
    static inline float ArcCosRad(float aValueMinus1toPlus1);
    /// アークサイン: [-1, 1] -> [-90, 90]度
    static inline float ArcSinDeg(float aValueMinus1toPlus1);
    /// アークコサイン : [-1, 1] -> [0, 180]度
    static inline float ArcCosDeg(float aValueMinus1toPlus1);
    /// アークタンジェント : ArcTan(aY/aX) -> -> (-Inf, Inf)ラジアン
    static inline float ArcTan2Rad(float aY, float aX);
    /// アークタンジェント : ArcTan(aY/aX) -> -> (-Inf, Inf)度
    static inline float ArcTan2Deg(float aY, float aX);
    //@}

    /// @name 比較・範囲関数
    //@{
    /// 0に等しいか（許容誤差あり）
    static inline bool IsZero(float aValue, float aTolerance = LIB_MATH_EPSILON);
    /// ２つのfloatが等しいか？（許容誤差あり）
    static inline bool Equals(float aValue1, float aValue2,
        float aTolerance = LIB_MATH_EPSILON);
    /// 大きい方を返す。
    template<typename TNum> static inline const TNum& Max(const TNum& aValue1,
        const TNum& aValue2);
    /// 小さい方を返す。
    template<typename TNum> static inline const TNum& Min(const TNum& aValue1,
        const TNum& aValue2);
    /// 切り捨て関数。
    static inline float Floor(float aValue);
    /// 切り上げ関数。
    static inline float Ceil(float aValue);
    /// 四捨五入する
    static inline float Round(float aValue);
    /// 指定範囲内に値が収まっているかを判定します。
    /// @details
    /// 条件 aMin <= aMax
    /// @param aMin     範囲A
    /// @param aMax     範囲B
    /// @param aValue 対象値
    /// @return 範囲内ならtrue
    template<typename TNum> static inline bool IsInRange(const TNum& aMin, const TNum& aMax,
        const TNum& aValue);
    /// aValueの大きさがaMin以下かを調べ、範囲に収めた値を返す
    template<typename TNum> static inline const TNum LimitMin(const TNum& aValue,
        const TNum& aMin);
    /// aValueの大きさがaMax以上かを調べ、範囲に収めた値を返す
    template<typename TNum> static inline const TNum LimitMax(const TNum& aValue,
        const TNum& aMax);
    /// aValueの大きさをaMin, aMaxの範囲に収めた値を返す
    template<typename TNum> static inline const TNum LimitMinMax(
        const TNum& aValue , const TNum& aMin, const TNum& aMax);
    template<typename TNum> static inline const TNum Clamp(const TNum& val,
        const TNum& aMin, const TNum& aMax);
    //@}

    /// @name 各種インライン関数
    //@{
    /// 絶対値(std::absと違い、unsignedな型も区別なく渡せます)
    template<typename TNum> static inline const TNum Abs(const TNum& aValue);
    /// 符号を返す。0なら0, 0より大きいなら+1, 0未満なら-1を返す。
    /// @details
    /// 以前は戻り型が int8_t でしたが、より適切と思われるTNumに変えました。
    /// (2013/06/11 hiroyuki)
    template<typename TNum> static inline TNum Sign(const TNum& aValue);
    /// 0でない符号を返す。0以上なら+1, 0未満なら-1を返す。
    template<typename TNum> static inline TNum NonZeroSign(const TNum& aValue);
    /// 2乗
    template<typename TNum> static inline const TNum Square(const TNum& aValue);
    /// 平方根
    static inline float Sqrt(float aValue);
    /// 浮動小数点での割り算の余り
    /// @details
    /// 戻り値を正確に言うと「ある整数Nについて、aValue = aUnitValue * N + R
    /// を満たすようなRで、ただしRの符号はaUnitValueと一致したもの」。
    ///
    /// 以下の例を参考にすると、仕様が分かりやすいかもしれない。
    /// @code
    /// Math::Repeat( 5.0f, 3.0f) // Result:  2.0
    /// Math::Repeat(-5.0f, 3.0f) // Result:  1.0
    /// Math::Repeat( 5.0f,-3.0f) // Result: -1.0
    /// Math::Repeat(-5.0f,-3.0f) // Result: -2.0
    /// @endcode
    ///
    /// 標準ライブラリのfmodfに近いが、負の数における扱いが違う。
    /// UnityのMathf.Repeatは同じ挙動らしい。
    ///
    /// 関連キーワード：fmodf, ModF, FMod, Modulo, Remainder, Loop
    static inline float Repeat(float aValue, float aUnitValue);
    /// 0未満にならない減算。
    template<typename TNum> static inline TNum SafeSub(const TNum& aLhs,
        const TNum& aRhs);
    /// ２次方程式を解く。
    /// @details
    /// ax^2 + bx + c = 0の解。
    ///
    /// 誤差が出にくいアルゴリズムを使用している。詳細は
    /// Numerical Recipes in C p158参照。
    ///
    /// @param aCoefA ２次の項の係数
    /// @param aCoefB １次の項の係数
    /// @param aCoefC 定数項
    /// @param[out] aOutRoot1 解１。解が０個の時は不定な値。
    /// @param[out] aOutRoot2 解２。解が０or１個の時は不定な値。
    /// @return 解の個数。
    static int SolveQuadraticEquation(float aCoefA, float aCoefB, float aCoefC,
        float* aOutRoot1, float* aOutRoot2);
    /// 切り上げする
    static inline float AlignUp(float aNum, int aAlignment);
    static inline int AlignUp(int aNum, int aAlignment);
    //@}

private:
    /// 標準ライブラリのfmodfのラッパー。
    /// @details
    /// Repeat()実装のために定義。外部には公開しておらず、
    /// 代わりにRepeat()を用いること。
    static inline float StdFmodfWrapper(float aNumerator, float aDenominator);

    /// Abs()の内部実装。TNumがunsignedの時
    template<typename TNum> static inline const TNum AbsImpl(const TNum& aValue,
        typename ::lib::TypeTraitsEnableIf<
            ::lib::TypeTraitsIsUnsigned<TNum>::Value>::Type*
        = nullptr);
    /// Abs()の内部実装。TNumがunsignedでない時
    template<typename TNum> static inline const TNum AbsImpl(const TNum& aValue,
        typename ::lib::TypeTraitsEnableIf<
            !::lib::TypeTraitsIsUnsigned<TNum>::Value>::Type*
        = nullptr);
};
//@}

//------------------------------------------------------------------------------
float Math::SinRad(const float aRadian)
{
    // GCCではstdにsinf等がない。グローバル空間にはsinfがある。
    // sinfはC99やC++11で定義されているからか？(2013/05/09 hiroyuki)
    return sinf(aRadian);
}

//------------------------------------------------------------------------------
float Math::CosRad(const float aRadian)
{
    return cosf(aRadian);
}

//------------------------------------------------------------------------------
float Math::TanRad(const float aRadian)
{
    return tanf(aRadian);
}

//------------------------------------------------------------------------------
float Math::SinDeg(const float aDegree)
{
    return SinRad(LIB_MATH_DEG_TO_RAD(aDegree));
}

//------------------------------------------------------------------------------
float Math::CosDeg(const float aDegree)
{
    return CosRad(LIB_MATH_DEG_TO_RAD(aDegree));
}

//------------------------------------------------------------------------------
float Math::TanDeg(const float aDegree)
{
    return TanRad(LIB_MATH_DEG_TO_RAD(aDegree));
}

//------------------------------------------------------------------------------
float Math::ArcSinRad(const float aValueMinus1toPlus1)
{
    return asinf(aValueMinus1toPlus1);
}

//------------------------------------------------------------------------------
float Math::ArcCosRad(const float aValueMinus1toPlus1)
{
    return acosf(aValueMinus1toPlus1);
}

//------------------------------------------------------------------------------
float Math::ArcSinDeg(const float aValueMinus1toPlus1)
{
    return LIB_MATH_RAD_TO_DEG(ArcSinRad(aValueMinus1toPlus1));
}

//------------------------------------------------------------------------------
float Math::ArcCosDeg(const float aValueMinus1toPlus1)
{
    return LIB_MATH_RAD_TO_DEG(ArcCosRad(aValueMinus1toPlus1));
}

//------------------------------------------------------------------------------
float Math::ArcTan2Rad(const float aY, const float aX)
{
    return atan2f(aY, aX);
}

//------------------------------------------------------------------------------
float Math::ArcTan2Deg(const float aY, const float aX)
{
    return LIB_MATH_RAD_TO_DEG(ArcTan2Rad(aY, aX));
}

//------------------------------------------------------------------------------
bool Math::IsZero(const float aValue, const float aTolerance)
{
    return (aValue < aTolerance) && (aValue > -aTolerance);
}

//------------------------------------------------------------------------------
bool Math::Equals(const float aValue1, const float aValue2,
    const float aTolerance)
{
    return (-aTolerance < aValue1 - aValue2) &&
        (aValue1 - aValue2 < aTolerance);
}

//------------------------------------------------------------------------------
template<typename TNum> const TNum& Math::Max(const TNum& aValue1,
    const TNum& aValue2)
{
    return aValue1 > aValue2 ? aValue1 : aValue2;
}

//------------------------------------------------------------------------------
template<typename TNum> const TNum& Math::Min(const TNum& aValue1,
    const TNum& aValue2)
{
    return aValue1 < aValue2 ? aValue1 : aValue2;
}

//------------------------------------------------------------------------------
float Math::Floor(const float aValue)
{
    return floorf(aValue);
}

//------------------------------------------------------------------------------
float Math::Ceil(const float aValue)
{
    return ceilf(aValue);
}

//------------------------------------------------------------------------------
float Math::Round(const float aValue)
{
    return floorf(aValue + 0.5f);
}

//------------------------------------------------------------------------------
template<typename TNum> bool Math::IsInRange(const TNum& aMin, const TNum& aMax,
    const TNum& aValue)
{
    return aMin <= aValue && aValue <= aMax;
}

//------------------------------------------------------------------------------
template<typename TNum> const TNum Math::LimitMin(const TNum& aValue,
    const TNum& aMin)
{
    if (aValue < aMin) {
        return aMin;
    } else {
        return aValue;
    }
}

//------------------------------------------------------------------------------
template<typename TNum> const TNum Math::LimitMax(const TNum& aValue,
    const TNum& aMax)
{
    if (aValue > aMax) {
        return aMax;
    } else {
        return aValue;
    }
}

//------------------------------------------------------------------------------
template<typename TNum> const TNum Math::LimitMinMax(const TNum& aValue,
    const TNum& aMin, const TNum& aMax)
{
    // 演算順序で結果が変わってしまう事故の元になる為、大小関係は守る
    // ::lib::LimitMinMaxAssign からの移植
    SYS_DEBUG_ASSERT(aMin <= aMax);
    if (aValue > aMax) {
        return aMax;
    } else if (aValue < aMin) {
        return aMin;
    } else {
        return aValue;
    }
}

//------------------------------------------------------------------------------
template<typename TNum> const TNum Math::Clamp(const TNum& aValue,
    const TNum& aMin, const TNum& aMax)
{
    return LimitMinMax(aValue, aMin, aMax);
}

//------------------------------------------------------------------------------
template<typename TNum> const TNum Math::Abs(const TNum& aValue)
{
    return AbsImpl(aValue);
}

//------------------------------------------------------------------------------
template<typename TNum> TNum Math::Sign(const TNum& aValue)
{
    if (aValue == 0) {
        return TNum(0);
    } else if (aValue < 0) {
        return TNum(-1);
    } else {
        return TNum(1);
    }
}

//------------------------------------------------------------------------------
template<typename TNum> TNum Math::NonZeroSign(const TNum& aValue)
{
    if (aValue < 0) {
        return TNum(-1);
    } else {
        return TNum(1);
    }
}

//------------------------------------------------------------------------------
template<typename TNum> const TNum Math::Square(const TNum& aValue)
{
    return aValue * aValue;
}

//------------------------------------------------------------------------------
float Math::Sqrt(const float aValue)
{
    return sqrtf(aValue);
}

//------------------------------------------------------------------------------
float Math::Repeat(const float aValue, const float aUnitValue)
{
    if (0.0f < aUnitValue) {
        if (0.0f <= aValue) {
            return StdFmodfWrapper(aValue, aUnitValue);
        } else {
            const float tmpResult = StdFmodfWrapper(-aValue, aUnitValue);
            return aUnitValue - tmpResult;
        }
    } else if (aUnitValue < 0.0f) {
        if (0.0f <= aValue) {
            const float tmpResult = StdFmodfWrapper(-aValue, aUnitValue);
            return aUnitValue - tmpResult;
        } else {
            return StdFmodfWrapper(aValue, aUnitValue);
        }
    } else {
        SYS_ASSERT_NOT_REACHED();
        return 0.0f;
    }
}

//------------------------------------------------------------------------------
float Math::AlignUp(float aNum, int aAlignment)
{
    if (aAlignment == 0) {
        return aNum;
    }
    return Ceil(aNum / aAlignment) * aAlignment;
}

//------------------------------------------------------------------------------
int Math::AlignUp(int aNum, int aAlignment)
{
    if (aAlignment == 0) {
        return aNum;
    }
    return ((aNum + aAlignment - 1) / aAlignment) * aAlignment;
}

//------------------------------------------------------------------------------
template<typename TNum> TNum Math::SafeSub(const TNum& aLhs, const TNum& aRhs)
{
    return (aLhs < aRhs) ? TNum(0) : (aLhs - aRhs);
}

//------------------------------------------------------------------------------
float Math::StdFmodfWrapper(float aNumerator, float aDenominator)
{
    return fmodf(aNumerator, aDenominator);
}

//------------------------------------------------------------------------------
template<typename TNum> const TNum Math::AbsImpl(
    const TNum& aValue,
    typename ::lib::TypeTraitsEnableIf<
        ::lib::TypeTraitsIsUnsigned<TNum>::Value>::Type*)
{
    return aValue;
}

//------------------------------------------------------------------------------
template<typename TNum> const TNum Math::AbsImpl(
    const TNum& aValue,
    typename ::lib::TypeTraitsEnableIf<
        !::lib::TypeTraitsIsUnsigned<TNum>::Value>::Type*)
{
    return aValue < 0 ? -aValue : aValue;
}

} // namespace
// EOF
