﻿// 文字コード:UTF-8
/// @file
#pragma once

#include <lib/FixedStringImpl.hpp>

//------------------------------------------------------------------------------
namespace lib {

/// @addtogroup LIB-String
//@{

/// 固定容量の文字列クラスのテンプレート。
/// @details
/// 内部でalloc,freeを呼ばない代わりに長い文字列は扱えない。
/// 文字列の長さはテンプレート引数で指定する。
template <int TArgCapacity>
class FixedStringPod
{
public:
    typedef char CharType;

    /// 自分の型のエイリアス。
    typedef FixedStringPod<TArgCapacity> MyType;

    /// 容量。
    static const int Capacity = TArgCapacity;

    /// printfフォーマットで作成。
    inline static const MyType FromFormat(const CharType* aFormat, ...);
    /// vprintfフォーマットで作成。
    inline static const MyType FromFormatVA(
        ::std::va_list aArg,
        const CharType* aFormat
        );

    /// Read用ポインタの取得。
    const CharType* readPtr() const { return prvBuffer; }
    /// 書き込みも可能なポインタの取得。
    CharType* ptr() { return prvBuffer; }
    /// ２つの文字列が等しいかどうか調べる。
    bool isEquals(const CharType* aStr) const { return FixedStringImpl::Strcmp(readPtr(), aStr) == 0; }
    /// 書式指定無しで代入する。
    void set(const CharType* aStr) { FixedStringImpl::Strncpy(ptr(), aStr, Capacity); }
    /// 文字列の長さを返す。
    int length() const { return FixedStringImpl::Strlen(readPtr()); }
    /// 空文字列ならばtrue。
    bool isEmpty() const { return length() == 0; }
    /// 末尾に文字列を追加する。バッファサイズを超える場合、可能な限り追加する。
    void append(const CharType* aStr) { FixedStringImpl::Append(ptr(), aStr, Capacity); }
    /// 文字列を空にする。
    void clear() { prvBuffer[0] = '\0'; }

    /// @name 演算子オーバーロード
    //@{
    bool operator==(const MyType& aRhs) const { return isEquals(aRhs.readPtr()); }
    bool operator!=(const MyType& aRhs) const { return !(*this == aRhs); }

    bool operator<(const MyType& aRhs) const { return FixedStringImpl::Strcmp(readPtr(), aRhs.readPtr()) < 0; }
    bool operator>(const MyType& aRhs) const { return !(*this < aRhs || *this == aRhs); }

    bool operator<=(const MyType& aRhs) const { return !(aRhs < *this); }
    bool operator>=(const MyType& aRhs) const { return !(*this < aRhs); }
    inline MyType& operator+=(const MyType& aRhs);
    inline const MyType operator+(const MyType& aRhs) const;
    //@}

    /// POD型にするため、publicにする。
    CharType prvBuffer[Capacity];
};

//@}

//------------------------------------------------------------------------------
template <int TArgCapacity>
const FixedStringPod<TArgCapacity> FixedStringPod<TArgCapacity>::FromFormat(
    const CharType* aFormat,
    ...
    )
{
    ::std::va_list arg;
    va_start(arg, aFormat);
    const MyType result = FromFormatVA(arg, aFormat);
    va_end(arg);
    return result;
}

//------------------------------------------------------------------------------
template <int TArgCapacity>
const FixedStringPod<TArgCapacity> FixedStringPod<TArgCapacity>::FromFormatVA(
    ::std::va_list aArg,
    const CharType* aFormat
    )
{
    MyType str;
    FixedStringImpl::VPrintf(str.ptr(), Capacity, aArg, aFormat);
    return str;
}

//------------------------------------------------------------------------------
template <int TArgCapacity>
FixedStringPod<TArgCapacity>& FixedStringPod<TArgCapacity>::
    operator+=(const FixedStringPod<TArgCapacity>& aRhs)
{
    append(aRhs.prvBuffer);
    return *this;
}

//------------------------------------------------------------------------------
template <int TArgCapacity>
const FixedStringPod<TArgCapacity> FixedStringPod<TArgCapacity>::
    operator+(const FixedStringPod<TArgCapacity>& aRhs) const
{
    FixedStringPod<TArgCapacity> tmp = *this;
    return tmp += aRhs;
}

} // namespace
// EOF
