﻿// 文字コード:UTF-8
/// @file
#pragma once

#include "lib/DynamicArray.hpp"

namespace lib {

/// キュー、「::lib::DynamicArray」のキュー版
template<typename TType>
class DynamicQueue
{
public:
    DynamicQueue();
    DynamicQueue(int aCap);
    ~DynamicQueue() {}
    void reset(int aCap);
    void reset();
    void clear();
    int capacity() const { return mCap; }
    bool isEmpty() const { return !mIsFull && mEndIdx == mBeginIdx; }
    bool isFull() const { return mIsFull; }
    int getCount() const { return mIsFull ? mCap : (((mEndIdx + mCap) - mBeginIdx) % mCap); }
    TType& at(int idx) { SYS_ASSERT(0 <= idx && idx < getCount()); return mAry.at((mBeginIdx + idx) % mCap); }
    const TType& at(int aIdx) const { return const_cast<DynamicQueue<TType>*>(this)->at(aIdx); }
    TType& first() { return at(0); }
    const TType& first() const { return at(0); }
    TType& last() { return at(getCount() - 1); }
    const TType& last() const { return at(getCount() - 1); }
    TType popFront();
    TType popBack();
    TType pop() { return popFront(); }
    void popIfFull();
    void pushFront(const TType& aVal);
    void pushBack(const TType& aVal);
    void push(const TType& aVal) { pushBack(aVal); }
    void remove(int aIdx);
private:
    int mCap;
    bool mIsFull;
    int mBeginIdx;
    int mEndIdx;
    DynamicArray<TType> mAry;
};

//------------------------------------------------------------------------------
template<typename TType>
DynamicQueue<TType>::DynamicQueue()
: mCap(0)
, mIsFull(false)
, mBeginIdx(0)
, mEndIdx(0)
{
}

//------------------------------------------------------------------------------
template<typename TType>
DynamicQueue<TType>::DynamicQueue(int aCap)
: mCap(0)
, mIsFull(false)
, mBeginIdx(0)
, mEndIdx(0)
{
    reset(aCap);
}

//------------------------------------------------------------------------------
template<typename TType>
void DynamicQueue<TType>::reset(int aCap)
{
    mCap = aCap;
    mIsFull = false;
    mBeginIdx = 0;
    mEndIdx = 0;
    mAry.reset(aCap);
    mAry.setCountToCapacity();
}

//------------------------------------------------------------------------------
template<typename TType>
void DynamicQueue<TType>::reset()
{
    mCap = 0;
    mIsFull = false;
    mBeginIdx = 0;
    mEndIdx = 0;
    mAry.reset();
}

//------------------------------------------------------------------------------
template<typename TType>
void DynamicQueue<TType>::clear()
{
    mIsFull = false;
    mBeginIdx = 0;
    mEndIdx = 0;
}

//------------------------------------------------------------------------------
template<typename TType>
TType DynamicQueue<TType>::popFront()
{
    SYS_ASSERT(!isEmpty());
    TType val = first();
    mBeginIdx = (mBeginIdx + 1) % mCap;
    mIsFull = false;
    return val;
}
//------------------------------------------------------------------------------
template<typename TType>
TType DynamicQueue<TType>::popBack()
{
    SYS_ASSERT(!isEmpty());
    TType val = last();
    mEndIdx = (mEndIdx - 1 + mCap) % mCap;
    mIsFull = false;
    return val;
}

//------------------------------------------------------------------------------
template<typename TType>
void DynamicQueue<TType>::popIfFull()
{
    if (isFull()) {
        pop();
    }
}

//------------------------------------------------------------------------------
template<typename TType>
void DynamicQueue<TType>::pushFront(const TType& aVal)
{
    SYS_ASSERT(!isFull());
    mBeginIdx = (mBeginIdx - 1 + mCap) % mCap;
    mIsFull = (mBeginIdx == mEndIdx);
    first() = aVal;
}

//------------------------------------------------------------------------------
template<typename TType>
void DynamicQueue<TType>::pushBack(const TType& aVal)
{
    SYS_ASSERT(!isFull());
    mEndIdx = (mEndIdx + 1) % mCap;
    mIsFull = (mBeginIdx == mEndIdx);
    last() = aVal;
}

//------------------------------------------------------------------------------
template<typename TType>
void DynamicQueue<TType>::remove(int aIdx)
{
    int num = getCount();
    for (int i = aIdx; i < num - 1; ++i) {
        at(i) = at(i + 1);
    }
    popBack();
}

} // namespace
// EOF
