﻿// 文字コード:UTF-8
/// @file

#pragma once


//------------------------------------------------------------------------------
namespace lib {

/// 動的配列
/// @detail
/// 基本方針は「std::vector」とほぼ同等だが、
/// アロケーションは「reset」以外では発生せず、アロケーションの明確な制御ができる
template<typename TType>
class DynamicArray
    : private ::lib::NonCopyable
{
public:
    using iterator = TType* ; ///< 非constのイテレータ。
    using const_iterator = const TType* ; ///< constのイテレータ。

    DynamicArray();
    DynamicArray(int aCap);
    DynamicArray(::std::initializer_list<TType> aValues);
    DynamicArray(DynamicArray&& aDynamicArray);
    ~DynamicArray();
    void reset();
    void reset(int aCap);
    void reset(::std::initializer_list<TType> aValues);
    void setLength(int aCount);
    void setLengthToCapacity() { setCount(mCap); }
    void setCount(int aCount);
    void setCountToCapacity() { setCount(mCap); }
    void clear();
    DynamicArray& operator=(DynamicArray&& aDynamicArray);
    DynamicArray& operator+=(const DynamicArray& aDynamicArray);
    DynamicArray operator+(const DynamicArray& aDynamicArray) const;
    TType& operator[](int aIndex) { return at(aIndex); }
    TType& at(int i);
    const TType& operator[](int aIndex) const { return at(aIndex); }
    const TType& at(int aIndex) const { return const_cast<DynamicArray<TType>*>(this)->at(aIndex); }
    TType& first();
    TType& last();
    bool isEmpty() const { return mLength == 0; }
    bool isFull() const { return mLength == mCap; }
    int length() const { return mLength; }
    int count() const { return mLength; }
    int capacity() const { return mCap; }
    void push(const TType& aVal);
    TType& pop();
    void insert(int aIndex, const TType& aVal);
    template <typename... TArgs> void emplace(TArgs... aConstructorArgs);
    // 指定インデックスの要素を雑に取り除く(最後の要素のインデックスが変化する副作用がある)
    void removeRoughly(int i);
    // 指定要素を含むか？
    bool contains(const TType& aValue) const;

    iterator begin() { return &mAry[0]; }
    iterator end()   { return &mAry[mLength]; }
    const_iterator begin() const { return &mAry[0]; }
    const_iterator end() const   { return &mAry[mLength]; }

private:
    void release();

    TType* mAry;
    int mCap;
    int mLength;
};

//------------------------------------------------------------------------------
template<typename TType>
DynamicArray<TType>::DynamicArray()
: mAry(0)
, mCap(0)
, mLength(0)
{}

//------------------------------------------------------------------------------
template<typename TType>
DynamicArray<TType>::DynamicArray(int aCap)
: DynamicArray()
{
    reset(aCap);
}

//------------------------------------------------------------------------------
template<typename TType>
DynamicArray<TType>::DynamicArray(::std::initializer_list<TType> aValues)
: DynamicArray()
{
    reset(aValues);
}

//------------------------------------------------------------------------------
template<typename TType>
DynamicArray<TType>::DynamicArray(DynamicArray&& aDynamicArray)
: DynamicArray()
{
    *this = ::std::move(aDynamicArray);
}

//------------------------------------------------------------------------------
template<typename TType>
DynamicArray<TType>::~DynamicArray()
{
    reset();
}

//------------------------------------------------------------------------------
template<typename TType>
void DynamicArray<TType>::reset()
{
    if (mAry) {
        delete[] mAry;
        mAry = 0;
        mLength = mCap = 0;
    }
}

//------------------------------------------------------------------------------
template<typename TType>
void DynamicArray<TType>::reset(int aCap)
{
    SYS_ASSERT(aCap >= 0);
    reset();
    // ゼロ長の配列を「new」してしまいメモリ確保が発生することを防ぐ
    if (aCap == 0) {
        return;
    }
    SYS_ASSERT(mAry == 0 && mCap == 0 && mLength == 0);
    mCap = aCap;
    mAry = new TType[aCap];
    mLength = 0;
}

//------------------------------------------------------------------------------
template<typename TType>
void DynamicArray<TType>::reset(::std::initializer_list<TType> aValues)
{
    reset(int(aValues.size()));
    for (auto val : aValues) {
        push(val);
    }
}

//------------------------------------------------------------------------------
template<typename TType>
void DynamicArray<TType>::clear()
{
    mLength = 0;
}

//------------------------------------------------------------------------------
template<typename TType>
DynamicArray<TType>& DynamicArray<TType>::operator=(DynamicArray<TType>&& aDynamicArray)
{
    reset();
    // 所有権を移す
    mAry    = aDynamicArray.mAry;
    mCap    = aDynamicArray.mCap;
    mLength = aDynamicArray.mLength;
    aDynamicArray.release();

    return *this;
}

//------------------------------------------------------------------------------
template<typename TType>
DynamicArray<TType>& DynamicArray<TType>::operator+=(const DynamicArray& aDynamicArray)
{
    SYS_ASSERT(length() + aDynamicArray.length() <= capacity());
    for (auto val : aDynamicArray) {
        push(val);
    }
    return *this;
}

//------------------------------------------------------------------------------
template<typename TType>
DynamicArray<TType> DynamicArray<TType>::operator+(const DynamicArray& aDynamicArray) const
{
    auto result = DynamicArray<TType>(capacity() + aDynamicArray.capacity());
    result += *this;
    result += aDynamicArray;
    return ::std::move(result);
}

//------------------------------------------------------------------------------
template<typename TType>
void DynamicArray<TType>::setCount(int aCount)
{
    SYS_ASSERT(0 <= aCount && aCount <= mCap);
    mLength = aCount;
}

//------------------------------------------------------------------------------
template<typename TType>
TType& DynamicArray<TType>::at(int aIndex)
{
    SYS_ASSERT(-mLength <= aIndex && aIndex < mLength && mLength <= mCap);
    return mAry[aIndex];
}

//------------------------------------------------------------------------------
template<typename TType>
TType& DynamicArray<TType>::first()
{
    return at(0);
}

//------------------------------------------------------------------------------
template<typename TType>
TType& DynamicArray<TType>::last()
{
    SYS_ASSERT(mLength > 0);
    return at(mLength - 1);
}

//------------------------------------------------------------------------------
template<typename TType>
void DynamicArray<TType>::push(const TType& aVal)
{
    SYS_ASSERT(mLength < mCap);
    mAry[mLength] = aVal;
    mLength++;
}

//------------------------------------------------------------------------------
template<typename TType>
template<typename... TArgs>
void DynamicArray<TType>::emplace(TArgs... aConstructorArgs)
{
    SYS_ASSERT(mLength < mCap);
    // オブジェクトを再構築する
    {
        auto address = &mAry[mLength];
        address->~TType();
        new (address) TType(aConstructorArgs...);
    }
    mLength++;
}

//------------------------------------------------------------------------------
template<typename TType>
TType& DynamicArray<TType>::pop()
{
    TType& l = last();
    mLength--;
    return l;
}

//------------------------------------------------------------------------------
template<typename TType>
void DynamicArray<TType>::insert(int aIndex, const TType& aVal)
{
    SYS_ASSERT(0 <= aIndex && aIndex <= mLength);
    SYS_ASSERT(mLength < mCap);
    mLength++;
    for (int i = mLength - 1; i > aIndex; --i) {
        mAry[i] = mAry[i - 1];
    }
    mAry[aIndex] = aVal;
}

//------------------------------------------------------------------------------
template<typename TType>
void DynamicArray<TType>::removeRoughly(int aIndex)
{
    SYS_ASSERT(0 <= aIndex && aIndex < mLength && mLength <= mCap);
    mAry[aIndex] = last();
    pop();
}

//------------------------------------------------------------------------------
template<typename TType>
bool DynamicArray<TType>::contains(const TType& aValue) const
{
    for (int i = 0; i < mLength; ++i) {
        if (mAry[i] == aValue) {
            return true;
        }
    }
    return false;
}

//------------------------------------------------------------------------------
template<typename TType>
void DynamicArray<TType>::release()
{
    mAry = nullptr;
    mCap = 0;
    mLength = 0;
}

} // namespace

// EOF
