﻿// 文字コード:UTF-8
/// @file
#include "lib/CString.hpp"

#include <cctype>
#include <cstring>

namespace lib {

//------------------------------------------------------------------------------
int CString::GetMaxUtf8CapacityFromUtf16Length(int aStringLength)
{
    // 最大で文字列長は3倍になる
    // ・UTF8にはJIS X 0213の第3・4水準漢字など6byte文字も存在するが、
    //   UTF16ではサロゲートペアになる為問題ない
    // ・WebKitのJSStringGetMaximumUTF8CStringSizeにも同等の処理がある
    return 3 * aStringLength + 1;
}

//------------------------------------------------------------------------------
int CString::GetLength(const char* aStr, int aCap)
{
    if (aCap <= 0) {
        return aCap;
    }
    SYS_ASSERT_POINTER(aStr);
    const char* nullPos = reinterpret_cast<const char*>(::std::memchr(aStr, 0, aCap));
    if (nullPos) {
        return static_cast<int>(nullPos - aStr);
    }
    // 見つからなかった
    return aCap;
}

//------------------------------------------------------------------------------
int CString::GetLength(const char* aStr, int aCap, int aCh)
{
    if (aCap <= 0) {
        return aCap;
    }
    SYS_ASSERT_POINTER(aStr);
    const char* nullPos = reinterpret_cast<const char*>(::std::memchr(aStr, aCh, aCap));
    if (nullPos) {
        return static_cast<int>(nullPos - aStr);
    }
    // 見つからなかった
    return aCap;
}

//------------------------------------------------------------------------------
int CString::Compare(const char* aS1, const char* aS2, int aCap, bool aWillIgnoreCase)
{
    if (aWillIgnoreCase == false) {
        return ::std::strncmp(aS1, aS2, aCap);
    }
    else {
        for (int i = 0; i < aCap; ++i) {
            char c1 = static_cast<char>(::std::tolower(aS1[i]));
            char c2 = static_cast<char>(::std::tolower(aS2[i]));
            if (c1 != c2) {
                return c1 - c2;
            }
            if (c1 == '\0' && c2 == '\0') {
                return 0;
            }
        }
        return 0;
    }
}

//------------------------------------------------------------------------------
bool CString::IsHexChar(char aCh)
{
    return
        ('0' <= aCh && aCh <='9') ||
        ('a' <= aCh && aCh <='f') ||
        ('A' <= aCh && aCh <='F');
}

//------------------------------------------------------------------------------
void CString::Copy(char* aDst, const char* aSrc, int aCap)
{
    const bool isAllCopied = CopyAsPossible(aDst, aSrc, aCap);
    SYS_ASSERT(isAllCopied);
}

//------------------------------------------------------------------------------
bool CString::CopyAsPossible(char* aDst, const char* aSrc, int aCap)
{
    SYS_ASSERT_POINTER(aDst);
    SYS_ASSERT_POINTER(aSrc);
    ::std::strncpy(aDst, aSrc, aCap);
    if (aCap > 0) {
        aDst[aCap - 1] = '\0';
    }
    // 何かしらの切り落としが発生した場合は失敗とする
    return GetLength(aSrc, aCap) < aCap;
}

//------------------------------------------------------------------------------
bool CString::StartsWith(const char* aStr, const char* aTargetStr, int aCap)
{
    SYS_ASSERT_POINTER(aStr);
    SYS_ASSERT_POINTER(aTargetStr);
    for (int i = 0; i < aCap; ++i) {
        if (aTargetStr[i] == 0 && aStr[i] == 0) {
            return true;
        }
        if (aTargetStr[i] != aStr[i]) {
            return aTargetStr[i] == 0;
        }
    }
    // ここに来たということは「aTargetStr」を最後まで比較できていないので、
    // 意図しない結果となる可能性がある為アサートする
    SYS_DEBUG_ASSERT_NOT_REACHED();
    return false;
}

//------------------------------------------------------------------------------
bool CString::Contains(const char* aStr, const char* aSubStr, bool aWillIgnoreCase)
{
    SYS_ASSERT_POINTER(aStr);
    SYS_ASSERT_POINTER(aSubStr);
    if (aSubStr[0] == 0) {
        return true;
    }
    for (int i = 0; aStr[i]; ++i) {
        for (int j = 0; aSubStr[j] && aStr[i + j]; ++j) {
            char ch = aStr[i + j];
            char subCh = aSubStr[j];
            if (aWillIgnoreCase) {
                ch = static_cast<char>(::std::tolower(ch));
                subCh = static_cast<char>(::std::tolower(subCh));
            }
            if (ch != subCh) {
                break;
            }
            if (aSubStr[j + 1] == 0) {
                return true;
            }
        }
    }
    return false;
}

//------------------------------------------------------------------------------
void CString::Format(char* aStr, int aCap, const char* aFmt, ...)
{
    ::std::va_list vaList = { 0 };
    va_start(vaList, aFmt);
    FormatV(aStr, aCap, aFmt, vaList);
    va_end(vaList);
} //lint !e438

//------------------------------------------------------------------------------
// ・注意点は「FormatVAsPossible」を参照
void CString::FormatV(char* aStr, int aCap, const char* aFmt, ::std::va_list aVaList)
{
    SYS_ASSERT(aCap > 0);
    int len = vsnprintf(aStr, aCap, aFmt, aVaList);
    // ・「FormatVAsPossible」のコメントを参考にする
#if DEBUG_IS_ENABLED
    SYS_DEBUG_ASSERT(len >= 0 && len < aCap);
#else
    // Warning抑止
    LIB_UNUSED(len);
#endif
    // フェイルセーフする
    // ・「FormatVAsPossible」と同等
    aStr[aCap - 1] = 0;
}

//------------------------------------------------------------------------------
void CString::FormatAsPossible(char* aStr, int aCap, const char* aFmt, ...)
{
    ::std::va_list vaList = { 0 };
    va_start(vaList, aFmt);
    FormatVAsPossible(aStr, aCap, aFmt, vaList);
    va_end(vaList);
} //lint !e438

//------------------------------------------------------------------------------
void CString::FormatVAsPossible(char* aStr, int aCap, const char* aFmt, ::std::va_list aVaList)
{
    SYS_ASSERT(aCap > 0);
    vsnprintf(aStr, aCap, aFmt, aVaList);
    // 以下のような環境での違いを考慮しなくて良いように、必ずnullptr終端を書き込む
    // ・VC2013
    //   ・文字が溢れた場合「-1」が返り、書き込まれた文字列はNULL終端にならない
    //   ・NULL終端のみが溢れた場合文字列長つまり「aCap」が返り、書き込まれた文字列はNULL終端にならない
    //     ・「-1」が返らない点に注意
    // ・Clang
    //   ・容量溢れが無かったとした場合のフォーマット後の文字列長が返る
    //   ・実際に書き込まれる文字列は、指定した容量に収まるNULL終端込みの文字列になる
    aStr[aCap - 1] = 0;
}

} // namespace
// EOF
