﻿/*---------------------------------------------------------------------------*
  Copyright (C)2015 Nintendo Co., Ltd.  All rights reserved.

  These coded instructions, statements, and computer programs contain
  proprietary information of Nintendo of America Inc. and/or Nintendo
  Company Ltd., and are protected by Federal copyright law.  They may
  not be disclosed to third parties or copied or duplicated in any form,
  in whole or in part, without the prior written consent of Nintendo.
 *---------------------------------------------------------------------------*/

#include <nn/os.h>
#include <nn/nn_Abort.h>
#include <nn/nn_SdkLog.h>
#include <nn/os/os_SdkThreadCommon.h>
#include <nn/nn_SystemThreadDefinition.h>
extern "C" {

#include <sys/types.h>
#include <sys/param.h>
#include <sys/systm.h>
#include <sys/malloc.h>
#include <sys/kernel.h>
#include <sys/mbuf.h>
#include <sys/domain.h>
#include <sys/protosw.h>
#include <sys/eventhandler.h>
#include <vm/uma.h>
#include <vm/uma_int.h>
#include <vm/uma_dbg.h>
#include <siglo/uma.h>
#include <siglo/thread.h>

uma_zone_t zone_mbuf;
uma_zone_t zone_pack;
uma_zone_t zone_clust;
uma_zone_t zone_jumbop;
uma_zone_t zone_ext_refcnt;
uma_zone_t zone_jumbo9;
uma_zone_t zone_jumbo16;

static const   int BOOT_PAGE_COUNT = 10;
static uint8_t bootmem[UMA_SLAB_SIZE * BOOT_PAGE_COUNT] __attribute__((aligned(PAGE_SIZE)));

static NetworkThread uma_reclaim_thread;
static const int uma_reclaim_sleep_seconds = 1;        // time between periodic reclaims.  In seconds.
static uint8_t uma_reclaim_thread_stack[1024 * 16] __attribute__((aligned(nn::os::StackRegionAlignment)));
/* mbuf and attached storage each has a zone marker
 * which is used to release that chunk into appropriate zone.
 * By default, all allocations are done from the net process
 * default zones. If allocations are done from any other pool,
 * allocating clients should mark zone in allocated mbuf
 * appropriately to ensure it will be released back to
 * the correct pool.
 */

static int mb_ctor_mbuf(void *mem, int size, void *arg, int how)
{
    struct mbuf *m;
    struct mb_args *args;
    int error;
    int flags;
    short type;
    args = (struct mb_args *)arg;
    type = args->type;
    /*
     * The mbuf is initialized later.  The caller has the
     * responsibility to set up any MAC labels too.
     */
    if (type == MT_NOINIT) {
        return 0;
    }
    m = (struct mbuf *)mem;
    flags = args->flags;
    error = m_init(m, NULL, size, how, type, flags);
    m->m_zone = &proc0->p_zones;
    return (error);
}

static void mb_dtor_mbuf(void *mem, int size, void *arg)
{
    struct mbuf *m;
    m = (struct mbuf *)mem;
    if ((m->m_flags & M_PKTHDR) && !SLIST_EMPTY(&m->m_pkthdr.tags)) {
        m_tag_delete_chain(m, NULL);
    }
    KASSERT((m->m_flags & M_EXT) == 0, ("%s: M_EXT set", __func__));
    KASSERT((m->m_flags & M_NOFREE) == 0, ("%s: M_NOFREE set", __func__));
}

static int mb_ctor_clust(void *mem, int size, void *arg, int how)
{
    struct uma_client_zones *zones;
    struct mbuf *m;
    uma_zone_t zone;
    u_int *refcnt;
    int type;
    m = (struct mbuf *)arg;
    zones = m ? m->m_zone : &proc0->p_zones;
    switch (size) {
    case MCLBYTES:
        type = EXT_CLUSTER;
        zone = zones->zone_clust;
        break;
#if MJUMPAGESIZE != MCLBYTES
    case MJUMPAGESIZE:
        type = EXT_JUMBOP;
        zone = zones->zone_jumbop;
        break;
#endif
    case MJUM9BYTES:
        type = EXT_JUMBO9;
        zone = zones->zone_jumbo9;
        break;
    case MJUM16BYTES:
        type = EXT_JUMBO16;
        zone = zones->zone_jumbo16;
        break;
    default:
        panic("unknown cluster size");
        break;
    }
    refcnt = uma_find_refcnt(zone, mem);
    *refcnt = 1;
    if (m != NULL) {
        m->m_ext.ext_buf = (caddr_t)mem;
        m->m_data = m->m_ext.ext_buf;
        m->m_flags |= M_EXT;
        m->m_ext.ext_free = NULL;
        m->m_ext.ext_arg1 = NULL;
        m->m_ext.ext_arg2 = NULL;
        m->m_ext.ext_size = size;
        m->m_ext.ext_type = type;
        m->m_ext.ext_flags = 0;
        m->m_ext.ref_cnt  = refcnt;
        m->m_ext.ext_zone = zones;
    }
    return (0);
}

static void mb_dtor_clust(void *mem, int size, void *arg)
{
    return;
}

static int mb_ctor_pack(void *mem, int size, void *arg, int how)
{
    struct mbuf *m;
    struct mb_args *args;
    int error, flags;
    short type;
    m = (struct mbuf *)mem;
    args = (struct mb_args *)arg;
    flags = args->flags;
    type = args->type;
    error = m_init(m, NULL, size, how, type, flags);
    /* m_ext is already initialized. */
    m->m_data  = m->m_ext.ext_buf;
    m->m_flags = (flags | M_EXT);
    m->m_zone  = &proc0->p_zones;
    return (error);
}

static void mb_dtor_pack(void *mem, int size, void *arg)
{
    struct mbuf *m;
    m = (struct mbuf *)mem;
    if ((m->m_flags & M_PKTHDR) != 0) {
        m_tag_delete_chain(m, NULL);
    }
    /* Make sure we've got a clean cluster back. */
    KASSERT((m->m_flags & M_EXT) == M_EXT, ("%s: M_EXT not set", __func__));
    KASSERT(m->m_ext.ext_buf != NULL, ("%s: ext_buf == NULL", __func__));
    KASSERT(m->m_ext.ext_free == NULL, ("%s: ext_free != NULL", __func__));
    KASSERT(m->m_ext.ext_arg1 == NULL, ("%s: ext_arg1 != NULL", __func__));
    KASSERT(m->m_ext.ext_arg2 == NULL, ("%s: ext_arg2 != NULL", __func__));
    KASSERT(m->m_ext.ext_size == MCLBYTES, ("%s: ext_size != MCLBYTES", __func__));
    KASSERT(m->m_ext.ext_type == EXT_PACKET, ("%s: ext_type != EXT_PACKET", __func__));
    KASSERT(*m->m_ext.ref_cnt == 1, ("%s: ref_cnt != 1", __func__));
    /*
     * If there are processes blocked on zone_clust, waiting for pages
     * to be freed up, * cause them to be woken up by draining the
     * packet zone.  We are exposed to a race here * (in the check for
     * the UMA_ZFLAG_FULL) where we might miss the flag set, but that
     * is deliberate. We don't want to acquire the zone lock for every
     * mbuf free.
     */
    if (uma_zone_exhausted_nolock(zone_clust)) {
        zone_drain(zone_pack);
    }
}

int m_pkthdr_init(struct mbuf *m, int how)
{
#ifdef MAC
    int error;
#endif
    m->m_data = m->m_pktdat;
    m->m_pkthdr.rcvif = NULL;
    SLIST_INIT(&m->m_pkthdr.tags);
    m->m_pkthdr.len = 0;
    m->m_pkthdr.flowid = 0;
    m->m_pkthdr.csum_flags = 0;
    m->m_pkthdr.fibnum = 0;
    m->m_pkthdr.cosqos = 0;
    m->m_pkthdr.rsstype = 0;
    m->m_pkthdr.l2hlen = 0;
    m->m_pkthdr.l3hlen = 0;
    m->m_pkthdr.l4hlen = 0;
    m->m_pkthdr.l5hlen = 0;
    m->m_pkthdr.PH_per.sixtyfour[0] = 0;
    m->m_pkthdr.PH_loc.sixtyfour[0] = 0;
#ifdef MAC
    /* If the label init fails, fail the alloc */
    error = mac_mbuf_init(m, how);
    if (error) {
        return (error);
    }
#endif
    return (0);
}

static int mb_zinit_pack(void *mem, int size, int how)
{
    struct mbuf *m;
    m = (struct mbuf *)mem;         /* m is virgin. */
    m->m_zone = &proc0->p_zones;
    if (uma_zalloc_arg(zone_clust, m, how) == NULL ||
        m->m_ext.ext_buf == NULL)
    {
        return (ENOMEM);
    }
    m->m_ext.ext_type = EXT_PACKET; /* Override. */
    return 0;
}

static void mb_zfini_pack(void *mem, int size)
{
    struct mbuf *m;
    m = (struct mbuf *)mem;
    uma_zfree_arg((m->m_ext.ext_zone)->zone_clust, m->m_ext.ext_buf, NULL);
}

static void mb_reclaim(void *junk)
{
    struct domain *dp;
    struct protosw *pr;

    for (dp = domains; dp != NULL; dp = dp->dom_next) {
        for (pr = dp->dom_protosw; pr < dp->dom_protoswNPROTOSW; pr++) {
            if (pr->pr_drain != NULL) {
                (*pr->pr_drain)();
            }
        }
    }
}

static void mbuf_init(void* dummy)
{
    proc0->p_zones.zone_mbuf = zone_mbuf = uma_zcreate(
                    MBUF_MEM_NAME, MSIZE,
                    mb_ctor_mbuf, mb_dtor_mbuf,
                    NULL, NULL,
                    MSIZE - 1, UMA_ZONE_MAXBUCKET
                );


    proc0->p_zones.zone_pack = zone_pack = uma_zsecond_create(
                    MBUF_PACKET_MEM_NAME,
                    mb_ctor_pack,  mb_dtor_pack,
                    mb_zinit_pack, mb_zfini_pack,
                    zone_mbuf
                );


    proc0->p_zones.zone_clust = zone_clust = uma_zcreate(
                    MBUF_CLUSTER_MEM_NAME, MCLBYTES,
                    mb_ctor_clust, mb_dtor_clust,
                    NULL, NULL,
                    UMA_ALIGN_PTR, UMA_ZONE_REFCNT
                );

    proc0->p_zones.zone_jumbop = zone_jumbop = uma_zcreate(
                    MBUF_JUMBOP_MEM_NAME, MJUMPAGESIZE,
                    mb_ctor_clust, mb_dtor_clust,
                    NULL, NULL,
                    UMA_ALIGN_PTR, UMA_ZONE_REFCNT
                );

    zone_ext_refcnt = uma_zcreate(
                    MBUF_EXTREFCNT_MEM_NAME, sizeof(u_int),
                    NULL, NULL,
                    NULL, NULL,
                    UMA_ALIGN_PTR, UMA_ZONE_ZINIT
                );

    proc0->p_zones.zone_jumbo9 = zone_jumbo9 = uma_zcreate(
                    MBUF_JUMBO9_MEM_NAME, MJUM9BYTES,
                    mb_ctor_clust, mb_dtor_clust,
                    NULL, NULL,
                    UMA_ALIGN_PTR, UMA_ZONE_REFCNT
                );


    proc0->p_zones.zone_jumbo16 = zone_jumbo16 = uma_zcreate(
                    MBUF_JUMBO16_MEM_NAME, MJUM16BYTES,
                    mb_ctor_clust, mb_dtor_clust,
                    NULL, NULL,
                    UMA_ALIGN_PTR, UMA_ZONE_REFCNT
                );

    uma_zone_set_owner(proc0->p_zones.zone_mbuf,    proc0);
    uma_zone_set_owner(proc0->p_zones.zone_pack,    proc0);
    uma_zone_set_owner(proc0->p_zones.zone_clust,   proc0);
    uma_zone_set_owner(proc0->p_zones.zone_jumbop,  proc0);
    uma_zone_set_owner(proc0->p_zones.zone_jumbo9,  proc0);
    uma_zone_set_owner(proc0->p_zones.zone_jumbo16, proc0);

    if (nmbufs > 0) {
        nmbufs = uma_zone_set_max(zone_mbuf, nmbufs);
    }
    if (nmbclusters > 0) {
        nmbclusters = uma_zone_set_max(zone_clust, nmbclusters);
    }
    if (nmbjumbop > 0) {
        nmbjumbop = uma_zone_set_max(zone_jumbop, nmbjumbop);
    }
    if (nmbjumbo9 > 0) {
        nmbjumbo9 = uma_zone_set_max(zone_jumbo9, nmbjumbo9);
    }
    if (nmbjumbo16 > 0) {
        nmbjumbo16 = uma_zone_set_max(zone_jumbo16, nmbjumbo16);
    }

    EVENTHANDLER_REGISTER(vm_lowmem, (void*)mb_reclaim, NULL, EVENTHANDLER_PRI_FIRST);

    return;
}

SYSINIT(mbuf, SI_SUB_MBUF, SI_ORDER_FIRST, mbuf_init, NULL);

static void uma_reclaim_thread_entry(void* dummy)
{
    while (true)
    {
        uma_reclaim();
        nn::os::SleepThread(nn::TimeSpan::FromSeconds(uma_reclaim_sleep_seconds));
    }
}

static void uma_sysinit(void* arg)
{
    uma_startup(bootmem, sizeof(bootmem) / UMA_SLAB_SIZE);
    uma_startup2();

    NN_ABORT_UNLESS(CreateNetworkThread(
        &uma_reclaim_thread,
        &uma_reclaim_thread_entry,
        NULL,
        uma_reclaim_thread_stack,
        sizeof(uma_reclaim_thread_stack),
        NN_SYSTEM_THREAD_PRIORITY(socket, UmaReclaim),
        0) == 0);

    SetNetworkThreadNamePointer(&uma_reclaim_thread, NN_SYSTEM_THREAD_NAME(socket, UmaReclaim));

    StartNetworkThread(&uma_reclaim_thread);
}

SYSINIT(uma_sysinit, SI_SUB_KMEM, SI_ORDER_FIRST,  uma_sysinit, NULL);

void create_zone_allocators(struct proc* proc, size_t mem_size)
{
    int cl_nmbclusters;
    int cl_nmbjumbop;
    int cl_nmbjumbo9;
    int cl_nmbjumbo16;
    int cl_nmbufs;

    proc->p_zones.zone_mbuf = uma_zcreate(
                                MBUF_MEM_NAME, MSIZE,
                                mb_ctor_mbuf, mb_dtor_mbuf,
                                NULL, NULL,
                                MSIZE - 1, UMA_ZONE_MAXBUCKET
                            );

    proc->p_zones.zone_clust = uma_zcreate(
                                MBUF_CLUSTER_MEM_NAME, MCLBYTES,
                                mb_ctor_clust, mb_dtor_clust,
                                NULL, NULL,
                                UMA_ALIGN_PTR, UMA_ZONE_REFCNT
                            );

    proc->p_zones.zone_jumbop = uma_zcreate(
                                MBUF_JUMBOP_MEM_NAME, MJUMPAGESIZE,
                                mb_ctor_clust, mb_dtor_clust,
                                NULL, NULL,
                                UMA_ALIGN_PTR, UMA_ZONE_REFCNT
                            );

    proc->p_zones.zone_jumbo9 = uma_zcreate(
                                MBUF_JUMBO9_MEM_NAME, MJUM9BYTES,
                                mb_ctor_clust, mb_dtor_clust,
                                NULL, NULL,
                                UMA_ALIGN_PTR, UMA_ZONE_REFCNT
                            );

    proc->p_zones.zone_jumbo16 = uma_zcreate(
                                MBUF_JUMBO16_MEM_NAME, MJUM16BYTES,
                                mb_ctor_clust, mb_dtor_clust,
                                NULL, NULL,
                                UMA_ALIGN_PTR, UMA_ZONE_REFCNT
                            );

    uma_zone_set_owner(proc->p_zones.zone_mbuf,    proc);
    uma_zone_set_owner(proc->p_zones.zone_clust,   proc);
    uma_zone_set_owner(proc->p_zones.zone_jumbop,  proc);
    uma_zone_set_owner(proc->p_zones.zone_jumbo9,  proc);
    uma_zone_set_owner(proc->p_zones.zone_jumbo16, proc);


    /*
     * Set limits for this client's uma zones
     */
#ifdef __SigloBSD__
    cl_nmbclusters = (mem_size / MCLBYTES) * 5 / 12;
    cl_nmbjumbop   = (mem_size / MJUMPAGESIZE) * 5 / 12;
#else
    cl_nmbclusters = mem_size / MCLBYTES/ 4;
    cl_nmbjumbop   = mem_size / MJUMPAGESIZE / 4;
#endif
    cl_nmbjumbo9   = mem_size / MJUM9BYTES / 6;
    cl_nmbjumbo16  = mem_size / MJUM16BYTES / 6;
    cl_nmbufs      = lmax(
                        mem_size / MSIZE / 5,
                        cl_nmbclusters + cl_nmbjumbop + cl_nmbjumbo9 + cl_nmbjumbo16
                     );

    uma_zone_set_max(proc->p_zones.zone_mbuf,    cl_nmbufs);
    uma_zone_set_max(proc->p_zones.zone_clust,   cl_nmbclusters);
    uma_zone_set_max(proc->p_zones.zone_jumbop,  cl_nmbjumbop);
    uma_zone_set_max(proc->p_zones.zone_jumbo9,  cl_nmbjumbo9);
    uma_zone_set_max(proc->p_zones.zone_jumbo16, cl_nmbjumbo16);

    proc->p_zones.zone_flags = 0;
}

void destroy_zone_allocators(struct proc* proc)
{
    /* if one of these has non-released items, uma will panic */
    uma_zdestroy(proc->p_zones.zone_jumbop);
    uma_zdestroy(proc->p_zones.zone_jumbo9);
    uma_zdestroy(proc->p_zones.zone_jumbo16);
    uma_zdestroy(proc->p_zones.zone_clust);
    uma_zdestroy(proc->p_zones.zone_mbuf);
    proc->p_zones.zone_jumbop  = NULL;
    proc->p_zones.zone_jumbo9  = NULL;
    proc->p_zones.zone_jumbo16 = NULL;
    proc->p_zones.zone_clust   = NULL;
    proc->p_zones.zone_mbuf    = NULL;
}

int get_zone_allocator_count(struct proc* proc)
{
    int count = 0;
    count += uma_zone_get_cur(proc->p_zones.zone_jumbop);
    count += uma_zone_get_cur(proc->p_zones.zone_jumbo9);
    count += uma_zone_get_cur(proc->p_zones.zone_jumbo16);
    count += uma_zone_get_cur(proc->p_zones.zone_clust);
    count += uma_zone_get_cur(proc->p_zones.zone_mbuf);
    return count;
}

}
