﻿/*---------------------------------------------------------------------------*
  Copyright (C)2015 Nintendo Co., Ltd.  All rights reserved.

  These coded instructions, statements, and computer programs contain
  proprietary information of Nintendo of America Inc. and/or Nintendo
  Company Ltd., and are protected by Federal copyright law.  They may
  not be disclosed to third parties or copied or duplicated in any form,
  in whole or in part, without the prior written consent of Nintendo.
 *---------------------------------------------------------------------------*/

#include <nn/os.h>
#include <nn/nn_Abort.h>
#include <nn/nn_SdkLog.h>

extern "C" {

#include <sys/types.h>
#include <sys/param.h>
#include <sys/systm.h>
#include <sys/kernel.h>
#include <sys/proc.h>
#include <sys/smp.h>
#include <vm/uma.h>
#include <siglo/thread.h>
#include <siglo/client.h>

#define _WANT_NETISR_INTERNAL
#include <net/netisr.h>
#include <net/netisr_internal.h>

/* main memory pool */
extern size_t   mempool_size;
extern uint8_t* mempool;

struct proc*   proc0;
struct client* client0;
struct thread  thread0;
static nn::os::ThreadType* _mainthread = NULL;

struct pcpu pcpus[MAXCPU];
int    mp_ncpus      = MAXCPU;
u_int  mp_maxid      = MAXCPU - 1;
u_int  mp_activecpus = SMP_CPU_MASK;
struct netisr_workstream nws[MAXCPU];
struct cpuhead cpuhead = STAILQ_HEAD_INITIALIZER(cpuhead);
uma_zone_t pcpu_zone_64;

struct proc* get_current_proc()
{
    return get_current_thread()->td_proc;
}

int get_current_cpu()
{
    return nn::os::GetCurrentProcessorNumber();
}

struct thread* get_current_thread()
{
    nn::os::ThreadType* siglotd = nn::os::GetCurrentThread();
    return (siglotd == _mainthread) ? &thread0 : BSD_THREAD_FROM_SIGLO(siglotd);
}

static void proc_sysinit(void* dummy)
{
    int i;

    /* associate main init thread with bsd's thread0 (main proc thread) */
    _mainthread = nn::os::GetCurrentThread();

    /* create main process, mempool_size is used to compute proc0 rx/tx limits */
    client_create(DEFAULT_PID, mempool, mempool_size, NULL, NULL);
    client0 = client_acquire_ref(DEFAULT_PID);
    proc0   = &(client0->cl_proc);

    /* main thread belongs to this process */
    thread0.td_ucred = &(client0->cl_ucred);
    thread0.td_proc  = &(client0->cl_proc);

    /* main processor struct */
    bzero((void*)&pcpus[0], sizeof(pcpus[0]));
    pcpus[0].pc_cpuid = 0;

    /* processor structs for additional cpus */
    for (i = 1; i < MAXCPU; i++) {
        if (!CPU_ABSENT(i)) {
            bzero((void*)&pcpus[i], sizeof(pcpus[i]));
            pcpus[i].pc_cpuid = i;
            /* netisr will start swi thread for each core on the list */
            STAILQ_INSERT_TAIL(&cpuhead, &pcpus[i], pc_allcpu);
        }
    }

    return;
}

SYSINIT(proc_sysinit, SI_SUB_PORTINIT, SI_ORDER_THIRD, proc_sysinit, NULL);

static void pcpu_zoneinit(void)
{
    pcpu_zone_64 = uma_zcreate(
                        "pcpu_zone64", sizeof(uint64_t),
                        NULL, NULL,
                        NULL, NULL,
                        UMA_ALIGN_PTR, UMA_ZONE_PCPU
                    );
    return;
}

SYSINIT(pcpu_zoneinit, SI_SUB_KMEM, SI_ORDER_ANY, pcpu_zoneinit, NULL);

FORCE_LINK_THIS(_proc_);

}

