﻿/*---------------------------------------------------------------------------*
  Copyright (C)2015 Nintendo Co., Ltd.  All rights reserved.

  These coded instructions, statements, and computer programs contain
  proprietary information of Nintendo of America Inc. and/or Nintendo
  Company Ltd., and are protected by Federal copyright law.  They may
  not be disclosed to third parties or copied or duplicated in any form,
  in whole or in part, without the prior written consent of Nintendo.
 *---------------------------------------------------------------------------*/
#include <sys/types.h>
#include <sys/param.h>
#include <sys/mbuf.h>

#include <net/ethernet.h>
#include <net/if.h>
#include <net/if_var.h>
#include <net/if_types.h>
#include <net/if_media.h>
#include <siglo/network.h>

#include <drivers/eth/eth_if_drv.h>


static struct ifnet             *__ifp = NULL;
static int                      __tx_pend_count = 0;
static const unsigned char      *__p_macaddress = NULL;
static int                      __link_state = LINK_STATE_DOWN;

#define TX_QUEUE_LENGTH         32
#define RX_QUEUE_LENGTH         32
#define MAX_SEGMENTS_PER_MBUF   64
#define TX_MAX_PENDING          16

typedef struct
{
    struct callout co;
    bool   link_up;
}eth_if_linkstate_callout_t;

typedef struct
{
    struct       callout co;
    const void   *pFrame;
    size_t       size;
}eth_if_receive_frame_callout_t;

/*
 * BSD Stack Callouts
 */
static int __ioctl(struct ifnet *ifp, u_long command, caddr_t data)
{
    struct ifreq *ifr = (struct ifreq *)data;
    int error = 0;
    switch (command)
    {
    case SIOCSIFMTU:
        ETH_IF_DRV_LOG_INFO("SIOCSIFMTU ifr_mtu = %d\n", ifr->ifr_mtu);
        ifp->if_mtu = ifr->ifr_mtu;
        break;
    case SIOCSIFFLAGS:
        if (ifp->if_flags & IFF_UP)
        {
            if ( !(ifp->if_drv_flags & IFF_DRV_RUNNING))
            {
                ifp->if_drv_flags &= ~(IFF_DRV_OACTIVE);
                ifp->if_drv_flags |= IFF_DRV_RUNNING;
            }
        }
        else
        {
            if (ifp->if_drv_flags & IFF_DRV_RUNNING)
            {
                ifp->if_drv_flags &= ~(IFF_DRV_RUNNING | IFF_DRV_OACTIVE);
            }
        }
        break;
    case SIOCADDMULTI:
        //ETH_IF_DRV_LOG_INFO("SIOCADDMULTI\n");
        break;
    case SIOCDELMULTI:
        //ETH_IF_DRV_LOG_INFO("SIOCDELMULTI\n");
        break;
    case SIOCGIFMEDIA:
        {
            struct ifmediareq *p_ifmr = (struct ifmediareq *)data;
            p_ifmr->ifm_status = IFM_AVALID | ((__link_state == LINK_STATE_UP) ? IFM_ACTIVE : 0);
            p_ifmr->ifm_active = IFM_ETHER;
            p_ifmr->ifm_count  = 0;
            p_ifmr->ifm_ulist  = NULL;
            break;
        }
    case SIOCSIFMEDIA:
        ETH_IF_DRV_LOG_INFO("SIOCSIFMEDIA\n");
        break;
    case SIOCSIFCAP:
        ETH_IF_DRV_LOG_INFO("SIOCSIFCAP if_capenable=0x%x\n", ifp->if_capenable);
        break;
    default:
        error = ether_ioctl(ifp, command, data);
        break;
    }
    return (error);
}

static void __start(struct ifnet *ifp)
{
    size_t segment_sizes[MAX_SEGMENTS_PER_MBUF];
    char *segment_buffers[MAX_SEGMENTS_PER_MBUF];

    if ((ifp->if_drv_flags & (IFF_DRV_RUNNING | IFF_DRV_OACTIVE)) !=
            IFF_DRV_RUNNING)
    {
        return;
    }

    // while the transmit queue is not overflowing
    while ( !(ifp->if_drv_flags & IFF_DRV_OACTIVE))
    {
        int segments;
        struct mbuf *m_n;
        struct mbuf *m_head = NULL;
        IFQ_DRV_DEQUEUE(&ifp->if_snd, m_head);
        if (m_head == NULL) break;
        m_n = m_head;
        for (segments = 0; (segments < MAX_SEGMENTS_PER_MBUF) && (m_n != NULL); segments++)
        {
            segment_buffers[segments] = mtod(m_n, char *);
            segment_sizes[segments] = m_n->m_len;
            m_n = m_n->m_next;
        }
        if (segments > 0)
        {
            if (eth_if_drv_send(segments, (const unsigned char **)segment_buffers, segment_sizes))
            {
                if (__sync_fetch_and_add(&__tx_pend_count, 1) >= TX_MAX_PENDING)
                {
                    ifp->if_drv_flags |= IFF_DRV_OACTIVE;
                }
            }
        }
        m_freem(m_head);
    }
}

static void __init(void *xsc)
{

}

static void __handle_linkstate_callout(void *arg)
{
    eth_if_linkstate_callout_t *p_c = (eth_if_linkstate_callout_t *)arg;
    if (p_c != NULL)
    {
        int link_state = LINK_STATE_DOWN;
        void (*if_callout)(struct ifnet *) = if_down;
        if (p_c->link_up)
        {
            link_state = LINK_STATE_UP;
            if_callout = if_up;
        }
        if_link_state_change(__ifp, link_state);
        (*if_callout)(__ifp);
        free(p_c, M_DEVBUF);
        __link_state = link_state;
    }
}

/*
 * These API are called from c++ side of eth_if_drv
 */
int eth_if_drv_attach(const unsigned char *macaddress)
{
    int rv = 0;
    struct ifnet *ifp = NULL;
    if (__ifp != NULL)
    {
        return -EALREADY;
    }
    if ((__ifp = ifp = if_alloc(IFT_ETHER)) == NULL)
    {
        return -ENOMEM;
    }

    __tx_pend_count    = 0;
    __p_macaddress     = macaddress;

    if_initname(__ifp, "eth", 0);
    ifp->if_flags = IFF_BROADCAST | IFF_SIMPLEX | IFF_MULTICAST;
    ifp->if_ioctl        = __ioctl;
    ifp->if_start        = __start;
    ifp->if_hwassist     = 0;
    ifp->if_capabilities = 0;
    ifp->if_capenable    = 0;
    ifp->if_init         = __init;
    ifp->if_drv_flags    |= IFF_DRV_RUNNING;
    ifp->if_drv_flags    &= ~IFF_DRV_OACTIVE;
    IFQ_SET_MAXLEN(&ifp->if_snd, TX_QUEUE_LENGTH);
    ifp->if_snd.ifq_drv_maxlen = RX_QUEUE_LENGTH;
    IFQ_SET_READY(&ifp->if_snd);

    /* attach the interface */
    ether_ifattach(ifp, macaddress);

    return rv;
}

int eth_if_drv_detach()
{
    int rv = 0;
    struct ifnet *ifp = __ifp;
    if (ifp == NULL)
    {
        return -EPERM;
    }
    ifp->if_flags &= ~IFF_UP;
    ether_ifdetach(ifp);
    if_free(__ifp);
    __ifp = NULL;
    return rv;
}

void eth_if_drv_linkup()
{
    eth_if_linkstate_callout_t *p_c = malloc(sizeof(eth_if_linkstate_callout_t), M_DEVBUF, 0);
    if (p_c != NULL)
    {
        callout_init(&p_c->co, CALLOUT_MPSAFE);
        p_c->co.c_arg  = p_c;
        p_c->co.c_func = __handle_linkstate_callout;
        p_c->link_up   = true;
        callout_schedule(&p_c->co, 0);
    }
}

void eth_if_drv_linkdown()
{
    if_down(__ifp);
    eth_if_linkstate_callout_t *p_c = malloc(sizeof(eth_if_linkstate_callout_t), M_DEVBUF, 0);
    if (p_c != NULL)
    {
        callout_init(&p_c->co, CALLOUT_MPSAFE);
        p_c->co.c_arg  = p_c;
        p_c->co.c_func = __handle_linkstate_callout;
        p_c->link_up   = false;
        callout_schedule(&p_c->co, 0);
    }
}

void eth_if_drv_send_complete()
{
    /* update overflow logic */
    if (__sync_fetch_and_sub(&__tx_pend_count, 1) == TX_MAX_PENDING)
    {
        __ifp->if_drv_flags &= ~IFF_DRV_OACTIVE;
    }
}

void eth_if_drv_receive_frame(const void *frame, size_t size)
{
    struct mbuf *m_head = m_devget((void *)frame, size, 2, __ifp, NULL);
    if (m_head != NULL)
    {
        (*__ifp->if_input)(__ifp, m_head);
    }
}

/*
 * Stubs for interface related calls (normally in FreeBSD kernel)
 */
int uuid_ether_add(const uint8_t *addr)
{
    //ETH_IF_DRV_LOG_INFO("uuid_ether_add addr=%p %p\n", addr, *addr);
    return 0;
}

int uuid_ether_del(const uint8_t *addr)
{
    //ETH_IF_DRV_LOG_INFO("uuid_ether_del addr=%p %p\n", addr, *addr);
    return 0;
}

int vprintf(const char *template, va_list ap)
{
        //ETH_IF_DRV_LOG_INFO("vprintf - %s\n", template);
        return 0;
}


