﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstring>
#include <cstdint>
#include <cstddef>
#include <cstdlib>
#include <new>
#include <nn/fs.h>
#include <pthread.h>
#include <nn/nn_Macro.h>
#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nn/os.h>
#include <nn/ro.h>
#include <nn/svc/svc_ThreadLocalRegion.h>
#include <nn/os/os_SdkMemoryAllocatorForThreadLocal.h>
#include <climits>
#include <nn/os/os_ThreadApi.h>
#include "ntd-test-tls.h"

extern void TestSecondFile(void);
extern void *__dso_handle;  // this is defined in Chris/Sources/Libraries/rocrt/rocrt.cpp

#define TEST_BROKEN_IMPORTED_TLS 0

#define MIN_ALIGN nn::os::ThreadStackAlignment

#define STACK_SIZE (3 * MIN_ALIGN)

const size_t            ThreadStackSize = STACK_SIZE;                 // スレッド操作スレッドのスタックサイズ
static NN_OS_ALIGNAS_THREAD_STACK char  g_ThreadStack1[ ThreadStackSize ];   // 1つ目のスレッドのスタック

static nn::os::ThreadType      g_Thread1;
static int horizon_max_num_modules;

thread_local short          GlobalShortFirstFile;
thread_local int            GlobalIntFirstFile = 100;
static thread_local char    StaticChar = 'f';
static thread_local char    *StaticPointer = (char *)"FirstFile";
__attribute__((weak)) thread_local int WeakTlsFirstStrongTlsSecond = 1234;
thread_local int StrongTlsFirstWeakTlsSecond = 4321;
thread_local int DuplicateNameTLS = 11111;
extern __attribute__((aligned(TLS_ALIGNMENT))) thread_local double  GlobalDoubleSecondFile;
extern thread_local float   GlobalFloatSecondFile;

short          NonTlsGlobalShortFirstFile;
int            NonTlsGlobalIntFirstFile = 100;
extern double  NonTlsGlobalDoubleSecondFile;
extern float   NonTlsGlobalFloatSecondFile;

static size_t tdata_align_abs, tbss_align_abs, tdata_align_rel, tbss_align_rel, tls_segment_align, tls_segment_size;
static size_t tdata_size, tbss_size, tdata_offset, tbss_offset;

#define TEST_ROOT_NAME STRINGIZE_VALUE_OF(TLS_TEST_NAME)
#define NRO_ROOT_NAME STRINGIZE_VALUE_OF(TLS_NRO_NAME)
#if ULONG_MAX == 0xffffffff
#define TPOFF_K 8
#define TEST_NAME "ARM32-" TEST_ROOT_NAME
#else
#define TPOFF_K 16
#define TEST_NAME "AARCH64-" TEST_ROOT_NAME
#endif
#define NRR_NAME "rom:/.nrr/" TEST_ROOT_NAME ".nrr"
#define NR0_NAME "rom:/nro/" NRO_ROOT_NAME ".nro"

typedef struct type_tls_data {
    void **dtv;
    void *tls_allocation;
    size_t tls_allocation_size;
    void *tls_dtor_list;
    void **dtv_copy;
} tls_data;

#define GET_TLS_DATA(TP)	((type_tls_data*)(TP + TPOFF_K - sizeof(type_tls_data)))

static size_t ReadAll(void* pOut, size_t bufferSize, const char* path)
{
    nn::Result result;
    nn::fs::FileHandle file;
    result = nn::fs::OpenFile(&file, path, nn::fs::OpenMode_Read);
    NN_ASSERT(result.IsSuccess());

    int64_t fileSize;
    result = nn::fs::GetFileSize(&fileSize, file);
    NN_ASSERT(result.IsSuccess());
    NN_ASSERT_LESS(fileSize, static_cast<int64_t>(bufferSize));

    size_t readSize;
    result = nn::fs::ReadFile(&readSize, file, 0, pOut, bufferSize);
    NN_ASSERT(result.IsSuccess());
    NN_ASSERT_EQUAL(static_cast<int64_t>(readSize), fileSize);

    nn::fs::CloseFile(file);

    return readSize;
}

static const size_t MaxFileSize  = 0x400000;
static void* nro;
static void* bss;
static void* nrr;
static size_t imageSize;
static size_t bufferSize;
static size_t nrrSize;
static nn::ro::Module module;
static nn::ro::RegistrationInfo info;
static long            num_dtrs_to_call = 0;
static char* cacheBuffer;

static void load_nro_module()
{
    nn::Result result;

    TESTLOG("load_nro_module calling nn::ro::Initialize()");
    nn::ro::Initialize();

    // ファイルシステムのメタデータキャッシュに必要なバッファサイズを修得
    size_t cacheSize = 0;
    TESTLOG("load_nro_module calling nn::fs::QueryMountRomCacheSize(&cacheSize)");
    result = nn::fs::QueryMountRomCacheSize(&cacheSize);
    NN_ASSERT(result.IsSuccess());

    // キャッシュバッファを確保
    TESTLOG("load_nro_module calling new(std::nothrow) char[cacheSize]");
    cacheBuffer = new(std::nothrow) char[cacheSize];
    NN_ASSERT_NOT_NULL(cacheBuffer);

    // ファイルシステムをマウント
    TESTLOG("load_nro_module calling nn::fs::MountRom(\"rom\", cacheBuffer, cacheSize)");
    result = nn::fs::MountRom("rom", cacheBuffer, cacheSize);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    // nrr を読み込み
    TESTLOG("load_nro_module calling nrr = aligned_alloc(nn::os::MemoryPageSize, MaxFileSize)");
    nrr = aligned_alloc(nn::os::MemoryPageSize, MaxFileSize);
    TESTLOG("load_nro_module calling ReadAll(nrr, MaxFileSize, \"%s\")", NRR_NAME);
    nrrSize = ReadAll(nrr, MaxFileSize, NRR_NAME);

    // nrr を登録
    TESTLOG("load_nro_module calling nn::ro::RegisterModuleInfo(&info, nrr)");
    result = nn::ro::RegisterModuleInfo(&info, nrr);
    NN_ASSERT(result.IsSuccess());

    TESTLOG("load_nro_module calling nro = aligned_alloc(nn::os::MemoryPageSize, MaxFileSize)");
    nro = aligned_alloc(nn::os::MemoryPageSize, MaxFileSize);
    TESTLOG("load_nro_module calling ReadAll(nro, MaxFileSize, \"%s\")", NR0_NAME);
    imageSize = ReadAll(nro, MaxFileSize, NR0_NAME);

    TESTLOG("load_nro_module calling nn::ro::GetBufferSize(&bufferSize, nro)");
    result = nn::ro::GetBufferSize(&bufferSize, nro);
    NN_ASSERT(result.IsSuccess());
    if (bufferSize != 0)
    {
    TESTLOG("load_nro_module calling bss = aligned_alloc(nn::os::MemoryPageSize, bufferSize)");
        bss = aligned_alloc(nn::os::MemoryPageSize, bufferSize);
    }
    else
    {
    TESTLOG("load_nro_module setting bss = 0");
        bss = 0;
    }

    // nro をロード(シンボルは遅延解決)
    TESTLOG("load_nro_module calling nn::ro::LoadModule(&module, nro, bss, bufferSize, nn::ro::BindFlag_Lazy)");
    result = nn::ro::LoadModule(&module, nro, bss, bufferSize, nn::ro::BindFlag_Lazy);
    NN_ASSERT(result.IsSuccess());
}

static void unload_nro_module()
{
    TESTLOG("unload_nro_module calling nn::ro::UnloadModule(&module)");
    nn::ro::UnloadModule(&module);
    TESTLOG("unload_nro_module calling nn::ro::UnregisterModuleInfo(&info)");
    nn::ro::UnregisterModuleInfo(&info);
    TESTLOG("unload_nro_module calling nn::ro::Finalize()");
    nn::ro::Finalize();

    // アンマウントする
    TESTLOG("unload_nro_module calling nn::fs::Unmount(\"rom\")");
    nn::fs::Unmount("rom");
    delete[] cacheBuffer;
    cacheBuffer = NULL;

    TESTLOG("unload_nro_module calling __nnmusl_get_tls_dtors_status() with 0x%08zX", __dso_handle);
    num_dtrs_to_call = __nnmusl_get_tls_dtors_status (__dso_handle);
    TESTCASE_MESSAGE(num_dtrs_to_call == 0, "there were %ld uncalled destructors\n", num_dtrs_to_call);

}

static void TestLocal ()
{
    TESTLOG("Entering TestLocal()");

    thread_local short GlobalShortFirstFile = 23;
    short NonTlsGlobalShortFirstFile = 23;

    TESTCASE_MESSAGE(GlobalShortFirstFile == 23, "GlobalShortFirstFile in TestLocal(): %d", GlobalShortFirstFile);
    TESTCASE_MESSAGE(NonTlsGlobalShortFirstFile == 23, "NonTlsGlobalShortFirstFile in TestLocal(): %d", NonTlsGlobalShortFirstFile);

    TESTLOG("Exiting TestLocal()");
}

/*---------------------------------------------------------------------------*
  Name:         TestThread

  Description:  This is the main function of the thread.
                It displays the resulting values of each variable.

  Arguments:    name of thread as string.

  Returns:      Always zero

 *---------------------------------------------------------------------------*/


static void TestThread(void *arg)
{
    char *self = (char *)__get_tp();
    TESTLOG("%s: TestThread arg is %s", (char *)arg, (char *)arg);
    TESTLOG("%s: TP is %p", (char *)arg, self);
    int gold_tpoff = (tls_segment_align > TPOFF_K) ? tls_segment_align : TPOFF_K;
    tls_data *td = GET_TLS_DATA(self);
    int i, j, mod_id = 0;
    void *this_threads_tls_image_from_tp = (void *)((size_t)self + (size_t)gold_tpoff);
    int strcmp_result = strcmp(StaticPointer, "FirstFile");
    int char_compare = (StaticChar == 'f') ? 1 : 0;
    typedef struct tls_var {
        void *addr;
        void *calc_addr;
        char *name;
        size_t size;
        size_t align;
        bool defined;
        bool initialized;
    } tls_var;
    void *pos;
    int first, prev, prev_all, prev_undef;
    bool seen_bss;

    for (i = 1; i <= horizon_max_num_modules; i++) {
        if (this_threads_tls_image_from_tp == td->dtv[i])
        {
            mod_id = i;
            break;
        }
    }
    TESTLOG("%s: mod_id is %d", (char *)arg, mod_id);
    TESTCASE_MESSAGE(mod_id != 0, "%s: mod_id is: %d", (char *)arg, mod_id);
    tls_var var_array[9];
    tls_var swap;
    int lower;

    // the following are defined in this module
    var_array[0].addr = (void *)&GlobalShortFirstFile;
    var_array[0].name = (char *)"GlobalShortFirstFile";
    var_array[0].size = sizeof(GlobalShortFirstFile);
    var_array[0].align = alignof(GlobalShortFirstFile);
    var_array[0].defined = true;
    var_array[0].initialized = false;

    var_array[1].addr = (void *)&GlobalIntFirstFile;
    var_array[1].name = (char *)"GlobalIntFirstFile";
    var_array[1].size = sizeof(GlobalIntFirstFile);
    var_array[1].align = alignof(GlobalIntFirstFile);
    var_array[1].defined = true;
    var_array[1].initialized = true;

    var_array[2].addr = (void *)&StaticChar;
    var_array[2].name = (char *)"StaticChar";
    var_array[2].size = sizeof(StaticChar);
    var_array[2].align = alignof(StaticChar);
    var_array[2].defined = true;
    var_array[2].initialized = true;

    var_array[3].addr = (void *)&StaticPointer;
    var_array[3].name = (char *)"StaticPointer";
    var_array[3].size = sizeof(StaticPointer);
    var_array[3].align = alignof(StaticPointer);
    var_array[3].defined = true;
    var_array[3].initialized = true;

    var_array[4].addr = (void *)&WeakTlsFirstStrongTlsSecond;
    var_array[4].name = (char *)"WeakTlsFirstStrongTlsSecond";
    var_array[4].size = sizeof(WeakTlsFirstStrongTlsSecond);
    var_array[4].align = alignof(WeakTlsFirstStrongTlsSecond);
    var_array[4].defined = true;
    var_array[4].initialized = true;

    var_array[5].addr = (void *)&StrongTlsFirstWeakTlsSecond;
    var_array[5].name = (char *)"StrongTlsFirstWeakTlsSecond";
    var_array[5].size = sizeof(StrongTlsFirstWeakTlsSecond);
    var_array[5].align = alignof(StrongTlsFirstWeakTlsSecond);
    var_array[5].defined = true;
    var_array[5].initialized = true;

    var_array[6].addr = (void *)&DuplicateNameTLS;
    var_array[6].name = (char *)"DuplicateNameTLS";
    var_array[6].size = sizeof(DuplicateNameTLS);
    var_array[6].align = alignof(DuplicateNameTLS);
    var_array[6].defined = true;
    var_array[6].initialized = true;

    // the following are undefined in this module
    var_array[7].addr = (void *)&GlobalDoubleSecondFile;
    var_array[7].name = (char *)"GlobalDoubleSecondFile";
    var_array[7].size = sizeof(GlobalDoubleSecondFile);
    var_array[7].align = alignof(GlobalDoubleSecondFile);
    var_array[7].defined = false;
    var_array[7].initialized = true; // initialized in other module

    var_array[8].addr = (void *)&GlobalFloatSecondFile;
    var_array[8].name = (char *)"GlobalFloatSecondFile";
    var_array[8].size = sizeof(GlobalFloatSecondFile);
    var_array[8].align = alignof(GlobalFloatSecondFile);
    var_array[8].defined = false;
    var_array[8].initialized = false; // not initialized in other module

    for (i = 0; i < 9; i++) {
        lower = i;
        for (j = i+1; j < 9; j++) {
            if (var_array[j].addr < var_array[lower].addr) {
                lower = j;
            }
        }
        if (lower != i) {
            swap = var_array[i];
            var_array[i] = var_array[lower];
            var_array[lower] = swap;
        }
        if (var_array[i].defined) {
            TESTLOG("%s: defined %s: %p", (char *)arg, var_array[i].name, var_array[i].addr);
        } else {
            TESTLOG("%s: undefined %s: %p", (char *)arg, var_array[i].name, var_array[i].addr);
        }
    }

    TESTLOG("%s: tls_segment_align = %zd", (char *)arg, tls_segment_align);
    TESTLOG("%s: TLS segment size = %zd", (char *)arg, tls_segment_size);
    TESTLOG("%s: TLS segment start = %p", (char *)arg, (size_t)this_threads_tls_image_from_tp);
    TESTLOG("%s: TLS segment initialized section start = %p", (char *)arg, (size_t)this_threads_tls_image_from_tp + tdata_offset);
    TESTLOG("%s: TLS segment initialized section alignment = %zd", (char *)arg, tdata_align_abs);
    TESTLOG("%s: TLS segment uninitialized section start = %p", (char *)arg, (size_t)this_threads_tls_image_from_tp + tbss_offset);
    TESTLOG("%s: TLS segment uninitialized section alignment = %zd", (char *)arg, tbss_align_abs);
    TESTLOG("%s: TLS segment end = %p", (char *)arg, (void *)((size_t)this_threads_tls_image_from_tp + tls_segment_size));

    first = 7;
    // work on defined vars first
    for (i = 0; i < 6; i++) {
        if (var_array[i].defined == true) {
            first = i;
            break;
        }
    }
    TESTCASE_MESSAGE(((first >= 0) && (first <= 6)), "%s: first defined var not found: %d", (char *)arg, first);
    TESTLOG("%s: The first defined TLS var in the segment is %s (%p)", (char *)arg, var_array[first].name, var_array[first].addr);
    pos = var_array[first].calc_addr = var_array[first].addr;
    prev = 7;
    seen_bss = var_array[first].initialized == false;
    if (seen_bss == false) {
        TESTCASE_MESSAGE(var_array[first].addr == (void *)((size_t)this_threads_tls_image_from_tp + tdata_offset), "%s: first variable (initialized) is at unexpected address (expected %p, got %p)",
            (char *)arg, (void *)((size_t)this_threads_tls_image_from_tp + tdata_offset), var_array[first].addr);
    } else {
        TESTCASE_MESSAGE(var_array[first].addr == (void *)((size_t)this_threads_tls_image_from_tp + tbss_offset), "%s: first variable (uninitialized) is at unexpected address (expected %p, got %p)",
            (char *)arg, (void *)((size_t)this_threads_tls_image_from_tp + tbss_offset), var_array[first].addr);
    }
    TESTLOG("%s: work on defined vars first", (char *)arg);
    for (i = 0; i < 6; i++) {
        if (var_array[i].defined == true) {
            if (i != first) {
                if ((seen_bss == false) && !var_array[i].initialized) {
                    // tbss_align_abs might not be what the tbss section is aligned to; sometimes it is aligned to tdata_align_abs
                    pos = (void *)((size_t)this_threads_tls_image_from_tp + tbss_offset);
                    seen_bss = true;
                } else {
                    pos = (void *)(((size_t)pos + (var_array[i].align - 1)) & ~(var_array[i].align - 1));
                }
                var_array[i].calc_addr = pos;
                TESTLOG("%s: %s is at %p and calculated it at %p", (char *)arg, var_array[i].name, var_array[i].addr, pos);
                TESTCASE_MESSAGE(pos == var_array[i].addr, "%s: defined %s current address (%p) expected address (%p)",
                    (char *)arg, var_array[i].name, var_array[i].addr, pos);
            }
            pos = (void *)((size_t)pos + var_array[i].size);
            if (var_array[i].initialized) {
                TESTCASE_MESSAGE(tdata_align_abs >= var_array[i].align, "%s: defined %s's alignment too big for .tdata (expected <= %zd, got %zd)",
                    (char *)arg, var_array[i].name, tdata_align_abs, var_array[i].align);
            } else {
                TESTCASE_MESSAGE(tbss_align_abs >= var_array[i].align, "%s: defined %s's alignment too big for .tbss (expected <= %zd, got %zd)",
                    (char *)arg, var_array[i].name, tbss_align_abs, var_array[i].align);
            }
            if (prev != 7) {
                if (var_array[prev].addr == var_array[i].addr) {
                    TESTCASE_MESSAGE(var_array[prev].addr != var_array[i].addr, "%s: previous %s (%p)  and defined %s (%p) overlap!",
                        (char *)arg, var_array[prev].name, var_array[prev].addr, var_array[i].name, var_array[i].addr);
                } else {
                    TESTCASE_MESSAGE((((size_t)var_array[prev].addr + var_array[prev].size) <= (size_t)var_array[i].addr), "%s: previous %s (%p - %p) is overlapped by defined %s (%p).",
                        (char *)arg, var_array[prev].name, var_array[prev].addr, (void *)((size_t)var_array[prev].addr + var_array[prev].size), var_array[i].name, var_array[i].addr);
                }
                //TESTCASE_MESSAGE((((size_t)var_array[prev].addr + var_array[prev].size) <= ((size_t)var_array[i].addr + var_array[i].size)), "%s: %s (addr + size %p) is stepping on defined %s (addr %p) or v. v.",
                //    (char *)arg, var_array[prev].name, (void *)((size_t)var_array[prev].addr + var_array[prev].size), var_array[i].name, (void *)((size_t)var_array[i].addr + var_array[i].size));
            }
            prev = i;
        }
    }
    prev_all = 7;
    prev_undef = 7;
    TESTLOG("%s: work on undefined vars next", (char *)arg);
    for (i = 0; i < 6; i++) {
        if (var_array[i].defined == false) {
            TESTCASE_MESSAGE(var_array[first].addr != var_array[i].addr, "%s: undefined %s (%p) and defined %s (%p) overlap!",
                (char *)arg, var_array[i].name, var_array[i].addr, var_array[first].name, var_array[first].addr);
#if TEST_BROKEN_IMPORTED_TLS
            if (var_array[i].initialized) {
                TESTCASE_MESSAGE(tdata_align_abs >= var_array[i].align, "%s: undefined %s's alignment too big for .tdata (expected <= %zd, got %zd)",
                    (char *)arg, var_array[i].name, tdata_align_abs, var_array[i].align);
            } else {
                TESTCASE_MESSAGE(tbss_align_abs >= var_array[i].align, "%s: undefined %s's alignment too big for .tbss (expected <= %zd, got %zd)",
                    (char *)arg, var_array[i].name, tbss_align_abs, var_array[i].align);
            }
#endif
            if (prev_all != 7) {
                if (var_array[prev_all].addr == var_array[i].addr) {
                    TESTCASE_MESSAGE(var_array[prev_all].addr != var_array[i].addr, "%s: previous %s (%p)  and undefined %s (%p) overlap!",
                        (char *)arg, var_array[prev_all].name, var_array[prev_all].addr, var_array[i].name, var_array[i].addr);
                } else {
                    TESTCASE_MESSAGE((((size_t)var_array[prev_all].addr + var_array[prev_all].size) <= (size_t)var_array[i].addr), "%s: previous %s (%p - %p) is overlapped by undefined %s (%p).",
                        (char *)arg, var_array[prev_all].name, var_array[prev_all].addr, (void *)((size_t)var_array[prev_all].addr + var_array[prev_all].size), var_array[i].name, var_array[i].addr);
                }
                //TESTCASE_MESSAGE((((size_t)var_array[prev_all].addr + var_array[prev_all].size) <= ((size_t)var_array[i].addr + var_array[i].size)), "%s: %s (addr + size %p) is stepping on undefined %s (addr %p) or v. v.",
                //    (char *)arg, var_array[prev_all].name, (void *)((size_t)var_array[prev_all].addr + var_array[prev_all].size), var_array[i].name, (void *)((size_t)var_array[i].addr + var_array[i].size));
            }
            if ((prev_undef != 7) && (prev_undef == prev_all)) {
                if (var_array[prev_undef].addr == var_array[i].addr) {
                    TESTCASE_MESSAGE(var_array[prev_undef].addr != var_array[i].addr, "%s: previous %s (%p)  and undefined %s (%p) overlap!",
                        (char *)arg, var_array[prev_undef].name, var_array[prev_undef].addr, var_array[i].name, var_array[i].addr);
                } else {
                    TESTCASE_MESSAGE((((size_t)var_array[prev_undef].addr + var_array[prev_undef].size) <= (size_t)var_array[i].addr), "%s: previous %s (%p - %p) is overlapped by undefined %s (%p).",
                        (char *)arg, var_array[prev_undef].name, var_array[prev_undef].addr, (void *)((size_t)var_array[prev_undef].addr + var_array[prev_undef].size), var_array[i].name, var_array[i].addr);
                }
                //TESTCASE_MESSAGE((((size_t)var_array[prev_undef].addr + var_array[prev_undef].size) <= ((size_t)var_array[i].addr + var_array[i].size)), "%s: %s (addr + size %p) is stepping on undefined %s (addr %p) or v. v.",
                //    (char *)arg, var_array[prev_undef].name, (void *)((size_t)var_array[prev_undef].addr + var_array[prev_undef].size), var_array[i].name, (void *)((size_t)var_array[i].addr + var_array[i].size));
            }
            prev_undef = i;
        }
        prev_all = i;
    }

    TESTLOG("%s: GlobalShortFirstFile = %d, expected 0", (char *)arg, GlobalShortFirstFile);
    TESTCASE_MESSAGE(GlobalShortFirstFile == 0, "%s: GlobalShortFirstFile when initialized: %d", (char *)arg, GlobalShortFirstFile);
    TESTLOG("%s: GlobalIntFirstFile = %d, expected 100", (char *)arg, GlobalIntFirstFile);
    TESTCASE_MESSAGE(GlobalIntFirstFile == 100, "%s: GlobalIntFirstFile when initialized: %d", (char *)arg, GlobalIntFirstFile);
    TESTLOG("%s: StaticChar = '%c', expected 'f'", (char *)arg, StaticChar);
    TESTCASE_MESSAGE(char_compare == 1, "%s: StaticChar char_compare when initialized: %d", (char *)arg, char_compare);
    TESTLOG("%s: StaticPointer = '%s', expected 'FirstFile'", (char *)arg, StaticPointer);
    TESTCASE_MESSAGE(strcmp_result == 0, "%s: StaticPointer strcmp result when initialized: %d", (char *)arg, strcmp_result);
    TESTLOG("%s: GlobalDoubleSecondFile = %f, expected 2.0", (char *)arg, GlobalDoubleSecondFile);

    TESTCASE_MESSAGE(WeakTlsFirstStrongTlsSecond == 1234, "%s: WeakTlsFirstStrongTlsSecond when initialized: %d", (char *)arg, WeakTlsFirstStrongTlsSecond);
    TESTCASE_MESSAGE(StrongTlsFirstWeakTlsSecond == 4321, "%s: StrongTlsFirstWeakTlsSecond when initialized: %d", (char *)arg, StrongTlsFirstWeakTlsSecond);
    TESTCASE_MESSAGE(DuplicateNameTLS == 11111, "%s: DuplicateNameTLS when initialized: %d", (char *)arg, DuplicateNameTLS);

    TESTCASE_MESSAGE(GlobalDoubleSecondFile == 2.0, "%s: GlobalDoubleSecondFile when initialized: %f", (char *)arg, GlobalDoubleSecondFile);
    TESTLOG("%s: GlobalFloatSecondFile = %f, expected 0.0", (char *)arg, GlobalFloatSecondFile);
    TESTCASE_MESSAGE(GlobalFloatSecondFile == 0.0, "%s: GlobalFloatSecondFile when initialized: %f", (char *)arg, GlobalFloatSecondFile);

    TESTCASE_MESSAGE(NonTlsGlobalShortFirstFile == 0, "%s: NonTlsGlobalShortFirstFile when initialized: %d", (char *)arg, NonTlsGlobalShortFirstFile);
    TESTCASE_MESSAGE(NonTlsGlobalIntFirstFile == 100, "%s: NonTlsGlobalIntFirstFile when initialized: %d", (char *)arg, NonTlsGlobalIntFirstFile);
    TESTCASE_MESSAGE(NonTlsGlobalDoubleSecondFile == 2.0, "%s: NonTlsGlobalDoubleSecondFile when initialized: %f", (char *)arg, NonTlsGlobalDoubleSecondFile);
    TESTCASE_MESSAGE(NonTlsGlobalFloatSecondFile == 0.0, "%s: NonTlsGlobalFloatSecondFile when initialized: %f", (char *)arg, NonTlsGlobalFloatSecondFile);

    TestLocal();

    TESTCASE_MESSAGE(GlobalShortFirstFile == 0, "%s: GlobalShortFirstFile at thread exit: %d", (char *)arg, GlobalShortFirstFile);
    TESTCASE_MESSAGE(NonTlsGlobalShortFirstFile == 0, "%s: NonTlsGlobalShortFirstFile at thread exit: %d", (char *)arg, NonTlsGlobalShortFirstFile);

    TESTLOG("%s: is exiting", (char *)arg);
}


/*---------------------------------------------------------------------------*
  Name:         main

  Description:  This is the main function of the demo.

                This demo displays Hello! on the main thread and sub-threads.

  Arguments:    None.

  Returns:      Always zero.

 *---------------------------------------------------------------------------*/
extern "C" void nnMain()
{
    nn::Result      result;

    NTD_TEST_START();
    NTD_TEST_GROUP_START(TEST_NAME, 1);
    horizon_max_num_modules = nnosGetModuleCountMax();
    TESTLOG("__EX_start = %p", (unsigned char *)__EX_start);
    TESTLOG("__tdata_start = %p", (unsigned char *)__tdata_start);
    TESTLOG("__tdata_end = %p", (unsigned char *)__tdata_end);
    TESTLOG("__tdata_align_abs = %p", (unsigned char *)__tdata_align_abs);
    TESTLOG("__tdata_align_rel = %p", (unsigned char *)__tdata_align_rel);
    TESTLOG("__tbss_start = %p", (unsigned char *)__tbss_start);
    TESTLOG("__tbss_end = %p", (unsigned char *)__tbss_end);
    TESTLOG("__tbss_align_abs = %p", (unsigned char *)__tbss_align_abs);
    TESTLOG("__tbss_align_rel = %p", (unsigned char *)__tbss_align_rel);
    tdata_align_abs = (size_t)__tdata_align_abs - (size_t)__EX_start;
    TESTLOG("tdata_align_abs = %zd", tdata_align_abs);
    tdata_align_rel = (size_t)__tdata_align_rel - (size_t)__tdata_start;
    TESTCASE_MESSAGE(tdata_align_abs == tdata_align_rel, "Inconsistent alignment for ,tdata abs and rel");
    tbss_align_abs = (size_t)__tbss_align_abs - (size_t)__EX_start;
    TESTLOG("tbss_align_abs = %zd", tbss_align_abs);
    tbss_align_rel = (size_t)__tbss_align_rel - (size_t)__tbss_start;
    TESTCASE_MESSAGE(tbss_align_abs == tbss_align_rel, "Inconsistent alignment for ,tbss abs and rel");
    // ld and lld make an empty .tdata section  have alignment 1; gold does 0 alignment
    if (tdata_align_abs > 1) {
        TESTCASE_MESSAGE(tdata_align_abs >= tbss_align_abs, "Unexpected .tbss alignment greater than .tdata alignment");
    }
    tls_segment_align = (tdata_align_abs > tbss_align_abs) ? tdata_align_abs : tbss_align_abs;
    TESTLOG("tls_segment_align = %ld", tls_segment_align);
    tls_segment_size = (unsigned char *)__tbss_end - (unsigned char *)__tdata_start;
    tdata_size = (unsigned char *)__tdata_end - (unsigned char *)__tdata_start;
    tbss_size = (unsigned char *)__tbss_end - (unsigned char *)__tbss_start;
    tdata_offset = 0;
    tbss_offset = (unsigned char *)__tbss_start - (unsigned char *)__tdata_start;
    TESTLOG("TLS segment size = %ld", tls_segment_size);
    TESTLOG("TLS segment size = %p", (void *)tls_segment_size);

    TESTLOG("nnMain calling load_nro_module()");
    load_nro_module();
    TESTLOG("nnMain after calling load_nro_module()");

    // スレッドを生成する
    result = nn::os::CreateThread( &g_Thread1, TestThread, (void *)"TestThreadOne", g_ThreadStack1, ThreadStackSize, nn::os::DefaultThreadPriority );
    NN_ASSERT( result.IsSuccess(), "Cannot create g_Thread1." );

    TESTLOG("g_Thread1 = 0x%08zX", (size_t)&g_Thread1);

    // スレッドの実行を開始する
    nn::os::StartThread( &g_Thread1 );
    TESTLOG("g_Thread1 started");

    TESTLOG("nnMain calling TestSecondFile()");
    TestSecondFile();
    TESTLOG("nnMain after calling TestSecondFile()");

    // スレッドが終了するのを待つ
    nn::os::WaitThread( &g_Thread1 );
    TESTLOG("g_Thread1 did wait");

    // スレッドを破棄する
    nn::os::DestroyThread( &g_Thread1 );
    TESTLOG("g_Thread1 destroyed");

    TESTLOG("nnMain calling unload_nro_module()");
    unload_nro_module();
    TESTLOG("nnMain after calling unload_nro_module()");

    NTD_TEST_GROUP_END(TEST_NAME, 1);
    NTD_TEST_END();
}

