﻿//
// File:       terrain_simulator.cpp
//
// Abstract:   This example shows how OpenCL can be used to create a procedural field of
//             grass on a generated terrain model which is then rendered with OpenGL.
//             Because OpenGL buffers are shared with OpenCL, the data can remain on the
//             graphics card, thus eliminating the API overhead of creating and submitting
//             the vertices from the host.
//
//             All geometry is generated on the compute device, and outputted into
//             a shared OpenGL buffer.  The terrain gets generated only within the
//             visible arc covering the camera's view frustum to avoid the need for
//             culling.  A page of grass is computed on the surface of the terrain as
//             bezier patches, and flow noise is applied to the angle of the blades
//             to simulate wind.  Multiple instances of grass are rendered at jittered
//             offsets to add more grass coverage without having to compute new pages.
//             Finally, a physically based sky shader (via OpenGL) is applied to
//             the background to provide an environment for the grass.
//
// Version:    <1.0>
//
// Disclaimer: IMPORTANT:  This Apple software is supplied to you by Apple Inc. ("Apple")
//             in consideration of your agreement to the following terms, and your use,
//             installation, modification or redistribution of this Apple software
//             constitutes acceptance of these terms.  If you do not agree with these
//             terms, please do not use, install, modify or redistribute this Apple
//             software.
//
//             In consideration of your agreement to abide by the following terms, and
//             subject to these terms, Apple grants you a personal, non - exclusive
//             license, under Apple's copyrights in this original Apple software ( the
//             "Apple Software" ), to use, reproduce, modify and redistribute the Apple
//             Software, with or without modifications, in source and / or binary forms;
//             provided that if you redistribute the Apple Software in its entirety and
//             without modifications, you must retain this notice and the following text
//             and disclaimers in all such redistributions of the Apple Software. Neither
//             the name, trademarks, service marks or logos of Apple Inc. may be used to
//             endorse or promote products derived from the Apple Software without specific
//             prior written permission from Apple.  Except as expressly stated in this
//             notice, no other rights or licenses, express or implied, are granted by
//             Apple herein, including but not limited to any patent rights that may be
//             infringed by your derivative works or by other works in which the Apple
//             Software may be incorporated.
//
//             The Apple Software is provided by Apple on an "AS IS" basis.  APPLE MAKES NO
//             WARRANTIES, EXPRESS OR IMPLIED, INCLUDING WITHOUT LIMITATION THE IMPLIED
//             WARRANTIES OF NON - INFRINGEMENT, MERCHANTABILITY AND FITNESS FOR A
//             PARTICULAR PURPOSE, REGARDING THE APPLE SOFTWARE OR ITS USE AND OPERATION
//             ALONE OR IN COMBINATION WITH YOUR PRODUCTS.
//
//             IN NO EVENT SHALL APPLE BE LIABLE FOR ANY SPECIAL, INDIRECT, INCIDENTAL OR
//             CONSEQUENTIAL DAMAGES ( INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
//             SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
//             INTERRUPTION ) ARISING IN ANY WAY OUT OF THE USE, REPRODUCTION, MODIFICATION
//             AND / OR DISTRIBUTION OF THE APPLE SOFTWARE, HOWEVER CAUSED AND WHETHER
//             UNDER THEORY OF CONTRACT, TORT ( INCLUDING NEGLIGENCE ), STRICT LIABILITY OR
//             OTHERWISE, EVEN IF APPLE HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
//
// Copyright ( C ) 2008 Apple Inc. All Rights Reserved.
//
////////////////////////////////////////////////////////////////////////////////////////////////////

#include <cmath>
#include <cassert>
#include <ctime>

#include <gfx/demo.h>

#if NN_GFX_IS_TARGET_GX
#include <cafe/gx2.h>
#define MTX_USE_PS
#include <cafe/mat.h>
#endif

#include "compute_types.h"
#include "compute_math.h"
#include "terrain_simulator.h"

/////////////////////////////////////////////////////////////////////////////

TerrainSimulator::TerrainSimulator() :
    m_uiSizeX(128),
    m_uiSizeY(128),
    m_fJitterAmount(0),
    m_fCameraFov(0),
    m_fNoiseAmplitude(0),
    m_uiVertexCount(0),
    m_uiVertexComponents(4),
    m_uiNormalCount(0),
    m_uiNormalComponents(4),
    m_uiTexCoordCount(0),
    m_uiTexCoordComponents(4),
    m_fFalloff(0),
    m_auiDispatch(0)
{
    m_auiGlobalDim[0] = m_auiGlobalDim[1] = m_auiGlobalDim[2] = 0;
    m_auiLocalDim[0] = m_auiLocalDim[1] = m_auiLocalDim[2] = 0;
}

TerrainSimulator::~TerrainSimulator()
{
}

void
TerrainSimulator::finalize()
{
    m_Pipeline.Finalize( );
    m_VariableTerrainUniforms.Finalize( );
    m_FixedTerrainUniforms.Finalize( );
#if NN_GFX_IS_TARGET_GX
    m_DispatchBuf.Finalize();
#endif
}

bool
TerrainSimulator::allocate(uint uiSizeX, uint uiSizeY)
{
    m_uiVertexCount = uiSizeX * uiSizeY;
    m_uiNormalCount = uiSizeX * uiSizeY;
    m_uiTexCoordCount = uiSizeX * uiSizeY;

    // Uniform buffers
    m_VariableTerrainUniforms.Initialize( sizeof( VariableTerrainUniforms ), NULL, nn::gfx::GpuAccess_ConstantBuffer | nn::gfx::GpuAccess_Read, 0 );
    m_FixedTerrainUniforms.Initialize( sizeof( FixedTerrainUniforms ), NULL, nn::gfx::GpuAccess_ConstantBuffer | nn::gfx::GpuAccess_Read, 0 );

    // Setup the fixed uniform buffer
    FixedTerrainUniforms *pFixedTerrainUniform;
    pFixedTerrainUniform = m_FixedTerrainUniforms.Map< FixedTerrainUniforms >();
    for (u32 i = 0; i < 512; i++)
    {
        // align elements on vec4 boundaries
        pFixedTerrainUniform->P[i].x = P[i];
    }
    for (u32 i = 0; i < 64; i++)
    {
        // align elements on vec4 boundaries
        pFixedTerrainUniform->G[i].x = G[i];
    }

#ifdef CAFE
    GX2EndianSwap(pFixedTerrainUniform, sizeof(FixedTerrainUniforms));
#endif

    m_FixedTerrainUniforms.Unmap();

#if NN_GFX_IS_TARGET_GX
    m_DispatchBuf.Initialize( sizeof( uint32_t ) * 4, NULL, nn::gfx::GpuAccess_IndirectBuffer, 0 );
    m_auiDispatch = m_DispatchBuf.Map< uint32_t >();
    m_auiDispatch[ 0 ] = m_uiSizeX / m_auiLocalDim[ 0 ];
    m_auiDispatch[ 1 ] = m_uiSizeY / m_auiLocalDim[ 1 ];
    m_auiDispatch[ 2 ] = 1;
    m_auiDispatch[ 3 ] = 0;
    m_DispatchBuf.Unmap();
#endif

    return true;
}

uint
TerrainSimulator::getRequiredVertexBufferSize(
    uint uiSizeX, uint uiSizeY)
{
    nn::gfx::Buffer::InfoType bufferInfo;
    uint uiVertexCount = uiSizeX * uiSizeY;
    size_t uiVertexBytes = static_cast< size_t >( uiVertexCount * sizeof(float) * m_uiVertexComponents );
    bufferInfo.SetDefault();
    bufferInfo.SetGpuAccessFlags( nn::gfx::GpuAccess_VertexBuffer | nn::gfx::GpuAccess_Read );
    bufferInfo.SetSize( uiVertexBytes );
    size_t align = nn::gfx::Buffer::GetBufferAlignment( &DEMODevice, bufferInfo );
    uiVertexBytes = ( uiVertexBytes + align - 1 ) & ~( align - 1 );
    return static_cast< uint >( uiVertexBytes );
}

uint
TerrainSimulator::getRequiredNormalBufferSize(
    uint uiSizeX, uint uiSizeY)
{
    nn::gfx::Buffer::InfoType bufferInfo;
    uint uiNormalCount = uiSizeX * uiSizeY;
    size_t uiNormalBytes = static_cast< size_t >( uiNormalCount * sizeof(float) * m_uiNormalComponents );
    bufferInfo.SetDefault();
    bufferInfo.SetGpuAccessFlags( nn::gfx::GpuAccess_VertexBuffer | nn::gfx::GpuAccess_Read );
    bufferInfo.SetSize( uiNormalBytes );
    size_t align = nn::gfx::Buffer::GetBufferAlignment( &DEMODevice, bufferInfo );
    uiNormalBytes = ( uiNormalBytes + align - 1 ) & ~( align - 1 );
    return static_cast< uint >( uiNormalBytes );
}

uint
TerrainSimulator::getRequiredTexCoordBufferSize(
    uint uiSizeX, uint uiSizeY)
{
    nn::gfx::Buffer::InfoType bufferInfo;
    uint uiTexCoordCount = uiSizeX * uiSizeY;
    size_t uiTexCoordBytes = static_cast< size_t >( uiTexCoordCount * sizeof(float) * m_uiTexCoordComponents );
    bufferInfo.SetDefault();
    bufferInfo.SetGpuAccessFlags( nn::gfx::GpuAccess_VertexBuffer | nn::gfx::GpuAccess_Read );
    bufferInfo.SetSize( uiTexCoordBytes );
    size_t align = nn::gfx::Buffer::GetBufferAlignment( &DEMODevice, bufferInfo );
    uiTexCoordBytes = ( uiTexCoordBytes + align - 1 ) & ~( align - 1 );
    return static_cast< uint >( uiTexCoordBytes );
}


void
TerrainSimulator::setProjectedCorners(
    const float4 afCorners[4])
{
    for (uint i = 0; i < 4; i++)
    {
        m_akProjectedCorners[i] = afCorners[i];
    }
}

bool
TerrainSimulator::setup(
    uint uiSizeX, uint uiSizeY, const char* pFilename )
{
    if (!allocate(uiSizeX, uiSizeY))
    {
        return false;
    }

    m_uiSizeX = uiSizeX;
    m_uiSizeY = uiSizeY;

    printf("Terrain Simulator: Vertices[%d] Count[%d, %d]\n",
            m_uiVertexCount, uiSizeX, uiSizeY);

    m_Pipeline.SetDefaults();
    DEMOGfxLoadShadersFromFile( &m_Pipeline.shaders, 0, pFilename );

    // Setup Local Work Group Size
    m_Pipeline.shaders.GetShader()->GetWorkGroupSize( &m_auiLocalDim[ 0 ], &m_auiLocalDim[ 1 ], &m_auiLocalDim[ 2 ] );

    // Uniform Location lookup
    m_iVUniformsLoc = m_Pipeline.shaders.GetInterfaceSlot( nn::gfx::ShaderStage_Compute, nn::gfx::ShaderInterfaceType_ConstantBuffer, "variable_terrain" );
    m_iFUniformsLoc = m_Pipeline.shaders.GetInterfaceSlot( nn::gfx::ShaderStage_Compute, nn::gfx::ShaderInterfaceType_ConstantBuffer, "fixed_terrain" );
    m_iExpBufLoc = m_Pipeline.shaders.GetInterfaceSlot( nn::gfx::ShaderStage_Compute, nn::gfx::ShaderInterfaceType_UnorderedAccessBuffer, "attribute_output" );

    DEMOAssert( m_iVUniformsLoc != 0xffffffff && "variable_terrain location is invalid." );
    DEMOAssert( m_iFUniformsLoc != 0xffffffff && "fixed terrain location is invalid." );

    // Initialize the pipeline
    nn::gfx::ComputePipelineInfo info;
    info.SetDefault();
    info.SetShaderPtr( m_Pipeline.shaders.GetShader() );
    m_Pipeline.pipeline.Initialize( &DEMODevice, info );

    return true;
}

bool
TerrainSimulator::compute()
{
    int aiGridResolution[] = {(int)m_uiSizeX, (int)m_uiSizeY};

/*
    float fFrequency = 0.0025f;
    float fAmplitude = 50.00f;
    float fPhase = 1.0f;
    float fLacunarity = 2.0345f;
    float fIncrement = 1.0f;
    float fOctaves = 1.0f;
    float fRoughness = 1.00f;
*/
    VariableTerrainUniforms* pVUniform = m_VariableTerrainUniforms.Map< VariableTerrainUniforms >( );
    memcpy(&pVUniform->grid_resolution, aiGridResolution, sizeof(int) * 2);
    memcpy(&pVUniform->camera_position, &m_kCameraPosition, sizeof(float) * 4);
    memcpy(&pVUniform->camera_rotation, &m_kCameraRotation, sizeof(float) * 4);
    memcpy(&pVUniform->camera_view, &m_kCameraView, sizeof(float) * 4);
    memcpy(&pVUniform->camera_left, &m_kCameraLeft, sizeof(float) * 4);
    memcpy(&pVUniform->camera_fov, &m_fCameraFov, sizeof(float) * 1);
    memcpy(&pVUniform->clip_range, &m_kClipRange, sizeof(m_kClipRange));
    memcpy(&pVUniform->vertex_count, &m_uiVertexCount, sizeof(uint) * 1);

    // Setup offsets for our buffer write
    pVUniform->vertex_output_offset = 0;
//    m_pVariableTerrainUniform->normal_output_offset = ((char*)m_afNormalData - ((char*)m_pExportData)) / sizeof(float4);
//    m_pVariableTerrainUniform->texcoords_output_offset = ((char*)m_afTexCoordData - ((char*)m_pExportData)) / sizeof(float4);

#ifdef CAFE
    GX2EndianSwap(pVUniform, sizeof(VariableTerrainUniforms));
#endif
    m_VariableTerrainUniforms.Unmap( );

#if NN_GFX_IS_TARGET_GX
    // Set shader mode to compute shaders
    GX2SetShaderMode( GX2_SHADER_MODE_COMPUTE_SHADER );
#endif

    // Generate commands
    DEMOCommandBuffer.SetConstantBuffer( m_iVUniformsLoc, nn::gfx::ShaderStage_Compute, m_VariableTerrainUniforms.gpuAddress, m_VariableTerrainUniforms.size );
    DEMOCommandBuffer.SetConstantBuffer( m_iFUniformsLoc, nn::gfx::ShaderStage_Compute, m_FixedTerrainUniforms.gpuAddress, m_FixedTerrainUniforms.size );
    DEMOCommandBuffer.SetUnorderedAccessBuffer( m_iExpBufLoc, nn::gfx::ShaderStage_Compute, m_ExportBufferAddress, m_ExportBufferSize );
    DEMOCommandBuffer.SetPipeline( &m_Pipeline.pipeline );
    DEMOCommandBuffer.InvalidateMemory( nn::gfx::GpuAccess_VertexBuffer | nn::gfx::GpuAccess_ConstantBuffer );
#if NN_GFX_IS_TARGET_GX
    DEMOCommandBuffer.DispatchIndirect( m_DispatchBuf.gpuAddress );
#else
    DEMOCommandBuffer.Dispatch(
        m_uiSizeX / m_auiLocalDim[ 0 ],
        m_uiSizeY / m_auiLocalDim[ 1 ],
        1 );
#endif

    // Flush output from GPU
    DEMOCommandBuffer.FlushMemory( nn::gfx::GpuAccess_UnorderedAccessBuffer );

#if NN_GFX_IS_TARGET_D3D
    nn::gfx::GpuAddress nullAddress;
    DEMOCommandBuffer.SetUnorderedAccessBuffer( m_iExpBufLoc, nn::gfx::ShaderStage_Compute, nullAddress, 0 );
#endif

#if NN_GFX_IS_TARGET_GX
    GX2SetShaderMode( GX2_SHADER_MODE_UNIFORM_BLOCK );
#endif

    return true;
}
