﻿//
// File:       main.cpp
//
// Abstract:   This example shows how OpenCL can be used to create a procedural field of
//             grass on a generated terrain model which is then rendered with OpenGL.
//             Because OpenGL buffers are shared with OpenCL, the data can remain on the
//             graphics card, thus eliminating the API overhead of creating and submitting
//             the vertices from the host.
//
//             All geometry is generated on the compute device, and outputted into
//             a shared OpenGL buffer.  The terrain gets generated only within the
//             visible arc covering the camera's view frustum to avoid the need for
//             culling.  A page of grass is computed on the surface of the terrain as
//             bezier patches, and flow noise is applied to the angle of the blades
//             to simulate wind.  Multiple instances of grass are rendered at jittered
//             offsets to add more grass coverage without having to compute new pages.
//             Finally, a physically based sky shader (via OpenGL) is applied to
//             the background to provide an environment for the grass.
//
// Version:    <1.0>
//
// Disclaimer: IMPORTANT:  This Apple software is supplied to you by Apple Inc. ("Apple")
//             in consideration of your agreement to the following terms, and your use,
//             installation, modification or redistribution of this Apple software
//             constitutes acceptance of these terms.  If you do not agree with these
//             terms, please do not use, install, modify or redistribute this Apple
//             software.
//
//             In consideration of your agreement to abide by the following terms, and
//             subject to these terms, Apple grants you a personal, non - exclusive
//             license, under Apple's copyrights in this original Apple software ( the
//             "Apple Software" ), to use, reproduce, modify and redistribute the Apple
//             Software, with or without modifications, in source and / or binary forms;
//             provided that if you redistribute the Apple Software in its entirety and
//             without modifications, you must retain this notice and the following text
//             and disclaimers in all such redistributions of the Apple Software. Neither
//             the name, trademarks, service marks or logos of Apple Inc. may be used to
//             endorse or promote products derived from the Apple Software without specific
//             prior written permission from Apple.  Except as expressly stated in this
//             notice, no other rights or licenses, express or implied, are granted by
//             Apple herein, including but not limited to any patent rights that may be
//             infringed by your derivative works or by other works in which the Apple
//             Software may be incorporated.
//
//             The Apple Software is provided by Apple on an "AS IS" basis.  APPLE MAKES NO
//             WARRANTIES, EXPRESS OR IMPLIED, INCLUDING WITHOUT LIMITATION THE IMPLIED
//             WARRANTIES OF NON - INFRINGEMENT, MERCHANTABILITY AND FITNESS FOR A
//             PARTICULAR PURPOSE, REGARDING THE APPLE SOFTWARE OR ITS USE AND OPERATION
//             ALONE OR IN COMBINATION WITH YOUR PRODUCTS.
//
//             IN NO EVENT SHALL APPLE BE LIABLE FOR ANY SPECIAL, INDIRECT, INCIDENTAL OR
//             CONSEQUENTIAL DAMAGES ( INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
//             SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
//             INTERRUPTION ) ARISING IN ANY WAY OUT OF THE USE, REPRODUCTION, MODIFICATION
//             AND / OR DISTRIBUTION OF THE APPLE SOFTWARE, HOWEVER CAUSED AND WHETHER
//             UNDER THEORY OF CONTRACT, TORT ( INCLUDING NEGLIGENCE ), STRICT LIABILITY OR
//             OTHERWISE, EVEN IF APPLE HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
//
// Copyright ( C ) 2008 Apple Inc. All Rights Reserved.
//
////////////////////////////////////////////////////////////////////////////////////////////////////

//
// This example shows how Compute can be used to create a procedural field of
// grass on a generated terrain model which is then rendered with GX2.
// Because we share buffers, the data we produce is consumed in-place by
// rendering, thus eliminating the API overhead of creating and submitting the
// vertices from the host.
//
// All geometry is generated on the GPU, and outputted into a shared buffer.
// A page of grass is computed on the surface of the terrain as bezier patches,
// and flow noise is applied to the angle of the blades to simulate wind.
// Multiple instances of grass are rendered at jittered offsets to add more
// grass coverage without having to compute new pages.
//
// Finally, a physically based sky shader (via VS/PS) is applied to the
// background to provide an environment for the grass.
//
////////////////////////////////////////////////////////////////////////////////////////////////////

#include <cstdio>
#include <string>
#include <cmath>

#include <gfx/demo.h>
#include <nnt.h>
#include <nnt/nnt_Argument.h>

#include "compute_types.h"
#include "camera.h"
#include "grass_simulator.h"
#include "terrain_simulator.h"
#include "mesh_renderer.h"
#include "sky_renderer.h"
#include "compute_math.h"

////////////////////////////////////////////////////
//
// Assets data, types and interface for demos
//
////////////////////////////////////////////////////

#define USE_MEM1    yes

#define SURFACE_WIDTH  (DEMOColorBufferInfo.GetWidth())
#define SURFACE_HEIGHT  (DEMOColorBufferInfo.GetHeight())

static int g_ShaderFileIdx = 0;
static const char *GRASS_SHADER_FILE[] =
{
    "shaders/csgrass/grass",
    "shaders/csgrass/grassHlslcc",
};
static const char *TERRAIN_SHADER_FILE[] =
{
    "shaders/csgrass/terrain_kernels",
    "shaders/csgrass/terrain_kernelsHlslcc",
};
static const char *MESH_SHADER_FILE[] =
{
    "shaders/csgrass/transform",
    "shaders/csgrass/transformHlslcc",
};
static const char *SKY_SHADER_FILE[] =
{
    "shaders/csgrass/sky",
    "shaders/csgrass/skyHlslcc",
};

static DEMOGfxBuffer g_ExportBuffer;
static bool s_firstFrame                            = true;
static float4 TerrainColor                          = make_float4(0.09f, 0.135f, 0.0f, 1.0f);

static bool Orbit                                   = false;
static bool Paused                                  = false;
static bool AnimatedSun                             = false;
static uint Width                                   = 1024;
static uint Height                                  = 1024;
static float AspectRatio;
static uint FieldPages                              = 1;
static uint MaxFieldPages                           = 5;
static uint FieldInstances                          = 9;
static uint MaxFieldInstances                       = 25;
static uint BladeCount                              = 128 * 128;
static uint MaxElementCount                         = 4;
static uint MaxSegmentCount                         = 6;
static uint Iteration                               = 0;
static float Exposure                               = 1.0f;
static float SunAzimuth                             = 300.0f;
static float FalloffDistance                        = 200.0f;

static uint2 TerrainResolution                      = make_uint2(128, 128);

static float2 ClipRange                             = make_float2(0.9f, 1.02f);
static float2 FieldSize                             = make_float2(128.0f, 128.0f);
static float2 BladeThicknessRange                   = make_float2(50.0f, 1.0f);
static float2 BladeLengthRange                      = make_float2(5.0f, 5.0f);
static float2 NoiseBias                             = make_float2(0.5f, 0.5f);
static float2 NoiseScale                            = make_float2(1.0f, 10.0f);

static float NoiseAmplitude                         = 0.25f;
static float FlowScale                              = 100.0f;
static float FlowSpeed                              = 100.0f;
static float FlowAmount                             = 30.0f;
static float BladeIntensity                         = 0.75f;
static float BladeOpacity                           = 0.2f;
static float JitterAmount                           = 4.0f;
static float RandomTable[256]                       = {0};

static float CameraFov                              = 70.0f;
static float CameraFarClip                          = 3.0f * TerrainResolution.x * TerrainResolution.x * (ClipRange.y - ClipRange.x); // Matches terrain grid resolution in terrain_kernel.cs
static float CameraNearClip                         = 0.1f;
static float3 CameraPosition                        = make_float3(0.0f, 20.0f, 0.0f);
static float3 CameraRotation                        = make_float3(0.0f, 20.0f, 0.0f);
const float CameraInertia                           = 0.1f;

static double TimeElapsed                           = 0.0;
static uint ReportStatsInterval                     = 30;
static uint FrameCount                              = 0;

static uint ShowStats                               = 1;
static uint ShowInfo                                = 0;
static char StatsString[1024]                       = "\0";
static char InfoString[1024]                        = "\0";

//////////////////////////////////////////////////////////////////////////////

// Pseudo-random permutation
const int P[512] =
{
    151,160,137,91,90,15,
    131,13,201,95,96,53,194,233,7,225,140,36,103,30,69,142,8,99,37,240,21,10,23,
    190, 6,148,247,120,234,75,0,26,197,62,94,252,219,203,117,35,11,32,57,177,33,
    88,237,149,56,87,174,20,125,136,171,168, 68,175,74,165,71,134,139,48,27,166,
    77,146,158,231,83,111,229,122,60,211,133,230,220,105,92,41,55,46,245,40,244,
    102,143,54, 65,25,63,161, 1,216,80,73,209,76,132,187,208, 89,18,169,200,196,
    135,130,116,188,159,86,164,100,109,198,173,186, 3,64,52,217,226,250,124,123,
    5,202,38,147,118,126,255,82,85,212,207,206,59,227,47,16,58,17,182,189,28,42,
    223,183,170,213,119,248,152, 2,44,154,163, 70,221,153,101,155,167, 43,172,9,
    129,22,39,253, 19,98,108,110,79,113,224,232,178,185, 112,104,218,246,97,228,
    251,34,242,193,238,210,144,12,191,179,162,241, 81,51,145,235,249,14,239,107,
    49,192,214, 31,181,199,106,157,184, 84,204,176,115,121,50,45,127, 4,150,254,
    138,236,205,93,222,114,67,29,24,72,243,141,128,195,78,66,215,61,156,180,
    151,160,137,91,90,15,
    131,13,201,95,96,53,194,233,7,225,140,36,103,30,69,142,8,99,37,240,21,10,23,
    190, 6,148,247,120,234,75,0,26,197,62,94,252,219,203,117,35,11,32,57,177,33,
    88,237,149,56,87,174,20,125,136,171,168, 68,175,74,165,71,134,139,48,27,166,
    77,146,158,231,83,111,229,122,60,211,133,230,220,105,92,41,55,46,245,40,244,
    102,143,54, 65,25,63,161, 1,216,80,73,209,76,132,187,208, 89,18,169,200,196,
    135,130,116,188,159,86,164,100,109,198,173,186, 3,64,52,217,226,250,124,123,
    5,202,38,147,118,126,255,82,85,212,207,206,59,227,47,16,58,17,182,189,28,42,
    223,183,170,213,119,248,152, 2,44,154,163, 70,221,153,101,155,167, 43,172,9,
    129,22,39,253, 19,98,108,110,79,113,224,232,178,185, 112,104,218,246,97,228,
    251,34,242,193,238,210,144,12,191,179,162,241, 81,51,145,235,249,14,239,107,
    49,192,214, 31,181,199,106,157,184, 84,204,176,115,121,50,45,127, 4,150,254,
    138,236,205,93,222,114,67,29,24,72,243,141,128,195,78,66,215,61,156,180,
};

#define ONE_F1                 (1.0f)
#define ZERO_F1                (0.0f)

// Psuedo-random gradient vectors
const float G[16 * 4] =
{
      +ONE_F1,  +ONE_F1, +ZERO_F1, +ZERO_F1,
      -ONE_F1,  +ONE_F1, +ZERO_F1, +ZERO_F1,
      +ONE_F1,  -ONE_F1, +ZERO_F1, +ZERO_F1,
      -ONE_F1,  -ONE_F1, +ZERO_F1, +ZERO_F1,
      +ONE_F1, +ZERO_F1,  +ONE_F1, +ZERO_F1,
      -ONE_F1, +ZERO_F1,  +ONE_F1, +ZERO_F1,
      +ONE_F1, +ZERO_F1,  -ONE_F1, +ZERO_F1,
      -ONE_F1, +ZERO_F1,  -ONE_F1, +ZERO_F1,
     +ZERO_F1,  +ONE_F1,  +ONE_F1, +ZERO_F1,
     +ZERO_F1,  -ONE_F1,  +ONE_F1, +ZERO_F1,
     +ZERO_F1,  +ONE_F1,  -ONE_F1, +ZERO_F1,
     +ZERO_F1,  -ONE_F1,  -ONE_F1, +ZERO_F1,
      +ONE_F1,  +ONE_F1, +ZERO_F1, +ZERO_F1,
      -ONE_F1,  +ONE_F1, +ZERO_F1, +ZERO_F1,
     +ZERO_F1,  -ONE_F1,  +ONE_F1, +ZERO_F1,
     +ZERO_F1,  -ONE_F1,  -ONE_F1, +ZERO_F1
};

//////////////////////////////////////////////////////////////////////////////

static TerrainSimulator TerrainSimulator;
static GrassSimulator   GrassSimulator;
static Camera           MainCamera;
static MeshRenderer     TerrainRenderer;
static MeshRenderer     GrassRenderer;
static SkyRenderer      SkyRenderer;

////////////////////////////////////////////////////
//
// Prototypes
//
////////////////////////////////////////////////////

static void DrawScene(f32 dt);

template< typename T >
T* AlignedAlloc( void** ppMemory, size_t size, size_t alignment = 1 )
{
    *ppMemory = reinterpret_cast< void* >( (
        reinterpret_cast< uintptr_t >( *ppMemory ) + alignment - 1 ) & ~( alignment - 1 ) );
    T* ret = static_cast< T* >( *ppMemory );
    *ppMemory = static_cast< uint8_t* >( *ppMemory ) + size;
    return ret;
}

////////////////////////////////////////////////////
//
// Functions
//
////////////////////////////////////////////////////

// メモリ
uint8_t* pMemoryHeap;
void* pMemory;

static void
IncreaseSunAzimuth(float delta = 1.1f)
{
    if (SunAzimuth > 50000)
    {
        Exposure *= (1.0f / delta);
    }
    else
    {
        SunAzimuth *= delta;

        GrassSimulator.setBladeIntensity(GrassSimulator.getBladeIntensity() * (1.01f / delta));
    }
}

static void
DecreaseSunAzimuth(float delta = 1.1f)
{
    if (SunAzimuth > 50000 && Exposure < 1.0f)
    {
        Exposure *= (delta);
    }
    else
    {
        SunAzimuth *= (1.0f / delta);

        if (GrassSimulator.getBladeIntensity() < 1.0f)
        {
            GrassSimulator.setBladeIntensity(GrassSimulator.getBladeIntensity() * (delta));
        }
    }
}

// Overall initialization, compare w/ Initialize in appleGrass main.cpp
// ビューポートシザーを初期化
nn::gfx::ViewportScissorState viewportScissor;
void InitViewport()
{
    // Center in window
    DEMOGfxSetViewportScissorState( &viewportScissor, &pMemory,
        static_cast< float >( ( DEMOColorBufferInfo.GetWidth() - Width ) / 2 ),
        static_cast< float >( ( DEMOColorBufferInfo.GetHeight() - Height ) / 2 ),
        (float)Width, (float)Height, 0.0f, 1.0f, (float)DEMOColorBufferInfo.GetHeight(), FALSE );
}

static int SceneInit()
{
    AspectRatio = (float)Width / (float)Height;

    InitViewport( );

    GrassSimulator.setMaxElementCount(MaxElementCount);
    GrassSimulator.setMaxSegmentCount(MaxSegmentCount);

    if ( !GrassSimulator.setup( BladeCount, (uint)FieldSize.x, (uint)FieldSize.y, GRASS_SHADER_FILE[g_ShaderFileIdx] ) )
    {
        exit(1);
    }

    if ( !TerrainSimulator.setup( TerrainResolution.x, TerrainResolution.y, TERRAIN_SHADER_FILE[g_ShaderFileIdx] ) )
    {
        exit(1);
    }

    if (!TerrainRenderer.createGridIndexBuffer(TerrainResolution.x, TerrainResolution.y, true))
    {
        exit(1);
    }

    for (uint i = 0; i < 256; i++)
    {
        RandomTable[i] = (DEMORand() / (float) DEMO_RAND_MAX);
    }

    MainCamera.setInertia(CameraInertia);
    MainCamera.setFovInDegrees(CameraFov);
    MainCamera.setAspect(AspectRatio);
    MainCamera.setNearClip(CameraNearClip);
    MainCamera.setFarClip(CameraFarClip);
    MainCamera.setPosition(CameraPosition);
    MainCamera.setViewport(Width, Height);
    MainCamera.pitch(CameraRotation.x);
    MainCamera.yaw(CameraRotation.y);
    MainCamera.update();

    GrassSimulator.setFalloffDistance(FalloffDistance);
    GrassSimulator.setCameraFov(MainCamera.getFovInDegrees());
    GrassSimulator.setCameraPosition(MainCamera.getPosition());
    GrassSimulator.setCameraRotation(MainCamera.getRotation());
    GrassSimulator.setCameraFrame(MainCamera.getUpDirection(), MainCamera.getViewDirection(), MainCamera.getLeftDirection());
    GrassSimulator.setBladeOpacity(BladeOpacity);
    GrassSimulator.setBladeIntensity(BladeIntensity);
    GrassSimulator.setBladeLengthRange(BladeLengthRange);
    GrassSimulator.setBladeThicknessRange(BladeThicknessRange);
    GrassSimulator.setNoiseBias(NoiseBias);
    GrassSimulator.setNoiseScale(NoiseScale);
    GrassSimulator.setNoiseAmplitude(NoiseAmplitude);
    GrassSimulator.setFlowScale(FlowScale);
    GrassSimulator.setFlowSpeed(FlowSpeed);
    GrassSimulator.setFlowAmount(FlowAmount);
    GrassSimulator.setJitterAmount(JitterAmount);

    uint terrainBytes = TerrainSimulator.getRequiredVertexBufferSize(TerrainResolution.x, TerrainResolution.y);
    uint grassVertexBytes = GrassSimulator.getRequiredVertexBufferSize(BladeCount);
    uint grassColorBytes = GrassSimulator.getRequiredColorBufferSize(BladeCount);
    uint totalExportBytes = terrainBytes + grassVertexBytes + grassColorBytes;
    // Allocate the export data buffer
#if NN_GFX_IS_TARGET_D3D
    g_ExportBuffer.Initialize( totalExportBytes, NULL, nn::gfx::GpuAccess_UnorderedAccessBuffer | nn::gfx::GpuAccess_Read | nn::gfx::GpuAccess_VertexBuffer, 0 );
#else
    g_ExportBuffer.Initialize( totalExportBytes, NULL, nn::gfx::GpuAccess_UnorderedAccessBuffer | nn::gfx::GpuAccess_Read, 0 );
#endif

    // Initialize the data
    float* pData = g_ExportBuffer.Map< float >();
    memset( pData, 0, g_ExportBuffer.size );
    g_ExportBuffer.Unmap();

    // Update the export buffer (a single buffer shared, thanks to Cafe)
    GrassSimulator.setExportBuffer( g_ExportBuffer.gpuAddress, g_ExportBuffer.size );
    TerrainSimulator.setExportBuffer( g_ExportBuffer.gpuAddress, g_ExportBuffer.size );

    // These are used to compute offsets we pass to the shader, to share the single SSBO
    size_t dataSize = 0;

    // This is used to compute GpuAddress for SetVertexBuffer
    nn::gfx::GpuAddress vbAddress = g_ExportBuffer.gpuAddress;

    TerrainRenderer.setVertexData(vbAddress,
                                  TerrainSimulator.getVertexComponentCount(),
                                  TerrainSimulator.getVertexCount());
    TerrainRenderer.setSolidColor(TerrainColor.x * GrassSimulator.getBladeIntensity(),
                                  TerrainColor.y * GrassSimulator.getBladeIntensity(),
                                  TerrainColor.z * GrassSimulator.getBladeIntensity());
    TerrainRenderer.setModelViewData( 16, 1 );
    TerrainRenderer.setup( nn::gfx::PrimitiveTopology_TriangleStrip, MESH_SHADER_FILE[g_ShaderFileIdx] );

    dataSize += terrainBytes;

    // Vertex + Color data are interleaved
    GrassSimulator.setVertexOffset( ( uint )( dataSize / sizeof( float4 ) ) );
    GrassSimulator.setColorOffset( ( uint ) ( dataSize / sizeof( float4 ) ) + 1 );
    vbAddress.Offset( terrainBytes );
    GrassRenderer.setVertexData( vbAddress,
                                GrassSimulator.getVertexComponentCount(),
                                GrassSimulator.getVertexCount());
    GrassRenderer.setColorData( vbAddress,
                               GrassSimulator.getColorComponentCount(),
                               GrassSimulator.getColorCount());
    GrassRenderer.setModelViewData( 16, MaxFieldInstances );
    GrassRenderer.setup( nn::gfx::PrimitiveTopology_LineList, MESH_SHADER_FILE[g_ShaderFileIdx] );

    SkyRenderer.setup( nn::gfx::PrimitiveTopology_TriangleStrip, SKY_SHADER_FILE[g_ShaderFileIdx] );

    return 1;
}

static void
Solve(bool bUseBlades)
{
    DEMOGfxDebugTagIndent("ComputeGrass");
    Iteration++;

    GrassSimulator.setCameraFov(MainCamera.getFovInDegrees());
    GrassSimulator.setCameraRotation(MainCamera.getRotation());
    GrassSimulator.setCameraPosition(MainCamera.getPosition());
    GrassSimulator.setCameraFrame(MainCamera.getUpDirection(),
                                  MainCamera.getViewDirection(),
                                  MainCamera.getLeftDirection());

#if NN_GFX_IS_TARGET_GX
    // Set shader mode to compute shaders
    GX2SetShaderMode( GX2_SHADER_MODE_COMPUTE_SHADER );
#endif

    if ( bUseBlades )
    {
        GrassSimulator.setClipRange(ClipRange);
        GrassSimulator.setBladeThicknessRange(BladeThicknessRange);
        GrassSimulator.computeGrassOnTerrain(Iteration);
    }
    else
    {
        GrassSimulator.setClipRange(ClipRange);
        GrassSimulator.setBladeThicknessRange(make_float2(BladeThicknessRange.x * 0.85f, BladeThicknessRange.y * 0.85f));
        GrassSimulator.computeGrassOnTerrain(Iteration);
        GrassSimulator.setBladeThicknessRange(BladeThicknessRange);
    }

#if NN_GFX_IS_TARGET_GX
    GX2SetShaderMode( GX2_SHADER_MODE_UNIFORM_BLOCK );
#endif

    DEMOGfxDebugTagUndent();
}

void RenderGrass()
{
    dglPushMatrix();
    {
        float fScaleX = -20.0f;
        float fScaleY = -20.0f;
        u32 baseInstance = 0;

        for (uint p = 0; p < FieldPages; p++)
        {
            // Update the modelView matrices of this instancing pass
            DEMOGfxBuffer& modelViewBuffer = GrassRenderer.getModelViewBuffer();
            float16* pModelViews = modelViewBuffer.Map< float16 >();
            u32 idx = 0;

            dglPushMatrix();
            if (p > 0)
            {
                dglTranslatef(fScaleX * RandomTable[p] , 0.0f, (fScaleY * RandomTable[p + 1]));
            }
            memcpy( &pModelViews[ idx++ ], g_modelviewMtx, sizeof( float16 ) );
            dglPopMatrix();

            for (uint i = 1; i < FieldInstances; i++)
            {
                dglPushMatrix();
                dglTranslatef(fScaleX * RandomTable[i + 5] , 0.0f, (fScaleY * RandomTable[i + 6]));
                memcpy( &pModelViews[ idx++ ], g_modelviewMtx, sizeof( float16 ) );
                dglPopMatrix();
            }

            modelViewBuffer.Unmap();

            if (!Paused)
            {
                Solve(false);
            }

            GrassRenderer.render( FieldInstances, baseInstance );
            baseInstance += FieldInstances;
        }

    }
    dglPopMatrix();
}

void RenderTerrain()
{
    DEMOGfxDebugTagIndent("ComputeTerrain");

    TerrainSimulator.setCameraFov(MainCamera.getFovInDegrees());
    TerrainSimulator.setCameraPosition(MainCamera.getPosition());
    TerrainSimulator.setCameraRotation(MainCamera.getRotation());
    TerrainSimulator.setCameraFrame(MainCamera.getUpDirection(),
                                    MainCamera.getViewDirection(),
                                    MainCamera.getLeftDirection());
    TerrainSimulator.setClipRange(ClipRange);
    TerrainSimulator.compute();

    dglPushMatrix();

    // Update the model view matrix
    DEMOGfxBuffer& modelViewBuffer = TerrainRenderer.getModelViewBuffer();
    float16* pModelViews = modelViewBuffer.Map< float16 >( );
    memcpy( &pModelViews[ 0 ], g_modelviewMtx, sizeof( float16 ) );
    modelViewBuffer.Unmap();

    TerrainRenderer.render( 1, 0 );

    dglPopMatrix();

    DEMOGfxDebugTagUndent();
}

void RenderSky()
{
    static bool SunSet = true;
    if (AnimatedSun)
    {
        if (SunSet)
        {
            IncreaseSunAzimuth(1.025f);
        }
        else
        {
            DecreaseSunAzimuth(1.025f);
        }

        if (Exposure < 0.00001f)
        {
            SunSet = false;
        }
        else if (SunAzimuth < 100.0f)
        {
            SunSet = true;
        }
    }

    dglMatrixMode(DGL_MODELVIEW);
    dglPushMatrix();
    dglLoadIdentity();

    SkyRenderer.setInvProjectionMatrix(inverse(*g_projMtx));
    SkyRenderer.setInvModelviewMatrix(inverse(*g_modelviewMtx));
    SkyRenderer.setCameraDirection(MainCamera.getViewDirection());
    SkyRenderer.setExposure(Exposure);
    SkyRenderer.setSunAzimuth(SunSet ? SunAzimuth : -SunAzimuth);

    SkyRenderer.render();

    dglPopMatrix();
}

static void
ReportStats(f32 dt)
{
    TimeElapsed += dt;

    if (FrameCount > ReportStatsInterval)
    {
        double dMilliseconds = TimeElapsed * 1000.0 / (double)FrameCount;
        float fFps = 1.0f / ((float)dMilliseconds / 1000.0f);

        uint uiMaxVertexCount = GrassSimulator.getVertexCount() * FieldInstances;
        sprintf(StatsString, "Vertices: %3.2f M  Blades: %d  Compute: %3.2f ms  Display: %3.2f fps\n",
                uiMaxVertexCount / 1000.0f / 1000.0f,
                BladeCount * FieldInstances * FieldPages,
                (float)dMilliseconds,
                fFps);

        FrameCount = 0;
        TimeElapsed = 0;
    }

    DEMOGfxSetDefaultViewportScissor();

    if (ShowStats)
    {
        DEMOFontPrintf(5, 1, StatsString);
    }

    if (ShowInfo)
    {
        DEMOFontPrintf(5, 2, InfoString);
        ShowInfo = (ShowInfo > 200) ? 0 : ShowInfo + 1;
        if (ShowInfo == 2)
        {
            printf("%s", InfoString);
        }
    }
}

static void DrawScene(f32 dt)
{
    DEMOGfxBeforeRender();

    FrameCount++;

    nn::gfx::ColorTargetView* pCurrentScanBuffer = DEMOGetColorBufferView();
    DEMOCommandBuffer.ClearColor( pCurrentScanBuffer, 0.2f, 0.2f, 0.2f, 1.0f, NULL );
    DEMOCommandBuffer.ClearDepthStencil(&DEMODepthBufferView, 1.0f, 0, nn::gfx::DepthStencilClearMode_DepthStencil, NULL);

#if NN_GFX_IS_TARGET_GX
    GX2SetShaderMode( GX2_SHADER_MODE_UNIFORM_BLOCK );
#endif

    // Setup the color/depth buffers
    const nn::gfx::ColorTargetView* renderTargets[1];
    renderTargets[ 0 ] = pCurrentScanBuffer;
    DEMOCommandBuffer.SetRenderTargets( 1, renderTargets, &DEMODepthBufferView );

    // Setup viewport scissor
    DEMOCommandBuffer.SetViewportScissorState(&viewportScissor);

    MainCamera.update(Orbit);
    MainCamera.enable();

    DEMOGfxDebugTagIndent("RenderSky");
    RenderSky();
    DEMOGfxDebugTagUndent();

    DEMOGfxDebugTagIndent("RenderTerrain");
    RenderTerrain();
    DEMOGfxDebugTagUndent();
    DEMOGfxDebugTagIndent("RenderGrass");
    RenderGrass();
    DEMOGfxDebugTagUndent();

    MainCamera.disable();

    // Report stats once we have collected a frame's time
    if (s_firstFrame == false)
    {
        DEMOGfxDebugTagIndent("PrintInfo");
        ReportStats(dt);
        DEMOGfxDebugTagUndent();
    }
    else
    {
        //GX2SetTVEnable(GX2_TRUE);
        s_firstFrame = false;
    }

    DEMOGfxDoneRender( );
}

//Update states with pad information
static BOOL ProcessPad()
{
    DEMOPadRead();
    u16 button = DEMOPadGetButtonDown(0);

    if (DEMO_PAD_BUTTON_X & button)
    {
         FieldPages = FieldPages < MaxFieldPages ? FieldPages + 1 : 1;
         sprintf(InfoString, "FieldPages = %d\n", FieldPages);
         ShowInfo = 1;
    }
    if (DEMO_PAD_BUTTON_Y & button)
    {
        FieldInstances = FieldInstances < MaxFieldInstances ? FieldInstances + 1 : 1;
        sprintf(InfoString, "FieldInstances = %d\n", FieldInstances);
        ShowInfo = 1;
    }

    if (DEMO_PAD_BUTTON_A & button)
    {
        FlowSpeed = GrassSimulator.getFlowSpeed();
        FlowSpeed = FlowSpeed < 200.0f ? FlowSpeed * 1.1f : 4.0f;
        GrassSimulator.setFlowSpeed(FlowSpeed);
        sprintf(InfoString, "FlowSpeed = %f\n", GrassSimulator.getFlowSpeed());
        ShowInfo = 1;
    }
    if (DEMO_PAD_BUTTON_B & button)
    {
        FlowAmount = GrassSimulator.getFlowAmount();
        FlowAmount = FlowAmount < 60.0f ? FlowAmount * 1.1f : 2.0f;
        GrassSimulator.setFlowAmount(FlowAmount);
        sprintf(InfoString, "FlowAmount = %f\n", GrassSimulator.getFlowAmount());
        ShowInfo = 1;
    }

    if (DEMO_PAD_BUTTON_UP & button)
    {
        BladeLengthRange = GrassSimulator.getBladeLengthRange();
        BladeLengthRange.x = BladeLengthRange.x < 20.0f ? BladeLengthRange.x * 1.1f : 5.0f;
        GrassSimulator.setBladeLengthRange(BladeLengthRange);
        sprintf(InfoString, "BladeLength = (%.2f, %.2f)\n", BladeLengthRange.x, BladeLengthRange.y);
        ShowInfo = 1;
    }
    if (DEMO_PAD_BUTTON_DOWN & button)
    {
        BladeLengthRange = GrassSimulator.getBladeLengthRange();
        BladeLengthRange.y = BladeLengthRange.y < 20.0f ? BladeLengthRange.y * 1.1f : 5.0f;
        GrassSimulator.setBladeLengthRange(BladeLengthRange);
        sprintf(InfoString, "BladeLength = (%.2f, %.2f)\n", BladeLengthRange.x, BladeLengthRange.y);
        ShowInfo = 1;
    }

    if (DEMO_PAD_BUTTON_LEFT & button)
    {
        BladeThicknessRange = GrassSimulator.getBladeThicknessRange();
        BladeThicknessRange.x = BladeThicknessRange.x < 100.0f ? BladeThicknessRange.x * 1.1f : 10.0f;
        GrassSimulator.setBladeThicknessRange(BladeThicknessRange);
        sprintf(InfoString, "BladeThickness = (%.2f, %.2f)\n", BladeThicknessRange.x, BladeThicknessRange.y);
        ShowInfo = 1;
    }
    if (DEMO_PAD_BUTTON_RIGHT & button)
    {
        BladeThicknessRange = GrassSimulator.getBladeThicknessRange();
        BladeThicknessRange.y = BladeThicknessRange.y < 100.0f ? BladeThicknessRange.y * 1.1f : 1.0f;
        GrassSimulator.setBladeThicknessRange(BladeThicknessRange);
        sprintf(InfoString, "BladeThickness = (%.2f, %.2f)\n", BladeThicknessRange.x, BladeThicknessRange.y);
        ShowInfo = 1;
    }
    if (DEMO_PAD_TRIGGER_L & button)
    {
        BladeIntensity = GrassSimulator.getBladeIntensity();
        BladeIntensity = BladeIntensity < 2.0f ? BladeIntensity * 1.1f : 0.2f;
        GrassSimulator.setBladeIntensity(BladeIntensity);
        sprintf(InfoString, "BladeIntensity = %f\n", BladeIntensity);
        ShowInfo = 1;
    }
    if (DEMO_PAD_TRIGGER_R & button)
    {
        JitterAmount = GrassSimulator.getJitterAmount();
        JitterAmount = JitterAmount < 10.0f ? JitterAmount * 1.1f : 1.0f;
        GrassSimulator.setJitterAmount(JitterAmount);
        sprintf(InfoString, "JitterAmount = %f\n", JitterAmount);
        ShowInfo = 1;
    }
    if (DEMO_PAD_TRIGGER_Z & button)
    {
        AnimatedSun = AnimatedSun ^ true;
    }

    {
        // L Stick translates the camera
        f32 sx = (f32)DEMOPadGetStickX(0) / 255.0f;
        MainCamera.strafe(sx * 2.0f);
        f32 sy = (f32)DEMOPadGetStickY(0) / 255.0f;
        MainCamera.forward(sy * 2.0f);
    }

    {
        // R Stick rotates the camera
        f32 sx = (f32)DEMOPadGetSubStickX(0) / 255.0f;
        MainCamera.yaw(-sx * 2.0f);
        f32 sy = (f32)DEMOPadGetSubStickY(0) / 255.0f;
        MainCamera.pitch(sy * 2.0f);
    }

    return 1;
}

//extern "C" void nnMain()
TEST(GfxCsGrass, Run)
{
    int argc = nnt::GetHostArgc();
    char** argv = nnt::GetHostArgv();

    // Set the PRNG to a fixed value so that our random numbers are predictable
    // for testing
    DEMOSRand(0xcafebadd);

    DEMOInit();
    DEMOTestInit(argc, argv);
    DEMOGfxInit(argc, argv);
    DEMOFontInit();
    DEMOTestIsUseHlslccGlsl() ? g_ShaderFileIdx = 1 : g_ShaderFileIdx = 0;
    Height = std::min( Height, static_cast< uint >( DEMOColorBufferInfo.GetHeight() ) );
    pMemoryHeap = new uint8_t[ 1024 * 1024 * 16 ];
    pMemory = pMemoryHeap;

    // Initialize scene
    SceneInit();

    OSTime  lastTime = OSGetTime();
    while (DEMOIsRunning())
    {
        f32 dtime = OSTicksToMilliseconds(OSGetTime() - lastTime) / 1000.0f;
        lastTime = OSGetTime();

        ProcessPad();
        DrawScene(dtime);
    }

    TerrainSimulator.finalize();
    GrassSimulator.finalize();
    TerrainRenderer.finalize();
    GrassRenderer.finalize();
    SkyRenderer.finalize();
    g_ExportBuffer.Finalize();
    viewportScissor.Finalize( &DEMODevice );

    delete[] pMemoryHeap;

    DEMOFontShutdown();
    DEMOTestShutdown();
    DEMOGfxShutdown();
    DEMOShutdown();

    SUCCEED();
}
