﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
  *--------------------------------------------------------------------------------*/

#include "thread_test.h"
/* The header above includes other headers this file needs:
     #include <stdio.h>
     #include <pthread.h>
     #include <errno.h>
     #include <unistd.h>
*/
#include <stdint.h> /* uint64_t (instead of NX's nn::Bit64) */
#include <sched.h>
#include "ntd-tests/ntd-tests.h"
#include <inttypes.h>

void *affinity_thread_func(void *arg)
{
  int i;
  for( i = 0; i < 10000000; ++i)
  {
  }
  return NULL;
}

/*
 * Test creating a mask, set()-ing the mask, then get()-ing the mask
 */
void test_affinity_basic()
{
  int ret_val;
  int i; /* Loop counter */
  cpu_set_t cpuset_in;
  cpu_set_t cpuset_out;
  pthread_t thread;
  int num_cpus_set; /* Check CPUs set at end of test */
  int cpus_to_set = 8;

  ret_val = pthread_create(&thread, NULL, affinity_thread_func, NULL);
  TESTCASE_MESSAGE(ret_val == 0, "pthread_create() returned %d", ret_val);

  /* Have affinity mask include CPU 0 to 'cpus_to_set' */
  CPU_ZERO(&cpuset_in);
  for (i = 0; i < cpus_to_set; i++)
     CPU_SET(i, &cpuset_in);

  ret_val = pthread_setaffinity_np(thread, sizeof(cpu_set_t), &cpuset_in);
  TESTCASE_MESSAGE(ret_val == 0,
      "pthread_setaffinity_np() failed. Expected %d, got %d\n", 0, ret_val);

  ret_val = pthread_getaffinity_np(thread, sizeof(cpu_set_t), &cpuset_out);
  TESTCASE_MESSAGE(ret_val == 0,
      "pthread_getaffinity_np() failed. Expected %d, got %d\n", 0, ret_val);

  num_cpus_set = CPU_COUNT(&cpuset_out);
  TESTCASE_MESSAGE(num_cpus_set <= cpus_to_set,
      "More CPUs set than expected. Expected between 0 and %d, got %d\n",
      cpus_to_set, num_cpus_set);

  ret_val = pthread_join(thread, NULL);
  TESTCASE_MESSAGE(ret_val == 0, "pthread_join() returned %d", ret_val);
}


/*
 * Test CPU_EQUAL()
 */
void test_CPU_EQUAL()
{
  cpu_set_t cpuset1;
  cpu_set_t cpuset2;

  CPU_ZERO(&cpuset1);
  CPU_ZERO(&cpuset2);

  CPU_SET(0,  &cpuset1);
  CPU_SET(1,  &cpuset1);
  CPU_SET(3,  &cpuset1);
  CPU_SET(5,  &cpuset1);
  CPU_SET(7,  &cpuset1);
  CPU_SET(30, &cpuset1);

  CPU_SET(0,  &cpuset2);
  CPU_SET(1,  &cpuset2);
  CPU_SET(3,  &cpuset2);
  CPU_SET(5,  &cpuset2);
  CPU_SET(7,  &cpuset2);
  CPU_SET(30, &cpuset2);

  TESTCASE_MESSAGE(CPU_EQUAL(&cpuset1, &cpuset2) != 0, "Expected sets to be equal\n");
}

/*
 * Test CPU_ZERO()
 */
void test_CPU_ZERO()
{
  cpu_set_t cpuset;
  int i;
  int num_errors = 0; /* Number of errors that occur */
  int error_val = -1; /* Save error if one occurs */

  CPU_SET(0, &cpuset);
  CPU_SET(1, &cpuset);
  CPU_SET(3, &cpuset);
  CPU_SET(4, &cpuset);
  CPU_SET(5, &cpuset);
  CPU_SET(7, &cpuset);

  CPU_ZERO(&cpuset);
  for (i = 0; i < (sizeof(cpuset.__bits)/sizeof(cpuset.__bits[0])); ++i)
  {
    if (cpuset.__bits[i] != 0)
    {
      if (error_val != -1)
        error_val = cpuset.__bits[i];
      ++num_errors;
    }
  }
  TESTCASE_MESSAGE(num_errors == 0,
      "Expected all values in cpuset to be 0. First nonzero value: %d. Number of nonzero values: %d\n",
      error_val, num_errors);
}

/*
 * Test CPU_SET()
 */
void test_CPU_SET()
{
  cpu_set_t cpuset;
  int ret_val;
  unsigned long set_val = 187;

  CPU_ZERO(&cpuset);

  ret_val = CPU_SET((sizeof(cpu_set_t)*8)+1, &cpuset);
  TESTCASE_MESSAGE(ret_val == 0, "Expected 0, got %d\n", ret_val);

  /* Last operation shouldn't have changed set, so shouldn't be need to CPU_ZERO */
  CPU_SET(0, &cpuset);
  CPU_SET(1, &cpuset);
  CPU_SET(3, &cpuset);
  CPU_SET(4, &cpuset);
  CPU_SET(5, &cpuset);
  CPU_SET(7, &cpuset);
  TESTCASE_MESSAGE(cpuset.__bits[0] == set_val,
      "Expected %lu. Got %lu\n", set_val, cpuset.__bits[0]);
}

/*
 * Test CPU_OR(), CPU_AND(), CPU_XOR()
 */
void test_CPU_AND_OR_XOR()
{
  cpu_set_t cpuset1;
  cpu_set_t cpuset2;
  cpu_set_t cpuset_out;
  unsigned long set1_val = 105;
  unsigned long set2_val = 39;
  unsigned long or_val  = set1_val | set2_val;
  unsigned long and_val = set1_val & set2_val;
  unsigned long xor_val = set1_val ^ set2_val;

  CPU_ZERO(&cpuset1);
  CPU_ZERO(&cpuset2);
  CPU_ZERO(&cpuset_out);

  CPU_SET(0, &cpuset1);
  CPU_SET(3, &cpuset1);
  CPU_SET(5, &cpuset1);
  CPU_SET(6, &cpuset1);
  TESTCASE_MESSAGE(cpuset1.__bits[0] == set1_val,
      "Expected %lu. Got %lu\n", set1_val, cpuset1.__bits[0]);

  CPU_SET(0, &cpuset2);
  CPU_SET(1, &cpuset2);
  CPU_SET(2, &cpuset2);
  CPU_SET(5, &cpuset2);
  TESTCASE_MESSAGE(cpuset2.__bits[0] == set2_val,
      "Expected %lu. Got %lu\n", set2_val, cpuset2.__bits[0]);

  CPU_OR(&cpuset_out, &cpuset1, &cpuset2);
  TESTCASE_MESSAGE(cpuset_out.__bits[0] == or_val,
      "Expected %lu. Got %lu\n", or_val, cpuset_out.__bits[0]);

  CPU_AND(&cpuset_out, &cpuset1, &cpuset2);
  TESTCASE_MESSAGE(cpuset_out.__bits[0] == and_val,
      "Expected %lu. Got %lu\n", and_val, cpuset_out.__bits[0]);

  CPU_XOR(&cpuset_out, &cpuset1, &cpuset2);
  TESTCASE_MESSAGE(cpuset_out.__bits[0] == xor_val,
      "Expected %lu. Got %lu\n", xor_val, cpuset_out.__bits[0]);
}

/*
 * Test CPU_CLR()
 */
void test_CPU_CLR()
{
  cpu_set_t cpuset;
  int i;
  int cpus_to_set = 13;
  int cpu_to_clear = 5;

  for (i = 0; i < cpus_to_set; i++)
     CPU_SET(i, &cpuset);

  for (i = 0; i < cpus_to_set; ++i)
    TESTCASE_MESSAGE(CPU_ISSET(i, &cpuset), "Expected CPU '%d' to be set\n", i);

  CPU_CLR(cpu_to_clear, &cpuset);

  for (i = 0; i < cpus_to_set; ++i)
  {
    if (i == cpu_to_clear)
      TESTCASE_MESSAGE(!CPU_ISSET(i, &cpuset), "Expected CPU '%d' to be cleared\n", i);
    else
      TESTCASE_MESSAGE(CPU_ISSET(i, &cpuset), "Expected CPU '%d' to be set\n", i);
  }
}

/*
 * Test CPU_ALLOC_SIZE()
 */

#define ALIGN_X_TO(x, align) ( (x) + ( (align) - 1 ) ) / (align)

void test_CPU_ALLOC_SIZE()
{
  unsigned long in[] = { 0, 1, 2, 3, 4, 8, 16, 120, 256, 9001, INT_MAX };
  int out;
  int expect;
  int i;

  /* Returned alloc size should be less than the input value (except 0 to 4) */
  for (i = 0; i < (sizeof(in)/sizeof(in[0])); ++i)
  {
    out = CPU_ALLOC_SIZE(in[i]);
    if (sizeof(size_t) == 4) {
      expect = ALIGN_X_TO(in[i], 32) * 4;
    } else {
      expect = ALIGN_X_TO(in[i], 64) * 8;
    }

    TESTCASE_MESSAGE(out == expect,
        "Expected CPU_ALLOC_SIZE(%lu) to return %d, but got %d\n", in[i], expect, out);
  }
}

/*
 * Test CPU_ALLOC() and CPU_FREE()
 */
void test_CPU_ALLOC()
{
  cpu_set_t *cpuset;
  size_t size;
  int cpu; /* Loop counter */
  int num_cpus_set; /* Check at end of test */
  int cpus_to_set = 4;

  cpuset = CPU_ALLOC(cpus_to_set);
  if (cpuset == NULL)
  {
    TESTCASE_MESSAGE(1, "CPU_ALLOC() failed to allocate memory for %d CPUs (alloc size: %d)\n",
        cpus_to_set, CPU_ALLOC_SIZE(cpus_to_set));
    return;
  }

  size = CPU_ALLOC_SIZE(cpus_to_set);

  CPU_ZERO_S(size, cpuset);
  for (cpu = 0; cpu < cpus_to_set; cpu += 2)
     CPU_SET_S(cpu, size, cpuset);

  num_cpus_set = CPU_COUNT_S(size, cpuset);
  TESTCASE_MESSAGE(num_cpus_set == 2,
      "Expected CPU count to be %d. Got %d\n", 2, num_cpus_set);

  CPU_FREE(cpuset);
}

void test_pthread_affinity()
{
    NTD_TEST_GROUP_START("affinity", 3);

    test_affinity_basic();
    test_CPU_ZERO();
    test_CPU_SET();
    test_CPU_EQUAL();
    test_CPU_AND_OR_XOR();
    test_CPU_CLR();
    test_CPU_ALLOC_SIZE();
    test_CPU_ALLOC();

    NTD_TEST_GROUP_END("affinity", 3);
}
