﻿/*******************************************************************************

  This file contains a series of tests for tmpnam()

*******************************************************************************/

#include "fileio_helper_functions.h" /* CloseFile() */
/* The header above also includes files this test uses:
     <stdio.h>  // tmpnam(), L_tmpnam */
#include <string.h> /* strcpy() */
#include "test.h" /* t_error(), t_status */

/* Comment out this header if errno doesn't work on the target hardware */
#include <errno.h> /* errno, strerror() */

#include "malloc_helper_functions.h" /* is_ptr_null() */


/*{{{{------------------------------------------------------------------------*/
/* Test basic functionality
   See if fopen() can open the file using the filename generated by tmpnam() */
void test_basic(const char *testname)
{
  FILE *file;
  char buffer[L_tmpnam];
  char *tmpnam_ptr;

  tmpnam_ptr = tmpnam(buffer);
  if (is_ptr_null_testname(tmpnam_ptr, testname)) return;

  file = fopen(buffer, "w");
  if (file == NULL)
  {
    t_error("fopen() failed to open in function '%s' using filename name "
            "generated by tmpnam(): '%s'. Expected fopen() to succeed\n",
            testname, buffer);
  }
  else
  {
    CloseFile(file, buffer, testname); file = NULL;
    DeleteFile(buffer, testname);
  }
}
/*------------------------------------------------------------------------}}}}*/




/*{{{{------------------------------------------------------------------------*/
/* Test the different methods of obtaining the return value */
void test_return_values(const char *testname)
{
  FILE *file;
  char buffer[L_tmpnam];
  char *passed_in_ptr;
  char *tmpnam_ptr;
  tmpnam_ptr = NULL;
  passed_in_ptr = (char*)malloc(sizeof(char) * L_tmpnam);
  if (is_ptr_null_testname(passed_in_ptr, testname)) return;

  tmpnam_ptr = tmpnam(buffer);
  if (is_ptr_null_testname(tmpnam_ptr, testname)) return;

  TESTCASE_STRINGS_MATCH(tmpnam_ptr, buffer);

  tmpnam_ptr = tmpnam(passed_in_ptr);
  if (is_ptr_null_testname(tmpnam_ptr, testname)) return;

  TESTCASE_STRINGS_MATCH(tmpnam_ptr, passed_in_ptr);

  free(passed_in_ptr);
}
/*------------------------------------------------------------------------}}}}*/




/*{{{{------------------------------------------------------------------------*/
/* Test passing in NULL for the 'str' parameter. That is: tmpnam(NULL)
   It should succeed.
   The C Standard states that:
   "If the argument is a null pointer, the tmpnam function leaves its result
    in an internal static object and returns a pointer to that object
    (subsequent calls to the tmpnam function may modify the same object)" */
void test_null_str(const char *testname)
{
  char *saved_string; /* Save string from tmpnam() for comparison */
  char *str_ret_val; /* Return value from strcpy() */
  char *tmpnam_ptr = NULL; /* Return value from tmpnam() */

  tmpnam_ptr = tmpnam(NULL);
  if (tmpnam_ptr == NULL)
  {
    t_error("tmpnam() failed when NULL was passed into it in function '%s'. "
            "Expected it to succeed\n", testname);
  }

  /* Allocate enough memory to contain the string */
  saved_string = (char*)malloc(sizeof(char) * (strlen(tmpnam_ptr) + 1)); /* +1 for '\0' */
  if (is_ptr_null_testname(saved_string, testname)) return;

  /* Copy the string over to save the contents for comparison later */
  str_ret_val = strcpy(saved_string, tmpnam_ptr);
  if (is_ptr_null_testname(str_ret_val, testname)) return;

  tmpnam_ptr = tmpnam(NULL);
  if (tmpnam_ptr == NULL)
  {
    t_error("tmpnam() failed when NULL was passed into it in function '%s'. "
            "Expected it to succeed\n", testname);
  }

  /* Make sure tmpnam() returned a different string */
  if (strcmp(saved_string, tmpnam_ptr) == 0)
  {
    t_error("tmpnam() returned the same string in function '%s' after subsequent "
            "calls to it while passing in NULL. Returned string: '%s'\n",
            testname, saved_string);
  }

  free(saved_string);
}
/*------------------------------------------------------------------------}}}}*/




/*{{{{------------------------------------------------------------------------*/
/* The C Standard states that the value of TMP_MAX must be at least 25, so
   use this test to verify that
   (ISO/IEC 9899:TC2, May 6, 2005, WG14/N1124)
   http://www.open-std.org/jtc1/sc22/wg14/www/docs/n1124.pdf (page 270) */
void verify_TMP_MAX_value(const char *testname)
{
  if (TMP_MAX < 25)
  {
    t_error("The value of TMP_MAX is less than 25 in function '%s'. C Standard "
            "states it must be 25 or greater. This TMP_MAX equals: %d\n",
            testname, TMP_MAX);
  }
}
/*------------------------------------------------------------------------}}}}*/




/*{{{{------------------------------------------------------------------------*/
/* Helper function to reduce code clutter */
/* The 'max_index' parameter represents number of elements in the array so far,
   and not necessarily the size of the array */
void test_subsequent_calls_helper(char **str_arr, int max_index, FILE **file_arr,
                                  const char *testname)
{
  int i; /* Loop counter */
  char *tmpnam_ptr; /* Return value from tmpnam() */
  char *str_ret_val; /* Return value from strcpy() */
  char buffer[L_tmpnam];

  tmpnam_ptr = tmpnam(buffer);
  if (is_ptr_null_testname(tmpnam_ptr, testname)) return;

  /* Allocate enough memory to contain the string */
  str_arr[max_index] = (char*)malloc(sizeof(char) * (strlen(buffer) + 1)); /* +1 for '\0' */
  if (is_ptr_null_testname(str_arr[max_index], testname)) return;

  /* Basically push_back tmpnam()'s string into the array */
  str_ret_val = strcpy(str_arr[max_index], buffer);
  if (is_ptr_null_testname(str_ret_val, testname)) return;

  /* There should be more than one element in the array before we start looking
     for duplicates */
  if (max_index > 0)
  {
    /* Check to see if the string we just added to the array already exists */
    for (i = 0; i < max_index; ++i)
    {
      if (strcmp(str_arr[i], str_arr[max_index]) == 0)
      {
        t_error("tmpnam() created a duplicate filename in function '%s'. Expected "
                "tmpnam() to always return a unique string. Duplicate string: "
                "'%s'\n", testname, str_arr[i]);
        break;
      }
    }
  }

  /* Now try to open a file with that new filename to make absolutely sure
     that tmpnam() created a unique filename of a non-existing file */
  file_arr[max_index] = fopen(str_arr[max_index], "r");
  if (file_arr[max_index] != NULL)
  {
    t_error("tmpnam() generated a filename that was able to be opened for 'r' "
            "in function '%s', meaning the file already existed, which shouldn't "
            "have happened. Filename was '%s'\n", testname, str_arr[max_index]);
    CloseFile(file_arr[max_index], str_arr[max_index], testname);
    /*DeleteFile(str_arr[max_index], testname);*/
  }
  else /* <--- Comment out this else-statement if errno doesn't work on the target hardware */
  {
    if (errno != ENOENT)
    {
      t_error("fopen() failed for an unexpected reason in function '%s'. Got: "
              "errno = %d, strerror() = '%s'. Expected: errno = %d, strerror() = '%s'. "
              "Filename was '%s'\n",
              testname, errno, strerror(errno), ENOENT, strerror(ENOENT), str_arr[max_index]);
    }
  }

  /* Now open the file for writing so it'll actually be created in order to
     make sure tmpnam() really doesn't try creating an existing filename */
  OpenFile(str_arr[max_index], "w", &file_arr[max_index], testname);
  CloseFile(file_arr[max_index], str_arr[max_index], testname);
}

/* Make sure each subsequent call to tmpnam() returns a different string.
   The C standard says:
   "tmpnam can generate TMP_MAX number of unique filenames, but any or all
    of them might be in use by existing files and thus not be suitable
    return values" */
void test_subsequent_calls(const char *testname)
{
  int i; /* Loop counter */
  char **str_arr; /* Array of all the saved return values from tmpnam() for comparisons */
  FILE **file_arr; /* Array of all the saved file pointers created with tmpnam() filenames */
  char *tmpnam_ptr; /* Return value from tmpnam() */
  int num_files_to_generate; /* Basically number of times to call tmpnam() */

  /* Although the C Standard states that tmpnam() can generate up to TMP_MAX
     number of unique filenames, we'll try to generate at least 25% of that
     total value, since some of the filenames might be in use, which would
     result in a false failure of tmpnam() */
  num_files_to_generate = TMP_MAX / 4;

  /* We don't want the test to take too long */
  if (num_files_to_generate > 100)
    num_files_to_generate = 100;

  /* Allocate enough memory to hold enough pointers to all the strings */
  str_arr = (char**)malloc(sizeof(char*) * num_files_to_generate);
  if (is_ptr_null_testname(str_arr, testname)) return;

  /* Allocate enough memory to hold all the file pointers */
  file_arr = (FILE**)malloc(sizeof(FILE*) * num_files_to_generate);
  if (is_ptr_null_testname(file_arr, testname)) return;

  /* Run the actual test. Generate all the files and filenames */
  for (i = 0; i < num_files_to_generate; ++i)
    test_subsequent_calls_helper(&str_arr[0], i, &file_arr[0], testname);

  /* Clean up the individual string pointers */
  for (i = 0; i < num_files_to_generate; ++i)
  {
    DeleteFile(str_arr[i], testname);
    free(str_arr[i]);
  }

  free(str_arr);
  free(file_arr);
}
/*------------------------------------------------------------------------}}}}*/




/*{{{{------------------------------------------------------------------------*/
/* Test using freopen() to open a file with a filename generated with tmpnam() */
void test_tmpnam_freopen(const char *testname)
{
  FILE *fopen_ptr;
  FILE *freopen_ptr;
  char tmpnam_buffer[L_tmpnam];
  char *tmpnam_ptr; /* Return value from tmpnam() */
  char *fread_buffer;
  const char *fwrite_text;
  fwrite_text = "test tmpnam text";

  tmpnam_ptr = tmpnam(tmpnam_buffer);
  if (is_ptr_null_testname(tmpnam_ptr, testname)) return;

  OpenFile(tmpnam_buffer, "w", &fopen_ptr, testname);
  WriteToFile(fopen_ptr, fwrite_text, testname);

  freopen_ptr = freopen(tmpnam_buffer, "r", fopen_ptr);
  if (is_ptr_null_testname(freopen_ptr, testname)) return;

  ReadEntireFile(freopen_ptr, &fread_buffer, testname);
  TESTCASE_STRINGS_MATCH(fread_buffer, fwrite_text);

  CloseFile(freopen_ptr, tmpnam_buffer, testname);
  DeleteFile(tmpnam_buffer, testname);
}
/*------------------------------------------------------------------------}}}}*/



int main(void)
{
  test_basic("test_basic");
  test_return_values("test_return_values");
  test_null_str("test_null_str");
  verify_TMP_MAX_value("verify_TMP_MAX_value");
  test_subsequent_calls("test_subsequent_calls");
  test_tmpnam_freopen("test_tmpnam_freopen");

  return t_status;
}
