﻿/*******************************************************************************

  This file contains a series of tests for rewind()

*******************************************************************************/

#include "ntd_extended_test.h"
#include <string.h>
#include <stdlib.h>

static void test_rewind_simple_helper(const char *mode)
{
    FILE *file;
    const char *filename;
    const char *write1 = "123";
    const char *write2 = "abc";
    char *fread_buffer;
    size_t fread_result;
    int i;
    filename = HOST_FILENAME("rewind_simple.txt");

    /* fputc() to file, rewind(), read file */
    file = FOPEN_TEST(filename, mode);
    for (i = 0; i < strlen(write1); ++i) {
        int result = fputc(write1[i], file);
        TESTCASE_MESSAGE(result == write1[i],
            "fputc(\'%c\', %p) returned %d, expected %d",
            write1[i], file, result, write1[i]);
    }
    rewind(file);
    NO_FERROR(file);
    FREAD_FILE_TEST(&fread_buffer, file);
    TESTCASE_STRINGS_MATCH(fread_buffer, write1);
    free(fread_buffer);
    FCLOSE_DELETE_TEST(file, filename);

    /* fwrite() to file, rewind(), read file */
    file = FOPEN_TEST(filename, mode);
    FWRITE_STRING_TEST(write2, file);
    rewind(file);
    NO_FERROR(file);
    FREAD_FILE_TEST(&fread_buffer, file);
    TESTCASE_STRINGS_MATCH(fread_buffer, write2);
    free(fread_buffer);
    FCLOSE_DELETE_TEST(file, filename);
}

static void test_rewind_simple()
{
    test_rewind_simple_helper("w+");
    test_rewind_simple_helper("wb+");
    test_rewind_simple_helper("a+");
    test_rewind_simple_helper("ab+");
}

static void test_rewind_feof_reset()
{
    FILE *file;
    const char *filename;
    char buf[256];
    filename = HOST_FILENAME("rewind_feof_reset.txt");

    /* Create the file if it hasn't been created already */
    file = FOPEN_TEST(filename, "w");
    FCLOSE_TEST(file, filename);

    file = FOPEN_TEST(filename, "r");
    int result = fseek(file, 0, SEEK_END);
    TESTCASE_MESSAGE(result == 0, "fseek(%p, 0, SEEK_END) returned %d expected 0",
        file, result);
    size_t read_size = fread(buf, 1, sizeof(buf), file);
    TESTCASE_MESSAGE(read_size == 0, "fread(%p, 1, %d, %p) returned %d expected 0",
        buf, sizeof(buf), file, read_size);
    rewind(file);
    NO_FERROR(file);
    TESTCASE_MESSAGE(feof(file) == 0, "feof(%p) returned %d expected 0",
        file, feof(file));
    FCLOSE_DELETE_TEST(file, filename);
}

static void test_rewind_ferror_reset()
{
    FILE *file;
    const char *filename;
    filename = HOST_FILENAME("rewind_ferror_reset.txt");

    /* Create file in case it doesn't exist yet */
    file = FOPEN_TEST(filename, "w");
    FCLOSE_TEST(file, filename);
    file = NULL;

    file = FOPEN_TEST(filename, "r");
    /* trigger a write error  on a file opened for read */
    const char *str = "123456789";
    size_t write_size = fwrite(str, 1, sizeof(str), file);
    TESTCASE_MESSAGE(write_size == 0 && ferror(file) != 0,
        "fwrite(\"%s\", 1, %d, %p) returned %d ferror(file) returns %d expected failure",
        str, sizeof(str), file, write_size, ferror(file));

    rewind(file);
    NO_FERROR(file);
    FCLOSE_DELETE_TEST(file, filename);
}

static void test_rewind_fgetc()
{
    FILE *file;
    const char *filename = HOST_FILENAME("rewind_fgetc.txt");
    const char *fwrite_text = "Text written to file";
    size_t fwrite_text_size = strlen(fwrite_text) + 1;
    char *fgetc_buffer1;
    char *fgetc_buffer2;
    int c;
    int i;

    fgetc_buffer1 = (char*)malloc(fwrite_text_size);
    fgetc_buffer2 = (char*)malloc(fwrite_text_size);
    TESTCASE(fgetc_buffer1 && fgetc_buffer2);
    if (!fgetc_buffer1 || !fgetc_buffer2) return;

    /* Create file if it doesn't already exist and write some content to it */
    file = FOPEN_TEST(filename, "w");
    FWRITE_STRING_TEST(fwrite_text, file);
    FCLOSE_TEST(file, filename);

    /* Use fgetc() to read the content that was just written */
    file = FOPEN_TEST(filename, "r");
    for (i = 0; !feof(file) && i < fwrite_text_size; ++i)
    {
        c = fgetc(file);
        if (c != EOF) {
            fgetc_buffer1[i] = c;
        }
    }
    fgetc_buffer1[i-1] = '\0';

    rewind(file);
    NO_FERROR(file);

    /* Read the file again using fgetc() after rewind() */
    for (i = 0; !feof(file) && i < fwrite_text_size; ++i)
    {
        c = fgetc(file);
        if (c != EOF) {
            fgetc_buffer2[i] = c;
        }
    }
    fgetc_buffer2[i-1] = '\0';

    TESTCASE_STRINGS_MATCH(fgetc_buffer1, fgetc_buffer2);

    free(fgetc_buffer1);
    free(fgetc_buffer2);

    FCLOSE_DELETE_TEST(file, filename);
}

static void test_multiple_rewind()
{
    FILE *file;
    const char *filename;
    int char_to_write = 'a';
    int fputc_result;
    int fgetc_result;
    filename = HOST_FILENAME("multiple_rewind.txt");

    file = FOPEN_TEST(filename, "w+");
    fputc_result = fputc(char_to_write, file);
    TESTCASE_MESSAGE(fputc_result == char_to_write,
        "fputc('%c', %p) returned %d expected %d",
        char_to_write, file, fputc_result, char_to_write);

    /* Call rewind() more than once in a row */
    rewind(file);
    NO_FERROR(file);
    rewind(file);
    NO_FERROR(file);

    fgetc_result = fgetc(file);
    TESTCASE_MESSAGE(feof(file) == 0, "unexpected end of file");
    TESTCASE_MESSAGE(fgetc_result == char_to_write, "fgetc(%p) expected '%d' got '%d'",
            file, char_to_write, fgetc_result);

    /* Call rewind() more than once in a row again */
    rewind(file);
    NO_FERROR(file);
    rewind(file);
    NO_FERROR(file);
    rewind(file);
    NO_FERROR(file);

    fgetc_result = fgetc(file);
    TESTCASE_MESSAGE(feof(file) == 0, "unexpected end of file");
    TESTCASE_MESSAGE(fgetc_result == char_to_write, "fgetc(%p) expected '%d' got '%d'",
            file, char_to_write, fgetc_result);

    FCLOSE_DELETE_TEST(file, filename);
}

static void test_fputc(char c, FILE *file)
{
    int fputc_result = fputc(c, file);
    TESTCASE_MESSAGE(fputc_result == c,
        "fputc('%c', %p) returned %d expected %d",
        c, file, fputc_result, c);
}

static void test_rewind_fputc_helper(const char *mode,
                              const char *expected_answer)
{
    FILE *file;
    char *fread_buffer;
    char *filename = HOST_FILENAME("rewind_fputc.txt");

    file = FOPEN_TEST(filename, mode);
    test_fputc('1', file);
    test_fputc('2', file);

    rewind(file);
    NO_FERROR(file);

    test_fputc('3', file);
    FCLOSE_TEST(file, filename);

    file = FOPEN_TEST(filename, "r");
    FREAD_FILE_TEST(&fread_buffer, file);

    /* in write mode we expect "12"
     * in append mode we expect "123"
     */
    TESTCASE_STRINGS_MATCH(fread_buffer, expected_answer);

    free(fread_buffer);
    FCLOSE_DELETE_TEST(file, filename);
}

static void test_rewind_fputc()
{
  test_rewind_fputc_helper("w", "32");
  test_rewind_fputc_helper("a", "123");
}

int ntd_extended_rewind(void)
{
    NTD_TEST_GROUP_START("rewind", 2);
    test_rewind_simple();
    test_rewind_feof_reset();
    test_rewind_ferror_reset();
    test_rewind_fgetc();
    test_multiple_rewind();
    test_rewind_fputc();
    return NTD_TEST_GROUP_END("rewind", 2);
}
