﻿/*---------------------------------------------------------------------------*
  Copyright (C)2016 Nintendo Co., Ltd.  All rights reserved.

  These coded instructions, statements, and computer programs contain
  proprietary information of Nintendo of America Inc. and/or Nintendo
  Company Ltd., and are protected by Federal copyright law.  They may
  not be disclosed to third parties or copied or duplicated in any form,
  in whole or in part, without the prior written consent of Nintendo.
 *---------------------------------------------------------------------------*/

/*******************************************************************************

  This file contains a series of tests for open_memstream() and fmemopen()

*******************************************************************************/

#include "ntd_extended_test.h"
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <errno.h>
//#include <wchar.h> /* open_wmemstream */


static void test_seek()
{
    FILE *f = NULL;
    char *buf;
    size_t buf_len;
    int saved_errno = 0;
    long int ftell_ret_val;
    int fseek_ret_val, sprintf_ret_val;

    /* Write text to buffer */
    buf = (char*)MALLOC_TEST(10 * sizeof(char));
    if (buf == NULL) return;
    sprintf_ret_val = sprintf(buf, "abc123");
    TESTCASE_MESSAGE(sprintf_ret_val > 0, "sprintf() failed in %s()", __func__);
    buf_len = strlen(buf);
    TESTCASE_MESSAGE(buf_len > 0, "strlen() returned unexpected value '%d'", buf_len);

    f = FMEMOPEN_TEST(buf, buf_len, "r");
    if (f == NULL) { free(buf); return; }

    ftell_ret_val = FTELL_TEST(f);
    TESTCASE_MESSAGE(ftell_ret_val == 0, "ftell(): expected 0, got %d", ftell_ret_val);

    FSEEK_TEST(f, 0, SEEK_END);

    ftell_ret_val = FTELL_TEST(f);
    TESTCASE_MESSAGE(ftell_ret_val == buf_len, "ftell(): expected %d, got %d",
                     buf_len, ftell_ret_val);

    errno = 0;
    fseek_ret_val = fseek(f, -1, SEEK_SET);
    saved_errno = errno;
    TESTCASE_MESSAGE(fseek_ret_val != 0, "Expected invalid fseek to return non-zero value");
    TESTCASE_MESSAGE(saved_errno == EINVAL, "errno = %d. Expected EINVAL (%d)",
                     saved_errno, EINVAL);

    ftell_ret_val = FTELL_TEST(f);
    TESTCASE_MESSAGE(ftell_ret_val == buf_len,
                    "Didn't expect different file pos after invalid fseek. Expected %d, got %d",
                     buf_len, ftell_ret_val);

    FCLOSE_TEST_NO_NAME(f);
    free(buf);
}


static void test_file_io_funcs_open_memstream()
{
    FILE *f = NULL;
    char *buf;
    size_t buf_len;
    size_t fread_ret_val;
    int saved_errno = 0;
    char *fwrite_str1 = "1234";
    char *fwrite_str2 = "abcdefghi";
    size_t fwrite_str_size1 = strlen(fwrite_str1);
    size_t fwrite_str_size2 = strlen(fwrite_str2);
    char *combined_strs = NULL;
    size_t combined_strs_size;

    combined_strs = (char*)MALLOC_TEST(
                    (fwrite_str_size1 + fwrite_str_size2) * sizeof(char));
    STRCPY_TEST(combined_strs, fwrite_str1);
    STRCAT_TEST(combined_strs, fwrite_str2);
    combined_strs_size = strlen(combined_strs);

    /* Test calling fwrite() on a stream opened with open_memstream() */

    f = OPEN_MEMSTREAM_TEST(&buf, &buf_len);
    if (f == NULL) return;

    FWRITE_STRING_TEST(fwrite_str1, f);
    FFLUSH_TEST(f);

    TESTCASE_STRINGS_MATCH(buf, fwrite_str1);
    TESTCASE_MESSAGE(buf_len == fwrite_str_size1,
                     "Sizes don't match in %s(). Expected %d, got %d",
                     __func__, buf_len, fwrite_str_size1);

    /* Perform an additional fwrite() to make sure open_memstream is able
       to grow the buffer dynamically */

    /* Seek to the end of the string that was just written
       (seeking to "(f, -1, SEEK_CUR)" didn't seem to work) */
    FSEEK_TEST(f, strlen(fwrite_str1), SEEK_SET);

    FWRITE_STRING_TEST(fwrite_str2, f);
    FFLUSH_TEST(f);

    TESTCASE_STRINGS_MATCH(buf, combined_strs);
    TESTCASE_MESSAGE(buf_len == combined_strs_size,
                     "Sizes don't match in %s(). Expected %d, got %d",
                     __func__, buf_len, combined_strs_size);

    /* No need to test fread() with open_memstream() since open_memstream() is
       designed to be write-only. It's implementation-defined whether or not
       fread() will work with open_memstream()
       --> http://stackoverflow.com/questions/30642450/can-i-read-stream-produced-by-open-memstream
    */

    free(buf);
    free(combined_strs);
    FCLOSE_TEST_NO_NAME(f);
}


static void test_file_io_funcs_fmemopen()
{
    FILE *f = NULL;
    size_t fread_ret_val = 0;
    char *fread_buf = NULL;
    int saved_errno = 0;
    char *buf;
    size_t buf_len = 0;
    char *fwrite_str = "1234";
    size_t fwrite_str_size = strlen(fwrite_str);

    /* Test calling fwrite() and fread() on a stream opened with fmemopen() */

    buf = (char*)MALLOC_TEST( (fwrite_str_size+1) * sizeof(char) );
    buf_len = fwrite_str_size+1;
    f = FMEMOPEN_TEST(buf, buf_len, "w");
    if (f == NULL) return;

    FWRITE_STRING_TEST(fwrite_str, f);
    FFLUSH_TEST(f);

    TESTCASE_STRINGS_MATCH(buf, fwrite_str);
    TESTCASE_MESSAGE(strlen(buf) == fwrite_str_size,
                     "Sizes don't match in %s(). Expected %d, got %d",
                     __func__, fwrite_str_size, strlen(buf));

    FCLOSE_TEST_NO_NAME(f);

    f = FMEMOPEN_TEST(buf, buf_len, "r");
    if (f == NULL) return;

    fread_buf = (char*)MALLOC_TEST((fwrite_str_size+1) * sizeof(char));
    if (fread_buf == NULL) { FCLOSE_TEST_NO_NAME(f); return; }

    fread_ret_val = FREAD_TEST(fread_buf, sizeof(char), fwrite_str_size, f);
    fread_buf[fwrite_str_size] = '\0';

    TESTCASE_MESSAGE(fread_ret_val == fwrite_str_size,
        "fread() failed. Expected %d, got %d", fwrite_str_size, fread_ret_val);
    TESTCASE_STRINGS_MATCH(fread_buf, fwrite_str);

    char *x_string = (char*)MALLOC_TEST(strlen(fread_buf)+1 * sizeof(char));
    if (x_string == NULL) { FCLOSE_TEST_NO_NAME(f); free(fread_buf); return; }
    for (int i = 0; i < strlen(fread_buf); ++i)
    {
      fread_buf[i] = 'x';
      x_string[i] = 'x';
    }
    x_string[fwrite_str_size] = '\0';

    /* We haven't seeked back to the beginning of the file, so a second
       fread() should return 0 and not modify fread_buf */

    fread_ret_val = FREAD_TEST(fread_buf, sizeof(char), fwrite_str_size, f);
    fread_buf[fwrite_str_size] = '\0';

    /* Doesn't seem clearly defined that fread() would return 0 at this EOF
       position. Might return 1. So just check if it's less than fwrite_str_size */
    TESTCASE_MESSAGE(fread_ret_val < fwrite_str_size,
        "Expected read at EOF to be less than %d. Got %d", fwrite_str_size, fread_ret_val);
    TESTCASE_MESSAGE(strcmp(fread_buf, fwrite_str) != 0,
        "Expected strings to be different: fread: '%s', fwrite_str = '%s'",
        fread_buf, fwrite_str);

    FSEEK_TEST(f, 0, SEEK_SET);

    /* Now fread() should be able to read the string */

    fread_ret_val = FREAD_TEST(fread_buf, sizeof(char), fwrite_str_size, f);
    TESTCASE_STRINGS_MATCH(fread_buf, fwrite_str);

    TESTCASE_MESSAGE(fread_ret_val == fwrite_str_size,
        "fread() failed. Expected %d, got %d", fwrite_str_size, fread_ret_val);
    TESTCASE_STRINGS_MATCH(fread_buf, fwrite_str);

    free(x_string);
    free(fread_buf);
    free(buf);
    FCLOSE_TEST_NO_NAME(f);
}


void test_EINVAL_open_memstream()
{
    /* current MUSL implementation requires that both pointers passed to memstream be non-null
     * passing an invalid pointer will just crash.
     */
#if 0
    /* open_memstream must return EINVAL if 'buf' or 'size' are NULL */
    FILE *f;
    char *buf = NULL;
    size_t buf_len = 10;
    int saved_errno = 0;

    errno = 0;
    f = open_memstream(NULL, NULL);
    saved_errno = errno;

    //TESTCASE_MESSAGE(saved_errno == EINVAL,
    //    "Expected errno to be %d (%s). Got %d (%s)",
    //    EINVAL, strerror(EINVAL), saved_errno, strerror(saved_errno));
    HORIZON_KNOWN_FAILURE_MESSAGE("RYNDA-340", saved_errno == EINVAL,
        "open_memstream() should return EINVAL if passed 'NULL' arg");

    errno = 0;
    f = open_memstream(&buf, NULL);
    saved_errno = errno;

    //TESTCASE_MESSAGE(saved_errno == EINVAL,
    //    "Expected errno to be %d (%s). Got %d (%s)",
    //    EINVAL, strerror(EINVAL), saved_errno, strerror(saved_errno));
    HORIZON_KNOWN_FAILURE_MESSAGE("RYNDA-340", saved_errno == EINVAL,
        "open_memstream() should return EINVAL if passed 'NULL' arg");

    errno = 0;
    f = open_memstream(NULL, &buf_len);
    saved_errno = errno;

    //TESTCASE_MESSAGE(saved_errno == EINVAL,
    //    "Expected errno to be %d (%s). Got %d (%s)",
    //    EINVAL, strerror(EINVAL), saved_errno, strerror(saved_errno));
    HORIZON_KNOWN_FAILURE_MESSAGE("RYNDA-340", saved_errno == EINVAL,
        "open_memstream() should return EINVAL if passed 'NULL' arg");

#endif
}



void test_EINVAL_fmemopen()
{
    /* fmemopen() should allocate a buffer if 'buf' is NULL. */

    FILE *f;
    size_t buf_len = 5;

    errno = 0;
    f = fmemopen(NULL, buf_len, "w");
    if (f != NULL) {
        fclose(f);
        return;
}
    TESTCASE_MESSAGE(f != NULL, "fmemopen(NULL, %d, \"%s\") returned %p expected not NULL", buf_len, f);
}


void test_fmemopen_null_buf()
{
    /* Passing NULL for 'buf' in fmemopen() *and* a mode with a '+' basically
       turns fmemopen() into a call to malloc, except you don't have a pointer
       to the buffer but rather a pointer to the FILE stream */

    FILE *f;
    size_t buf_len;
    char *fread_buf;
    char *fwrite_str = "1234";
    size_t fwrite_str_size = strlen(fwrite_str);

    buf_len = 10;

    f = FMEMOPEN_TEST(NULL, buf_len, "w+");

    size_t fwrite_ret_val =
        FWRITE_TEST(fwrite_str, sizeof(char), fwrite_str_size, f);

    TESTCASE_MESSAGE(fwrite_ret_val == fwrite_str_size,
        "fwrite() returned %d. Expected %d", fwrite_ret_val, fwrite_str_size);

    FSEEK_TEST(f, 0, SEEK_SET);

    fread_buf = (char*)MALLOC_TEST((fwrite_str_size+1) * sizeof(char));
    size_t fread_ret_val = FREAD_TEST(fread_buf, sizeof(char), fwrite_str_size, f);
    fread_buf[fwrite_str_size] = '\0';

    TESTCASE_MESSAGE(fread_ret_val == fwrite_str_size,
        "fread() returned %d. Expected %d", fread_ret_val, fwrite_str_size);
    TESTCASE_STRINGS_MATCH(fread_buf, fwrite_str);

    FCLOSE_TEST_NO_NAME(f);
    free(fread_buf);
}


void test_open_wmemstream()
{
    FILE *open_wmemstream(wchar_t **bufp, size_t *sizep);

    FILE *f;
    wchar_t *buf;
    size_t buf_len;
    int wmemcmp_ret_val;
    wchar_t *fwrite_str = L"abcd";
    size_t fwrite_str_size = wcslen(fwrite_str);
    int saved_errno = 0;

    errno = 0;
    f = open_wmemstream(&buf, &buf_len);
    saved_errno = errno;
    TESTCASE_MESSAGE(saved_errno == 0, "Expected errno to be 0. Got %d (%s)",
        saved_errno, strerror(saved_errno));

    int fwprintf_ret_val = fwprintf(f, L"%ls", fwrite_str);
    FFLUSH_TEST(f);
    buf_len = wcslen(buf);

    TESTCASE_MESSAGE(fwprintf_ret_val == fwrite_str_size,
        "Expected fwrite() to return %d. Got %d", fwrite_str_size, fwprintf_ret_val);
    TESTCASE_MESSAGE(buf_len == fwrite_str_size,
        "Expected fwrite() to return %d. Got %d", fwrite_str_size, buf_len);

    wmemcmp_ret_val = wmemcmp(fwrite_str, buf, fwrite_str_size);

    TESTCASE_MESSAGE(wmemcmp_ret_val == 0,
        "Wide char strings don't match. Expected wmemcmp() to return 0. Got %d",
        wmemcmp_ret_val);

    FCLOSE_TEST_NO_NAME(f);
}


void test_fmemopen_fprintf()
{
    FILE *f = NULL;
    size_t fread_ret_val = 0;
    int fprintf_ret_val = 0;
    char *fread_buf = NULL;
    int saved_errno = 0;
    char *buf;
    size_t buf_len = 0;
    char *fprintf_str = "1234";
    size_t fprintf_str_size = strlen(fprintf_str);

    /* Test calling fprintf() on a stream opened with fmemopen() */

    buf = (char*)MALLOC_TEST( (fprintf_str_size+1) * sizeof(char) );
    buf_len = fprintf_str_size+1;
    f = FMEMOPEN_TEST(buf, buf_len, "w");
    if (f == NULL) return;

    errno = 0;
    fprintf_ret_val = fprintf(f, "%s", fprintf_str);
    FFLUSH_TEST(f);

    TESTCASE_STRINGS_MATCH(buf, fprintf_str);
    TESTCASE_MESSAGE(strlen(buf) == fprintf_str_size,
                     "Sizes don't match in %s(). Expected %d, got %d",
                     __func__, fprintf_str_size, strlen(buf));

    FCLOSE_TEST_NO_NAME(f);

    f = FMEMOPEN_TEST(buf, buf_len, "r");
    if (f == NULL) return;

    fread_buf = (char*)MALLOC_TEST((fprintf_str_size+1) * sizeof(char));
    if (fread_buf == NULL) { FCLOSE_TEST_NO_NAME(f); return; }

    fread_ret_val = FREAD_TEST(fread_buf, sizeof(char), fprintf_str_size, f);
    fread_buf[fprintf_str_size] = '\0';

    TESTCASE_MESSAGE(fread_ret_val == fprintf_str_size,
        "fread() failed. Expected %d, got %d", fprintf_str_size, fread_ret_val);
    TESTCASE_STRINGS_MATCH(fread_buf, fprintf_str);

    free(fread_buf);
    free(buf);
    FCLOSE_TEST_NO_NAME(f);
}


/*
void test_fmemopen_null_mode()
{
    // Standard says "Return EINVAL if the value of the 'mode' argument is
    // not valid." Doesn't specify if 'mode' is NULL.
    // It's not required for some functions, like fmemopen or fprintf, to handle
    // being passed NULL pointers. So this test ends up not being needed

    FILE *f;
    char *buf;
    size_t buf_len;
    char *str = "1234";
    size_t str_size = strlen(str);
    int saved_errno = 0;

    buf = (char*)MALLOC_TEST( (str_size+1) * sizeof(char) );
    buf_len = str_size+1;

    errno = 0;
    f = FMEMOPEN_TEST(buf, buf_len, NULL);
    saved_errno = errno;

    TESTCASE_MESSAGE(saved_errno == EINVAL,
        "Expected invalid fmemopen() to set errno to EINVAL (%d, '%s'). Got %d (%s)",
        EINVAL, strerror(EINVAL), saved_errno, strerror(saved_errno));
}
*/


int ntd_extended_memstream(void)
{
    NTD_TEST_GROUP_START("memstream", 2);

    test_seek();
    test_file_io_funcs_open_memstream();
    test_file_io_funcs_fmemopen();
    test_EINVAL_open_memstream();
    test_EINVAL_fmemopen();
    test_fmemopen_null_buf();
    test_open_wmemstream();
    test_fmemopen_fprintf();

    /* test_fmemopen_null_mode(); <-- Test isn't needed */

    return NTD_TEST_GROUP_END("memstream", 2);
}
