﻿/*---------------------------------------------------------------------------*
  Copyright (C)2015 Nintendo Co., Ltd.  All rights reserved.

  These coded instructions, statements, and computer programs contain
  proprietary information of Nintendo of America Inc. and/or Nintendo
  Company Ltd., and are protected by Federal copyright law.  They may
  not be disclosed to third parties or copied or duplicated in any form,
  in whole or in part, without the prior written consent of Nintendo.
 *---------------------------------------------------------------------------*/

/* NTD-TEST-LIBC */

/* define _BSD_SOURCE to allow strdup */
#define _BSD_SOURCE 1
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <stdbool.h>
#include <unistd.h>
#include <stdarg.h>
#include <errno.h>
#include <stdint.h>
#include "ntd-test-libc.h"

#define NUM_TEST_GROUPS 100
#define NUM_TEST_LEVELS 10

typedef struct test_group {
    const char *name;
    int         level;
    int         passed;
    int         failed;
} test_group;

static test_group       results[NUM_TEST_GROUPS] = {0};
static int              test_group_for_level[NUM_TEST_LEVELS] = {0};
static int current_test_group = -1;
static int current_test_group_level = -1;
static const char *pad = "          ";

void NTD_TEST_START(void)
{
    printf("STARTING NTD-TEST-LIBC\n\n");
}

void NTD_TEST_GROUP_START(const char *name, int level)
{
    size_t pad_len = strlen(pad);
    current_test_group++;
    if (current_test_group > NUM_TEST_GROUPS) {
        printf("TOO MANY TEST GROUPS INCREASE NUM_TEST_GROUPS\n");
        exit(1);
    }
    results[current_test_group].name = name;
    results[current_test_group].level = level;
    results[current_test_group].passed = 0;
    results[current_test_group].failed = 0;

    test_group_for_level[level] = current_test_group;
    current_test_group_level = level;
    printf("%s%-*s\n",
           &pad[pad_len - level],
           20-level,
           name);
}

int NTD_TEST_GROUP_END(const char *name, int level)
{
    if (strcmp(name,results[test_group_for_level[level]].name) != 0) {
        printf("\n\n!!! NTD_TEST_GROUP_END name '%s' does not match '%s'\n\n\n",
               name,
               results[current_test_group].name
               );
    }
    if (level > current_test_group_level) {
        printf("\n\n!!! NTD_TEST_GROUP_END name '%s' level is greater than %d, found %d\n\n\n",
               name,
               current_test_group_level,
               level
               );
    }
    return results[current_test_group].failed;
}

const char *short_path(const char *file)
{
    const char *partial_path = strstr(file, "LibcTest/");
    if (partial_path) {
        return partial_path + 9;
    }
    partial_path = strstr(file, "LibcTest\\");
    if (partial_path) {
        return partial_path + 9;
    }
    return file;
}

int NTD_TESTCASE_NOMESSAGE(bool passed)
{
    if (passed) {
        results[current_test_group].passed++;
    } else {
        results[current_test_group].failed++;
    }
    return (int) passed;
}

static char *last_message = NULL;
static int repeat_count = 0;

static void append_errno_if_not_zero()
{
    if (errno != 0) {
        printf(", errno = %d(%s)\n", errno, strerror(errno));
    } else {
        printf("\n");
    }
    errno = 0;
}

typedef struct KnownFailures {
    const char *msg;
    bool        passed;
    const char *file;
    int         line;
    const char *expression;
    int         count;
} KnownFailures;

#define MAX_KNOWN_FAILURES 5000

static int known_failures_seen = 0;
static KnownFailures known_failures[MAX_KNOWN_FAILURES];

void NTD_TESTCASE_KNOWN_FAILURE_MESSAGE(const char *msg, bool passed, const char *file, int line, const char *expression, const char *message, ...)
{
    char *new_message = NULL;

    int i;
    for (i = 0; i < known_failures_seen; i++) {
        if (known_failures[i].msg == msg && known_failures[i].passed == passed) {
            known_failures[i].count++;
            return;
        }
    }
    if (message) {
        va_list ap;
        va_start(ap, message);
        vasprintf(&new_message, message, ap);
        va_end(ap);
    }
    if (known_failures_seen < MAX_KNOWN_FAILURES) {
        i = known_failures_seen;
        known_failures_seen++;
        known_failures[i].msg = msg;
        known_failures[i].passed = passed;
        known_failures[i].file = short_path(file);
        known_failures[i].line = line;
        if (new_message) {
            char *new_expr = NULL;
            asprintf(&new_expr, "%s - %s", expression, new_message);
            known_failures[i].expression = new_expr;
        } else {
            known_failures[i].expression = strdup(expression);
        }
        ntd_test_ignore_allocation(known_failures[i].expression);
        known_failures[i].count = 1;
    } else {
        NTD_TESTCASE_MESSAGE(passed, file, line, expression, "%s", new_message);
    }
    if (new_message != NULL) {
        free(new_message);
    }
}

void NTD_TESTCASE_KNOWN_FAILURE(const char *msg, bool passed, const char *file, int line, const char *expression)
{
    NTD_TESTCASE_KNOWN_FAILURE_MESSAGE(msg, passed, file, line, expression, NULL);
}

int NTD_TESTCASE_RESULT(int passed, const char *file, int line, const char *expression)
{
    NTD_TESTCASE(passed, file, line, expression);
    return passed;
}

int NTD_CURRENT_FAIL_COUNT(void)
{
    return results[current_test_group].failed;
}

int NTD_TESTCASE_MESSAGE(bool passed, const char *file, int line, const char *expression, const char *message, ...)
{
    va_list ap;
    if (passed) {
        results[current_test_group].passed++;
    } else {
        char *newmessage = NULL;
        results[current_test_group].failed++;
        asprintf(&newmessage,  "FAILED: %s, line %d - %s", short_path(file), line, expression);

        if (last_message != NULL && !strcmp(newmessage, last_message)) {
            free(newmessage);
            repeat_count++;
        } else {
            if (repeat_count > 0) {
                printf("REPEATED: previous message repeated %d times\n", repeat_count);
            }
            if (message) {
                va_start(ap, message);
                char *new_message;
                vasprintf(&new_message, message, ap);
                printf("%s (%s)", newmessage, new_message);
                if (new_message) {
                    free(new_message);
                }
            } else {
                printf("%s", newmessage);
            }
            append_errno_if_not_zero();
            if (last_message != NULL) free(last_message);
            last_message = newmessage;
            repeat_count = 0;
        }
    }
    return passed;
}

void NTD_TESTCASE(bool passed, const char *file, int line, const char *expression)
{
    NTD_TESTCASE_MESSAGE(passed, file, line, expression, NULL);
}

void NTD_TESTWARNING(const char *func, const char *message, ...)
{
    va_list ap;
    va_start(ap, message);
    char *new_message;
    vasprintf(&new_message, message, ap);
    printf( "WARNING: %s,%s %s\n", results[current_test_group].name, func, new_message );
    if (new_message) {
        free(new_message);
    }
}

void NTD_TEST_END(void)
{
    int i;
    int total_failed = 0;
    int total_passed = 0;
    size_t pad_len = strlen(pad);
    printf("FINISHED NTD-TEST-LIBC");
    printf("\n\nSUMMARY:\n");
    for (i = 0; i <= current_test_group; i++) {
        if (results[i].passed || results[i].failed) {
            total_failed += results[i].failed;
            total_passed += results[i].passed;

            printf("%s%-*s PASSED: % 7d",
                &pad[pad_len - results[i].level],
                24-results[i].level,
                results[i].name,
                results[i].passed);
            if (results[i].failed) {
                printf(" FAILED: % 4d\n", results[i].failed);
            } else {
                printf("\n");
            }
        } else {
            printf("%s%-*s\n",
                   &pad[pad_len - results[i].level],
                   24-results[i].level,
                   results[i].name
                   );
        }
    }
    printf("======================== ===============");
    if (total_failed > 0) {
        printf(" ============");
    }
    printf("\n%-24s PASSED: % 7d",
           "TOTAL",
           total_passed
    );
    if (total_failed > 0) {
        printf(" FAILED: % 4d\n", total_failed);
    } else {
        printf("\n");
    }
    if (known_failures_seen > 0) {
        printf("\n\n");
        for (i = 0; i < known_failures_seen; i++) {
            KnownFailures *fail = &known_failures[i];
            if (!fail->passed) {
                if (fail->count > 1) {
                    printf("BUG %s, %d REPEATS FIRST AT: ",
                        fail->msg, fail->count);
                } else {
                     printf("BUG %s: ",
                        fail->msg);

                }
                printf("%s, line %d -  %s\n",
                    short_path(fail->file),
                    fail->line,
                    fail->expression
                    );
            }
        }
        printf("\n\n");
        for (i = 0; i < known_failures_seen; i++) {
            KnownFailures *fail = &known_failures[i];
            if (fail->passed) {
                if (fail->count > 1) {
                    printf("BUG %s FIXED, %d REPEATS FIRST AT: ",
                        fail->msg, fail->count);
                } else {
                     printf("BUG %s FIXED: ",
                        fail->msg);

                }
                printf("%s, line %d -  %s\n",
                    short_path(fail->file),
                    fail->line,
                    fail->expression
                    );
            }
        }
    }
    printf("\n");
}

void NTD_RUN_TESTS(testfunc_ptr tests[], const char *names[], int count)
{
    int i;
    int n = count;

    for (i = 0; i < n; i++) {
        int curr_fails = NTD_CURRENT_FAIL_COUNT();
        int result = tests[i]();
#if 1
if ((result && curr_fails == NTD_CURRENT_FAIL_COUNT()))
printf("%s: result = %d, curr_fails = %d, NTD_CURRENT_FAIL_COUNT() = %d\n",
    names[i],
    result,
    curr_fails,
    NTD_CURRENT_FAIL_COUNT());
#endif
        TESTCASE_MESSAGE(
            !(result && curr_fails == NTD_CURRENT_FAIL_COUNT()),
            names[i]);
    }
}

#if __NX__
#define NTD_TEST_TMPFILE_NAME "host:/ntd-test-tmpfile"
#else
#define NTD_TEST_TMPFILE_NAME "ntd-test-tmpfile"
#endif
FILE *NTD_TEST_TMPFILE(const char *func_name)
{
    char *name;
    static int unique = 0;
    asprintf(&name, NTD_TEST_TMPFILE_NAME "-%s-%04d", func_name, unique++);
    if (!name) return NULL;
    if (access(name, F_OK)) remove(name);
    FILE *file = fopen(name, "w+");
    free(name);
    return file;
}

void delete_if_exists(const char *src_file, int line, const char *filename)
{
    if (!access(filename, F_OK)) {
        int result = remove(filename);
        NTD_TESTCASE_MESSAGE(result == 0, src_file, line, "result == 0",
            "remove(\"%s\") failed result = %d", filename, result);
    }
}

FILE *fopen_test(const char *src_file, int line, const char *filename, const char *mode)
{
    FILE *file = fopen(filename, mode);

    NTD_TESTCASE_MESSAGE(file != NULL, src_file, line, "file != NULL",
        "fopen(\"%s\", \"%s\") returned %p expected not NULL", filename, mode, file);
    return file;
}

FILE *freopen_test(const char *src_file, int line, const char *filename, const char *mode, FILE *infile)
{
    FILE *file = freopen(filename, mode, infile);

    NTD_TESTCASE_MESSAGE(file != NULL, src_file, line, "file != NULL",
        "freopen(\"%s\", \"%s\", %p) returned %p expected not NULL", filename, mode, infile, file);
    return file;
}

size_t fwrite_test(const char *src_file, int line, const char *str, size_t size, size_t nitems, FILE *file)
{
    NTD_TESTCASE_MESSAGE(file != NULL, src_file, line,
        "'file' passed in as NULL in %s()\n", __func__);
    size_t result = 0;
    if (file != NULL)
    {
        result = fwrite(str, size, nitems, file);
        NTD_TESTCASE_MESSAGE(result != 0, src_file, line,
            "fwrite(\"%s\", %d, %d, %p) returned %d",
            str, size, nitems, file, result);
    }
    return result;
}

size_t fwrite_string_test(const char *src_file, int line, const char *str, FILE *file)
{
    return fwrite_test(src_file, line, str, 1, strlen(str), file);
}

size_t fread_test(const char *src_file, int line, char *str, size_t size, size_t nitems, FILE *file)
{
    NTD_TESTCASE_MESSAGE(file != NULL, src_file, line,
        "'file' passed in as NULL in %s()\n", __func__);
    size_t result = 0;
    if (file != NULL)
    {
        result = fread(str, size, nitems, file);
        NTD_TESTCASE_MESSAGE(result != 0, src_file, line, "result != 0",
            "fread(%p, %d, %d, %p) returned %d",
            str, size, nitems, file, result);
    }
    return result;
}

size_t fread_file_test(const char *src_file, int line, char **out_data, FILE *file)
{
    NTD_TESTCASE_MESSAGE(file != NULL, src_file, line,
        "'file' passed in as NULL in %s()\n", __func__);
    *out_data = NULL;
    if (file != NULL) {
        int result = fseek(file, 0, SEEK_END);
        NTD_TESTCASE_MESSAGE(result == 0, src_file, line,
            "result == 0", "fseek(%p, 0, SEEK_END) returned %d",
            file, result);
        long size = ftell(file);
        NTD_TESTCASE_MESSAGE(size >= 0, src_file, line, "size >= 0",
            "ftell(%p) returned %ld expected >= 0",
            file, size);
        char *file_buf = malloc(size + 1);
        NTD_TESTCASE_MESSAGE(file_buf != NULL, src_file, line, "file_buf != NULL",
            "malloc(%d) returned %p expected not NULL",
            size+1, file_buf);
        if (!file_buf) return 0;
        result = fseek(file, 0, SEEK_SET);
        NTD_TESTCASE_MESSAGE(result == 0, src_file, line,
            "result == 0", "fseek(%p, 0, SEEK_SET) returned %d",
            file, result);
        size_t read_size = fread(file_buf, 1, size, file);
        NTD_TESTCASE_MESSAGE(read_size != 0, src_file, line, "read_size != 0",
            "fread(\"%p\", 1, %d, %p) returned %d",
            file_buf, size, file, read_size);
        file_buf[size] = '\0';
        *out_data = file_buf;
        return read_size;
    } else {
        NTD_TESTCASE_MESSAGE(file != NULL, src_file, line, "file != NULL",
            "fread_file_test(%p, file = %p) file is NULL", out_data, file);
        return 0;
    }
}

int fclose_test(const char *src_file, int line, FILE *file, const char *filename, int delete_after_closing)
{
    int result = 0; /* fclose() return value */
    if (file != NULL)
    {
        result = fclose(file);
        NTD_TESTCASE_MESSAGE(result == 0, src_file, line, "result == 0",
            "fclose(%p) failed, result = %d", file, result);
        file = NULL;
    }
    if (delete_after_closing) DELETE_IF_EXISTS(filename);
    return result;
}

void create_file_test(const char *src_file, int line, const char *filename, const char *string)
{
    int result = 0;

    if (access(filename, F_OK)) {
        FILE *file = fopen_test(src_file, line, filename, "w");
        if (file) {
            fwrite_string_test(src_file, line, string, file);
            result = fclose_test(src_file, line, file, filename, 0);
            TESTCASE_MESSAGE(result == 0, "fclose(%p) failed result = %d", file, result);
        }
    }
    errno = 0;
}

long int ftell_test(const char *src_file, int line, FILE *file)
{
    long int result = 0; /* ftell() return value */

    NTD_TESTCASE_MESSAGE(file != NULL, src_file, line,
        "'file' passed in as NULL in %s()\n", __func__);

    if (file != NULL)
    {
      result = ftell(file);
      NTD_TESTCASE_MESSAGE(result != -1L, src_file, line, "result != -1L",
          "ftell(%p) failed", file);
    }
    return result;
}

int fseek_test(const char *src_file, int line, FILE *file, long int offset, int origin)
{
    long int result = 0; /* fseek() return value */

    NTD_TESTCASE_MESSAGE(file != NULL, src_file, line,
        "'file' passed in as NULL in %s()\n", __func__);

    if (file != NULL)
    {
      result = fseek(file, offset, origin);
      NTD_TESTCASE_MESSAGE(result == 0, src_file, line, "result == 0",
          "fseek(%p) failed", file);
    }

    return result;
}

int fflush_test(const char *src_file, int line, FILE *file)
{
    int result = 0; /* fflush() return value */

    NTD_TESTCASE_MESSAGE(file != NULL, src_file, line,
        "'file' passed in as NULL in %s()\n", __func__);

    result = fflush(file);
    NTD_TESTCASE_MESSAGE(result == 0, src_file, line, "result == 0",
        "fflush(%p) failed", file);

    return result;
}

FILE *fmemopen_test(const char *src_file, int line, char *buf, size_t buf_len, const char *mode)
{
    FILE *file = fmemopen(buf, buf_len, mode);
    NTD_TESTCASE_MESSAGE(file != NULL, src_file, line, "file != NULL",
        "fmemopen('%s') failed", mode);

    return file;
}

FILE *open_memstream_test(const char *src_file, int line, char **buf, size_t *buf_len)
{
    NTD_TESTCASE_MESSAGE(buf != NULL, src_file, line,
        "'buf' passed in as NULL in %s()\n", __func__);
    NTD_TESTCASE_MESSAGE(buf_len != NULL, src_file, line,
        "'buf_len' passed in as NULL in %s()\n", __func__);

    FILE *file = open_memstream(buf, buf_len);
    NTD_TESTCASE_MESSAGE(file != NULL, src_file, line, "file != NULL",
        "open_memstream(%p) failed", buf);

    return file;
}

void *malloc_test(const char *src_file, int line, size_t size)
{
    void *data = malloc(size);
    NTD_TESTCASE_MESSAGE(data != NULL, src_file, line, "data != NULL",
        "malloc(%d) failed", size);

    return data;
}

void *calloc_test(const char *src_file, int line, size_t count, size_t size)
{
    void *data = calloc(count, size);
    NTD_TESTCASE_MESSAGE(data != NULL, src_file, line, "data != NULL",
        "calloc(%d) failed", size);

    return data;
}

char *strcpy_test(const char *src_file, int line, char *dest, char *src)
{
    char *result; /* strcpy() return value */

    NTD_TESTCASE_MESSAGE(dest != NULL, src_file, line,
        "'dest' passed in as NULL in %s()\n", __func__);
    NTD_TESTCASE_MESSAGE(src != NULL, src_file, line,
        "'src' passed in as NULL in %s()\n", __func__);

    result = strcpy(dest, src);
    NTD_TESTCASE_MESSAGE(result != NULL, src_file, line,
        "strcpy() returned NULL in %s()\n", __func__);

    return result;
}

char *strcat_test(const char *src_file, int line, char *dest, char *src)
{
    char *result; /* strcat() return value */

    NTD_TESTCASE_MESSAGE(dest != NULL, src_file, line,
        "'dest' passed in as NULL in %s()\n", __func__);
    NTD_TESTCASE_MESSAGE(src != NULL, src_file, line,
        "'src' passed in as NULL in %s()\n", __func__);

    result = strcat(dest, src);
    NTD_TESTCASE_MESSAGE(result != NULL, src_file, line,
        "strcat() returned NULL in %s()\n", __func__);

    return result;
}

void addmillisecondsfromnow(struct timespec * time, int ms)
{
    const int64_t NS_PER_MS = 1000000;
    const int64_t NS_PER_S  = 1000000000;

    clock_gettime(CLOCK_REALTIME, time);

    int64_t nsec = time->tv_nsec + (ms * NS_PER_MS);
    if (nsec >= NS_PER_S) {
        nsec -= NS_PER_S;
        time->tv_sec++;
    }
    time->tv_nsec = nsec;
}

static int quick_exit_count = 0;

static void rynda_382_first(void)
{
    NTD_TEST_GROUP_START("quick_exit", 1);

    TESTCASE_MESSAGE(quick_exit_count == 0, "expected 0, got %d",
        quick_exit_count);
    quick_exit_count++;
}

static void rynda_382_second(void)
{
    TESTCASE_MESSAGE(quick_exit_count == 1, "expected 0, got %d",
        quick_exit_count);
    quick_exit_count++;
}

static void rynda_382_final(void)
{
    TESTCASE_MESSAGE(quick_exit_count == 2, "expected 0, got %d",
        quick_exit_count);
    quick_exit_count++;
    NTD_TEST_GROUP_END("quick_exit", 1);
    free(last_message);
    last_message = NULL;
    leak_test();
    /* call the final test summary here */
    NTD_TEST_END();
}

static void rynda_382(void)
{
    int result = 0;
    /* The main function for NTD-created tests */
    NTD_TEST_GROUP_START("at_quick_exit", 0);

    // at_quick_exit functions run in reverse order so install in reverse of order desired
    result = at_quick_exit(rynda_382_final);
    TESTCASE_MESSAGE(result == 0, "at_quick_exit(rynda_382_final) failed %d",
            result);
    result = at_quick_exit(rynda_382_second);
    TESTCASE_MESSAGE(result == 0, "at_quick_exit(rynda_382_second) failed %d",
            result);
    result = at_quick_exit(rynda_382_first);
    TESTCASE_MESSAGE(result == 0, "at_quick_exit(rynda_382_first) failed %d",
            result);

    NTD_TEST_GROUP_END("at_quick_exit", 0);
}

extern void pdclib_tests(void);
extern void jira_regression_tests(void);
extern void ntd_tests();
extern void musl_libc_tests();

void testsuite_main(void)
{
    NTD_TEST_START();

    rynda_382();

    jira_regression_tests();
    ntd_tests();
    pdclib_tests();
    musl_libc_tests();
    // leak_test(); called in rynda_382_final

    /* call quick_exit() to test rynda_382 */
    quick_exit(0);
    /* NTD_TEST_END() should be called as a quick_exit() cleanup routine */
    NTD_TEST_END();
}

