﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <stdlib.h>
#include <stdio.h>
#include <dirent.h>
#include <errno.h>
#include <stddef.h>
#include <string.h>
#include <unistd.h>

#include <sys/stat.h>
#include <sys/types.h>

#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/fs.h>
#include <nn/os.h>

#include <pthread.h>
#include <regex.h>
#include <search.h>
#include <wchar.h>
#include <fcntl.h>
#include <libintl.h>

#include "tests.h"

/******************************************************
Function Name: LMRTestFixture.bindtextdomain
Description  : Verifies bindtextdomain doesn't use more memory than it used to
******************************************************/
void* bindtextdomainBytes = malloc(36);
TEST_F(LMRTestFixture, bindtextdomain)
{
    const char* domainname = "bar";
    const char* dirname = "foobarthing";

    free(bindtextdomainBytes);
    const char* ret = bindtextdomain(domainname, dirname);

    EXPECT_STREQ(ret, dirname);
}

/******************************************************
Function Name: LMRTestFixture.fdopen
Description  : Verifies fdopen doesn't use more memory than it used to
******************************************************/
TEST_F(LMRTestFixture, fdopen)
{
    int fd;
    FILE* f;

#if NN_BUILD_TARGET_PLATFORM_ADDRESS_64
    const int BIG_MALLOC = 1256;
#elif NN_BUILD_TARGET_PLATFORM_ADDRESS_32
    const int BIG_MALLOC = 1176;
#else
#error Unsupported platform type
#endif

    FreeMallocMemory(BIG_MALLOC);
    fd = open(TEST_PATH_FILE1, O_WRONLY | O_CREAT | O_TRUNC);
    EXPECT_GE(fd, 3);

    EXPECT_NE(f = fdopen(fd, "r"), (void*)NULL) << " errno = " << strerror(errno) << "\n";

    fclose(f);
    ReallocMallocMemory(BIG_MALLOC);
}

/******************************************************
Function Name: LMRTestFixture.insert
Description  : Verifies insert doesn't use more memory than it used to
******************************************************/
int test_insert_cmp(const void* left, const void* right)
{
    return *(int*)left - *(int*)right;
}

TEST_F(LMRTestFixture, insert)
{
    int a = 1;
    int b = 2;
    int c = 3;
    void* root = (void*)NULL;

#if NN_BUILD_TARGET_PLATFORM_ADDRESS_64
    const int MALLOC1 = 30;
    const int MALLOC2 = 31;
    const int MALLOC3 = 32;
#elif NN_BUILD_TARGET_PLATFORM_ADDRESS_32
    const int MALLOC1 = 14;
    const int MALLOC2 = 15;
    const int MALLOC3 = 16;
#else
#error Unsupported platform type
#endif

    FreeMallocMemory(MALLOC1);
    int** retA = (int**)tsearch(&a, &root, test_insert_cmp);
    FreeMallocMemory(MALLOC2);
    int** retB = (int**)tsearch(&b, &root, test_insert_cmp);
    FreeMallocMemory(MALLOC3);
    int** retC = (int**)tsearch(&c, &root, test_insert_cmp);
    EXPECT_NE(retA, (void*)NULL);
    EXPECT_NE(retB, (void*)NULL);
    EXPECT_NE(retC, (void*)NULL);
    EXPECT_EQ(**retA, a);
    EXPECT_EQ(**retB, b);
    EXPECT_EQ(**retC, c);

    tdelete(&a, &root, test_insert_cmp);
    tdelete(&b, &root, test_insert_cmp);
    tdelete(&c, &root, test_insert_cmp);

    ReallocMallocMemory(MALLOC1);
    ReallocMallocMemory(MALLOC2);
    ReallocMallocMemory(MALLOC3);
}

/******************************************************
Function Name: LMRTestFixture.open_memstream
Description  : Verifies open_memstream doesn't use more memory than it used to
******************************************************/
TEST_F(LMRTestFixture, open_memstream)
{
    FILE* f;
    char* s = 0;
    size_t l;

// open_memstream() added an additional call to malloc()
// Currently, the way the test and allocator work, when running the 64-bit test
// in the low-memory environment the test creates, that malloc() call succeeds
// only if it can allocate a full pool (4096)
#if NN_BUILD_TARGET_PLATFORM_ADDRESS_64
    const int BIG_MALLOC = 2721;
    //const int BIG_MALLOC = 1256;
#elif NN_BUILD_TARGET_PLATFORM_ADDRESS_32
    const int BIG_MALLOC = 1192;
#else
#error Unsupported platform type
#endif

    FreeMallocMemory(BIG_MALLOC);
    EXPECT_NE(f = open_memstream(&s, &l), (void*)NULL);
    FreeMallocMemory(15);
    EXPECT_EQ(putc('a', f), 'a');
    EXPECT_EQ(putc('b', f), 'b');
    EXPECT_EQ(putc('c', f), 'c');
    EXPECT_EQ(putc('d', f), 'd');
    EXPECT_EQ(putc('e', f), 'e');
    EXPECT_EQ(putc('f', f), 'f');
    EXPECT_EQ(putc('g', f), 'g');
    EXPECT_EQ(putc('h', f), 'h');
    EXPECT_EQ(putc('i', f), 'i');
    EXPECT_EQ(putc('j', f), 'j');
    EXPECT_EQ(putc('k', f), 'k');
    EXPECT_EQ(putc('l', f), 'l');
    EXPECT_EQ(putc('m', f), 'm');
    EXPECT_EQ(putc('n', f), 'n');
    EXPECT_EQ(fflush(f), 0);
    fclose(f);

    EXPECT_STREQ(s, "abcdefghijklmn");
    free(s);

    ReallocMallocMemory(15);
    ReallocMallocMemory(BIG_MALLOC);
}

/******************************************************
Function Name: LMRTestFixture.open_wmemstream
Description  : Verifies open_wmemstream doesn't use more memory than it used to
******************************************************/
TEST_F(LMRTestFixture, open_wmemstream)
{
    FILE* f;
    wchar_t* s = 0;
    size_t l;
    wchar_t fwrite_str[] = L"ab";
    size_t fwrite_str_size = wcslen(fwrite_str);

// open_wmemstream() added an additional call to malloc()
// Currently, the way the test and allocator work, when running the 64-bit test
// in the low-memory environment the test creates, that malloc() call succeeds
// only if it can allocate a full pool (4096)
#if NN_BUILD_TARGET_PLATFORM_ADDRESS_64
    const int BIG_MALLOC = 2721;
    //const int BIG_MALLOC = 280;
#elif NN_BUILD_TARGET_PLATFORM_ADDRESS_32
    const int BIG_MALLOC = 184;
    //const int BIG_MALLOC = 176;
#else
#error Unsupported platform type
#endif

    FreeMallocMemory(BIG_MALLOC);
    EXPECT_NE(f = open_wmemstream(&s, &l), (void*)NULL);
    FreeMallocMemory(12);
    EXPECT_EQ(fwprintf(f, L"%ls", fwrite_str), fwrite_str_size);
    EXPECT_EQ(fflush(f), 0);
    fclose(f);

    NN_LOG("s = %ls\n", s);
    EXPECT_EQ(wmemcmp(s, fwrite_str, fwrite_str_size), 0);
    free(s);

    ReallocMallocMemory(12);
    ReallocMallocMemory(BIG_MALLOC);
}

/******************************************************
Function Name: LMRTestFixture.pthread_create
Description  : Verifies pthread_create doesn't use more memory than it used to
******************************************************/
void* test_pthread_create_thread(void* arg)
{
    return 0;
}

void* pthread_createBytes = malloc(520);
TEST_F(LMRTestFixture, pthread_create)
{
    pthread_t thread;
    int arg;

    FreeMallocMemory(90112);
    FreeMallocMemory(1072);
    //FreeMallocMemory(520);
    free(pthread_createBytes);

    EXPECT_EQ(pthread_create(&thread, NULL, test_pthread_create_thread, &arg), 0);

    EXPECT_EQ(pthread_join(thread, NULL), 0);

    ReallocMallocMemory(90112);
    ReallocMallocMemory(1072);
    //ReallocMallocMemory(520); Due to RYNDA-579, 520 bytes are not freed
}

/******************************************************
Function Name: LMRTestFixture.putenv
Description  : Verifies putenv doesn't use more memory than it used to
******************************************************/
TEST_F(LMRTestFixture, putenv)
{
    char* empty = NULL;
    if (!environ)
        environ = &empty;

    EXPECT_EQ(clearenv(), 0) << "clearenv returned " << strerror(errno) << "\n";
    if (environ != NULL)
        EXPECT_EQ(*environ, (void*)NULL);

    FreeMallocMemory(8);

    char text[] = "TEST=1";
    EXPECT_EQ(putenv(text), 0) << "putenv returned " << strerror(errno) << "\n";
    EXPECT_STREQ(environ[0], text);

    // Due to a bug in clearenv, we have to free this ourselves
    // Note:  Since no one uses these functions, the bug might never be fixed
    free(environ);
    ReallocMallocMemory(8);
}

/******************************************************
Function Name: LMRTestFixture.regcomp
Description  : Verifies regcomp doesn't use more memory than it used to
******************************************************/
TEST_F(LMRTestFixture, regcomp)
{
    char buf[200];
    char pat[] = "a\\0";
    regex_t r;
    int res;

#if NN_BUILD_TARGET_PLATFORM_ADDRESS_64
    const int MALLOC1 = 4096;
#elif NN_BUILD_TARGET_PLATFORM_ADDRESS_32
    const int MALLOC1 = 2048;
#else
#error Unsupported platform type
#endif

#if NN_BUILD_TARGET_PLATFORM_ADDRESS_64
    FreeMallocMemory(1023);
#endif

    FreeMallocMemory(MALLOC1);
    FreeMallocMemory(1024);
    FreeMallocMemory(336);
    FreeMallocMemory(112);
    FreeMallocMemory(104);
    FreeMallocMemory(48);
    FreeMallocMemory(24);
    FreeMallocMemory(16);
    FreeMallocMemory(15);
    FreeMallocMemory(14);
    FreeMallocMemory(13);
    FreeMallocMemory(8);
    FreeMallocMemory(7);
    FreeMallocMemory(6);
    FreeMallocMemory(5);

    res = regcomp(&r, pat, 0);
    if (res)
    {
        regerror(res, &r, buf, sizeof(buf));
        ADD_FAILURE() << "regcomp(" << pat << ") returned " << res << " (" << buf << ") wanted 0\n";
    }

#if NN_BUILD_TARGET_PLATFORM_ADDRESS_64
    ReallocMallocMemory(1023);
#endif

    ReallocMallocMemory(MALLOC1);
    ReallocMallocMemory(1024);
    ReallocMallocMemory(48);
    ReallocMallocMemory(24);
    ReallocMallocMemory(14);
    ReallocMallocMemory(13);
    ReallocMallocMemory(6);
    ReallocMallocMemory(5);

    regfree(&r);

    ReallocMallocMemory(336);
    ReallocMallocMemory(112);
    ReallocMallocMemory(104);
    ReallocMallocMemory(16);
    ReallocMallocMemory(15);
    ReallocMallocMemory(8);
    ReallocMallocMemory(7);
}

/******************************************************
Function Name: LMRTestFixture.regexec
Description  : Verifies regexec doesn't use more memory than it used to
******************************************************/
TEST_F(LMRTestFixture, regexec)
{
    char buf[200];
    char pat[] = "a\\0";
    regex_t r;
    int res;

#if NN_BUILD_TARGET_PLATFORM_ADDRESS_64
    const int MALLOC1 = 4096;
    const int MALLOC2 = 204;
#elif NN_BUILD_TARGET_PLATFORM_ADDRESS_32
    const int MALLOC1 = 2048;
    const int MALLOC2 = 100;
#else
#error Unsupported platform type
#endif

#if NN_BUILD_TARGET_PLATFORM_ADDRESS_64
    FreeMallocMemory(1023);
#endif

    FreeMallocMemory(MALLOC1);
    FreeMallocMemory(1024);
    FreeMallocMemory(336);
    FreeMallocMemory(112);
    FreeMallocMemory(104);
    FreeMallocMemory(48);
    FreeMallocMemory(24);
    FreeMallocMemory(20);
    FreeMallocMemory(16);
    FreeMallocMemory(15);
    FreeMallocMemory(14);
    FreeMallocMemory(13);
    FreeMallocMemory(12);
    FreeMallocMemory(8);
    FreeMallocMemory(7);
    FreeMallocMemory(6);

    res = regcomp(&r, pat, 0);
    if (res)
    {
        regerror(res, &r, buf, sizeof(buf));
        ADD_FAILURE() << "regcomp(" << pat << ") returned " << res << " (" << buf << ") wanted 0\n";
    }

    ReallocMallocMemory(MALLOC1);
    ReallocMallocMemory(1024);
    ReallocMallocMemory(48);
    ReallocMallocMemory(24);
    ReallocMallocMemory(20);
    ReallocMallocMemory(14);
    ReallocMallocMemory(13);
    ReallocMallocMemory(12);
    ReallocMallocMemory(6);

    FreeMallocMemory(MALLOC2);

    res = regexec(&r, "a0", 0, 0, 0);
    if (res)
    {
        regerror(res, &r, buf, sizeof(buf));
        ADD_FAILURE() << "regexec(/" << pat << "/ ~ \"a0\") returned " << res << " (" << buf << ") wanted 0\n";
    }

    ReallocMallocMemory(MALLOC2);

#if NN_BUILD_TARGET_PLATFORM_ADDRESS_64
    ReallocMallocMemory(1023);
#endif

    regfree(&r);

    ReallocMallocMemory(336);
    ReallocMallocMemory(112);
    ReallocMallocMemory(104);
    ReallocMallocMemory(16);
    ReallocMallocMemory(15);
    ReallocMallocMemory(8);
    ReallocMallocMemory(7);
}

/******************************************************
Function Name: LMRTestFixture.scandir
Description  : Verifies scandir doesn't use more memory than it used to
******************************************************/
TEST_F(LMRTestFixture, scandir)
{
    struct dirent** namelist = NULL;
    errno = 0;

#if NN_BUILD_TARGET_PLATFORM_ADDRESS_64
    const int BIG_MALLOC = 2104;
#elif NN_BUILD_TARGET_PLATFORM_ADDRESS_32
    const int BIG_MALLOC = 2096;
#else
#error Unsupported platform type
#endif

    FreeMallocMemory(BIG_MALLOC);
    FreeMallocMemory(280);
    FreeMallocMemory(279);
    FreeMallocMemory(38);
    FreeMallocMemory(24);
    FreeMallocMemory(4);

    int count = scandir(TEST_PATH, &namelist, NULL, NULL);
    EXPECT_EQ(count, 2) << "errno = " << errno << "\n";
    EXPECT_NE(namelist, (void*)NULL) << "errno = " << errno << "\n";

    EXPECT_STREQ(namelist[0]->d_name, "file1.txt");
    EXPECT_STREQ(namelist[1]->d_name, "file2.txt");

    free(namelist[0]);
    free(namelist[1]);
    free(namelist);

    ReallocMallocMemory(BIG_MALLOC);
    ReallocMallocMemory(280);
    ReallocMallocMemory(279);
    ReallocMallocMemory(38);
    ReallocMallocMemory(24);
    ReallocMallocMemory(4);
}

/******************************************************
Function Name: LMRTestFixture.setenv
Description  : Verifies setenv doesn't use more memory than it used to
******************************************************/
TEST_F(LMRTestFixture, setenv)
{
    char text[] = "TEST";
    char value[] = "2";

    char* empty = NULL;
    if (!environ)
        environ = &empty;

    EXPECT_EQ(clearenv(), 0) << "clearenv returned " << strerror(errno) << "\n";
    if (environ != NULL)
        EXPECT_EQ(*environ, (void*)NULL);

    FreeMallocMemory(8);
    FreeMallocMemory(7);
    FreeMallocMemory(6);

    EXPECT_EQ(setenv(text, value, 0), 0) << "setenv returned " << strerror(errno) << "\n";
    EXPECT_STREQ(getenv(text), value);

    // Due to a bug in clearenv, we have to free this ourselves
    // Note:  Since no one uses these functions, the bug might never be fixed
    free(environ[0]);
    free(environ);

    ReallocMallocMemory(6);
    ReallocMallocMemory(7);
    ReallocMallocMemory(8);
}

/******************************************************
Function Name: LMRTestFixture.strdup
Description  : Verifies strdup doesn't use more memory than it used to
******************************************************/
TEST_F(LMRTestFixture, strdup)
{
    const char* teststr  = "Hello, world!!!";
    char* testres;

    FreeMallocMemory(16);
    EXPECT_NE(testres = strdup(teststr), (void*)NULL);
    EXPECT_STREQ(testres, teststr);

    free(testres);
    ReallocMallocMemory(16);
}

/******************************************************
Function Name: LMRTestFixture.strndup
Description  : Verifies strndup doesn't use more memory than it used to
******************************************************/
TEST_F(LMRTestFixture, strndup)
{
    const char* teststr  = "Hello, world!!!";
    char* testres;

    FreeMallocMemory(16);
    EXPECT_NE(testres = strndup(teststr, strlen(teststr)), (void*)NULL);
    EXPECT_STREQ(testres, teststr);

    free(testres);
    ReallocMallocMemory(16);
}

/******************************************************
Function Name: LMRTestFixture.textdomain
Description  : Verifies textdomain doesn't use more memory than it used to
******************************************************/
TEST_F(LMRTestFixture, textdomain)
{
    char text[] = "I can't let you do that Dave.";

    FreeMallocMemory(256);
    char* res = textdomain(text);
    EXPECT_STREQ(text, res);

    free(res);
    ReallocMallocMemory(256);
}

/******************************************************
Function Name: LMRTestFixture.vasprintf
Description  : Verifies vasprintf doesn't use more memory than it used to
******************************************************/
void test_vasprintf_func(char** dest, const char* message, ...)
{
    va_list ap;
    va_start(ap, message);
    vasprintf(dest, message, ap);
    va_end(ap);
}

TEST_F(LMRTestFixture, vasprintf)
{
    char* buffer;
    char text[] = "Hello, again!!!";

    FreeMallocMemory(16);
    test_vasprintf_func(&buffer, text);
    EXPECT_STREQ(buffer, text);

    free(buffer);
    ReallocMallocMemory(16);
}

/******************************************************
Function Name: LMRTestFixture.vfscanf
Description  : Verifies vfscanf doesn't use more memory than it used to
******************************************************/
void test_vfscanf_func(FILE* stream, const char* message, ...)
{
    va_list ap;
    va_start(ap, message);
    vfscanf(stream, message, ap);
    va_end(ap);
}

TEST_F(LMRTestFixture, vfscanf)
{
    FILE* f;
    int fd;
    char fwrite_str[] = "ab";
    char buffer[100] = "";

    FreeMallocMemory(1256);
    EXPECT_NE(f = fopen(TEST_PATH_FILE1, "w"), (void*)NULL);
    fwrite(fwrite_str, 1, strlen(fwrite_str), f);
    fflush(f);
    fclose(f);

    EXPECT_NE(f = fopen(TEST_PATH_FILE1, "r"), (void*)NULL);
    test_vfscanf_func(f, "%2c", buffer);
    fclose(f);
    ReallocMallocMemory(1256);

    EXPECT_STREQ(buffer, fwrite_str);
}

/******************************************************
Function Name: LMRTestFixture.vfwscanf
Description  : Verifies vfwscanf doesn't use more memory than it used to
******************************************************/
void test_vfwscanf_func(FILE* stream, const wchar_t* message, ...)
{
    va_list ap;
    va_start(ap, message);
    vfwscanf(stream, message, ap);
    va_end(ap);
}

TEST_F(LMRTestFixture, vfwscanf)
{
    FILE* f;
    int fd;
    char fwrite_str[] = "ab";
    size_t fwrite_str_size = strlen(fwrite_str);
    char buffer[100] = "";

    FreeMallocMemory(1256);
    EXPECT_NE(f = fopen(TEST_PATH_FILE1, "w"), (void*)NULL);
    EXPECT_EQ(fwprintf(f, L"%s", fwrite_str), fwrite_str_size);
    fflush(f);
    fclose(f);

    EXPECT_NE(f = fopen(TEST_PATH_FILE1, "r"), (void*)NULL);
    test_vfwscanf_func(f, L"%2c", buffer);
    fclose(f);
    ReallocMallocMemory(1256);

    EXPECT_STREQ(buffer, fwrite_str);
}

/******************************************************
Function Name: LMRTestFixture.wcsdup
Description  : Verifies wcsdup doesn't use more memory than it used to
******************************************************/
TEST_F(LMRTestFixture, wcsdup)
{
    const wchar_t* teststr  = L"Hello, Dave";
    wchar_t* testres;

    FreeMallocMemory(48);
    EXPECT_NE(testres = wcsdup(teststr), (void*)NULL);
    EXPECT_EQ(wmemcmp(testres, teststr, wcslen(teststr)), 0) << "testres = " << testres << " (expected " << teststr << ")\n";

    free(testres);
    ReallocMallocMemory(48);
}
