﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Abort.h>
#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>
#include <nn/fs/fs_Directory.h>
#include <nn/fs/fs_File.h>
#include <nn/fs/fs_FileSystem.h>
#include <nn/fs/fs_Host.h>
#include <nn/fs/fs_Result.h>

#include <nnt/gtest/detail/gtest-cstring.h>
#include <nnt/gtest/detail/gtest-heap.h>
#include <nnt/gtest/detail/gtest-host.h>

namespace nnt { namespace testing { namespace detail {

namespace {

class FileSystem final
{
    NN_DISALLOW_COPY(FileSystem);
    NN_DISALLOW_MOVE(FileSystem);

private:
    int m_Count;

    bool m_Mounts;

public:
    FileSystem() NN_NOEXCEPT : m_Count(0), m_Mounts(false) { /* 何もしない */ }

    ~FileSystem() NN_NOEXCEPT { /* 何もしない */ }

    static FileSystem& Get() NN_NOEXCEPT
    {
        static FileSystem fileSystem;

        return fileSystem;
    }

    bool Mount() NN_NOEXCEPT
    {
        if (m_Count == 0)
        {
            const ::nn::Result result = ::nn::fs::MountHostRoot();

            m_Mounts = result.IsSuccess();

            if (result.IsFailure() &&
                !::nn::fs::ResultMountNameAlreadyExists::Includes(result))
            {
                return false;
            }
        }

        ++m_Count;

        return true;
    }

    void Unmount() NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_GREATER(m_Count, 0);

        --m_Count;

        if (m_Count == 0 && m_Mounts)
        {
            ::nn::fs::UnmountHostRoot();
        }
    }
};

class Directory final
{
    NN_DISALLOW_COPY(Directory);
    NN_DISALLOW_MOVE(Directory);

public:
    NNT_TESTING_DETAIL_HEAP_IS_ALLOCATABLE();

private:
    bool m_IsOpen;

    ::nn::fs::DirectoryHandle m_Handle;

public:
    Directory() NN_NOEXCEPT : m_IsOpen(false) { /* 何もしない */ }

    ~Directory() NN_NOEXCEPT
    {
        if (m_IsOpen) { this->Close(); }
    }

    bool Open(const char* path) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(path);

        NN_SDK_REQUIRES(!m_IsOpen);

        if (!FileSystem::Get().Mount())
        {
            return false;
        }

        m_IsOpen = ::nn::fs::OpenDirectory(
            &m_Handle, path, ::nn::fs::OpenDirectoryMode_All).IsSuccess();

        if (!m_IsOpen)
        {
            FileSystem::Get().Unmount();
        }

        return m_IsOpen;
    }

    void Close() NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(m_IsOpen);

        ::nn::fs::CloseDirectory(m_Handle);

        FileSystem::Get().Unmount();

        m_IsOpen = false;
    }
};

class File final
{
    NN_DISALLOW_COPY(File);
    NN_DISALLOW_MOVE(File);

public:
    NNT_TESTING_DETAIL_HEAP_IS_ALLOCATABLE();

private:
    bool m_IsOpen;
    bool m_NeedsFlush;

    ::nn::fs::FileHandle m_Handle;

public:
    File() NN_NOEXCEPT
        : m_IsOpen(false)
        , m_NeedsFlush(false)
    {
        // 何もしない
    }

    ~File() NN_NOEXCEPT
    {
        if (m_IsOpen)
        {
            this->Close();
        }
    }

    bool Open(const char* path, const char* mode) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(path);
        NN_SDK_REQUIRES_NOT_NULL(mode);
        NN_SDK_REQUIRES(!m_IsOpen);

        if (!FileSystem::Get().Mount())
        {
            return false;
        }

        const int openMode = GetOpenMode(mode);

        if ((openMode & ::nn::fs::OpenMode_Write) != 0)
        {
            const ::nn::Result result = ::nn::fs::CreateFile(path, 0);

            if (result.IsFailure() &&
                !::nn::fs::ResultPathAlreadyExists::Includes(result))
            {
                FileSystem::Get().Unmount();

                return false;
            }

            m_NeedsFlush = true;
        }

        m_IsOpen = ::nn::fs::OpenFile(&m_Handle, path, openMode).IsSuccess();

        if (!m_IsOpen)
        {
            FileSystem::Get().Unmount();
        }

        return m_IsOpen;
    }

    void Close() NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(m_IsOpen);

        if (m_NeedsFlush)
        {
            ::nn::fs::FlushFile(m_Handle);

            m_NeedsFlush = false;
        }

        ::nn::fs::CloseFile(m_Handle);

        FileSystem::Get().Unmount();

        m_IsOpen = false;
    }

    bool Write(int offset, const void* buffer, int size) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_GREATER_EQUAL(offset, 0);
        NN_SDK_REQUIRES_NOT_NULL(buffer);
        NN_SDK_ASSERT_GREATER_EQUAL(size, 0);

        NN_SDK_REQUIRES(m_IsOpen);

        return ::nn::fs::WriteFile(
            m_Handle, offset, buffer, static_cast<size_t>(size),
            ::nn::fs::WriteOption::MakeValue(0)).IsSuccess();
    }

    bool GetSize(int* pOutValue) const NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pOutValue);

        NN_SDK_REQUIRES(m_IsOpen);

        int64_t value = 0;

        if (::nn::fs::GetFileSize(&value, m_Handle).IsFailure())
        {
            return false;
        }
        else
        {
            *pOutValue = static_cast<int>(value);

            return true;
        }
    }

    bool SetSize(int value) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(m_IsOpen);

        return ::nn::fs::SetFileSize(m_Handle, value).IsSuccess();
    }

    bool Flush() NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(m_IsOpen);

        return ::nn::fs::FlushFile(m_Handle).IsSuccess();
    }

private:
    static int GetOpenMode(const char* mode) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(mode);

        if (::nnt::testing::detail::CString::Strcmp(mode, "r") == 0)
        {
            return ::nn::fs::OpenMode_Read;
        }
        else if (::nnt::testing::detail::CString::Strcmp(mode, "w") == 0)
        {
            return ::nn::fs::OpenMode_Write;
        }
        else if (::nnt::testing::detail::CString::Strcmp(mode, "a") == 0)
        {
            return ::nn::fs::OpenMode_Write | ::nn::fs::OpenMode_AllowAppend;
        }
        else
        {
            NN_ABORT("Unexpected file access mode: %s", mode);
        }
    }
};

} // namespace

bool Host::IsDirectory(const char* path) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(path);

    return Directory().Open(path);
}

bool Host::IsFile(const char* path) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(path);

    return File().Open(path, "r");
}

bool Host::Exists(const char* path) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(path);

    if (!FileSystem::Get().Mount())
    {
        return false;
    }

    bool exists = IsDirectory(path);

    if (!exists)
    {
        exists = IsFile(path);
    }

    FileSystem::Get().Unmount();

    return exists;
}

bool Host::CreateDirectory(const char* path) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(path);

    if (!FileSystem::Get().Mount())
    {
        return false;
    }

    const ::nn::Result result = ::nn::fs::CreateDirectory(path);

    FileSystem::Get().Unmount();

    return result.IsSuccess();
}

void* Host::OpenFile(const char* path, const char* mode) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(path);
    NN_SDK_REQUIRES_NOT_NULL(mode);

    auto pFile = new File();

    NN_SDK_ASSERT_NOT_NULL(pFile);

    if (!pFile->Open(path, mode))
    {
        delete pFile;

        return nullptr;
    }

    return pFile;
}

bool Host::WriteFile(
    void* handle, int offset, const void* buffer, int size) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(handle);
    NN_SDK_ASSERT_GREATER_EQUAL(offset, 0);
    NN_SDK_REQUIRES_NOT_NULL(buffer);
    NN_SDK_ASSERT_GREATER_EQUAL(size, 0);

    auto pFile = reinterpret_cast<File*>(handle);

    return pFile->Write(offset, buffer, size);
}


bool Host::FlushFile(void* handle) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(handle);

    auto pFile = reinterpret_cast<File*>(handle);

    return pFile->Flush();
}

bool Host::GetFileSize(void* handle, int* pOutValue) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(handle);
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);

    auto pFile = reinterpret_cast<File*>(handle);

    return pFile->GetSize(pOutValue);
}

bool Host::SetFileSize(void* handle, int value) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(handle);
    NN_SDK_ASSERT_GREATER_EQUAL(value, 0);

    auto pFile = reinterpret_cast<File*>(handle);

    return pFile->SetSize(value);
}

void Host::CloseFile(void* handle) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(handle);

    auto pFile = reinterpret_cast<File*>(handle);

    pFile->Close();

    delete pFile;
}

}}} // namespace nnt::testing::detail
