﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_SdkAssert.h>

#include <nnt/gtest/detail/gtest-cstring.h>
#include <nnt/gtest/detail/gtest-heap.h>

namespace nnt { namespace testing { namespace detail {

char* CString::Strncpy(char* ptr, const char* str, size_t num) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(ptr);
    NN_SDK_REQUIRES_NOT_NULL(str);
    size_t i = 0;
    for (;i < num; ++i)
    {
        if (!str[i]) { break; }
        ptr[i] = str[i];
    }
    for (;i < num; ++i)
    {
        ptr[i] = '\0';
    }
    return ptr;
}

namespace {

int CompareChar(char lhs, char rhs)
{
    if (lhs > rhs)
    {
        return 1;
    }
    else if (lhs < rhs)
    {
        return -1;
    }
    else
    {
        return 0;
    }
}

} // namespace

int CString::Strcmp(const char* lhs, const char* rhs) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(lhs);
    NN_SDK_REQUIRES_NOT_NULL(rhs);

    while ((*lhs == *rhs) && (*lhs) && (*rhs))
    {
        ++lhs;
        ++rhs;
    }

    return CompareChar(*lhs, *rhs);
}

int CString::Strncmp(const char* lhs, const char* rhs, size_t num) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(lhs);
    NN_SDK_REQUIRES_NOT_NULL(rhs);

    if (num == 0) { return 0; }

    for (size_t i = 1; (i < num) && (*lhs == *rhs) && (*lhs) && (*rhs); ++i)
    {
        ++lhs;
        ++rhs;
    }

    return CompareChar(*lhs, *rhs);
}

namespace {

char GetLowerCase(char ch)
{
    if ('A' <= ch && ch <= 'Z')
    {
        return ch + ('a' - 'A');
    }
    else
    {
        return ch;
    }
}

} // namespace

int CString::Strcasecmp(const char* lhs, const char* rhs) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(lhs);
    NN_SDK_REQUIRES_NOT_NULL(rhs);

    while ((GetLowerCase(*lhs) == GetLowerCase(*rhs)) && (*lhs) && (*rhs))
    {
        ++lhs;
        ++rhs;
    }

    return CompareChar(GetLowerCase(*lhs), GetLowerCase(*rhs));
}

const char* CString::Strchr(const char *str, int character) NN_NOEXCEPT
{
    for (;; ++str)
    {
        if (*str == character) { return str; }
        if (*str == '\0') { return nullptr; }
    }
}

const char* CString::Strrchr(const char* str, int character) NN_NOEXCEPT
{
    const char* ptr = nullptr;
    for (;; ++str)
    {
        if (*str == character) { ptr = str; }
        if (*str == '\0') { return ptr; }
    }
}

const char *CString::Strstr(
    const char* haystack, const char* needle) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(haystack);
    NN_SDK_REQUIRES_NOT_NULL(needle);

    const auto needleCount = CString::Strlen(needle);
    if (needleCount == 0)
    {
        return haystack;
    }

    const auto haystackCount = CString::Strlen(haystack);
    if (haystackCount < needleCount)
    {
        return nullptr;
    }

    const auto jumpCount = 1 << (sizeof(char) * 8);
    auto *jumps =
        static_cast<size_t*>(Heap::Allocate(sizeof(size_t) * jumpCount));
    NN_SDK_ASSERT_NOT_NULL(jumps);
    for (auto i = 0u; i < jumpCount; ++i)
    {
        jumps[i] = needleCount;
    }

    const auto lastIndex = needleCount - 1;

    for (auto i = 0u; i < lastIndex; ++i)
    {
        jumps[static_cast<unsigned char>(needle[i])] = lastIndex - i;
    }

    const char *p = nullptr;

    for (size_t i = 0u; i < haystackCount;
         i += jumps[static_cast<unsigned char>(haystack[i + lastIndex])])
    {
        auto j = static_cast<int>(lastIndex);
        while (j >= 0 && (haystack[i + j] == needle[j])) { --j; }
        if (j < 0)
        {
            p = &haystack[i + j + 1];
            break;
        }
    }

    Heap::Free(jumps);

    return p;
}


void* CString::Memset(void* ptr, int value, size_t num) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(ptr);
    auto p = static_cast<char*>(ptr);
    for (size_t i = 0; i < num; ++i) { p[i] = static_cast<char>(value); }
    return ptr;
}

size_t CString::Strlen(const char* str) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(str);
    size_t count = 0;
    while (str[count]) { ++count; }
    return count;
}

}}} // namespace nnt::testing::detail
