﻿#pragma once

#include "cdr.h"

namespace nn {

namespace cdr {

//------------------------------------------------------------------------------
/**
* @brief        Reads the memory of the nxdmp file at the specified address.
* @param[out]   pOutBuffer The preallocated buffer to hold the memory read request.
* @param[out]   pOutAmountRead The number of bytes read.
* @param[in]    address The Target memory address to read.
* @param[in]    bufferSize The size of the preallocated buffer, in bytes.
*
* @result       nn::cdr::Error_Ok if successful, else an error code.
*/
Error           ReadMemory( void* pOutBuffer, uint32_t* pOutAmountRead, uint64_t address, uint32_t bufferSize );

// Screenshot types
enum ScreenshotType
{
    ScreenshotType_Raw = 0,
    ScreenshotType_BMP = 1,
    ScreenshotType_JPG = 2,
    ScreenshotType_PNG = 3,
};


//------------------------------------------------------------------------------
/**
* @brief        Gets the thread information for the currently running thread.
* @param[out]   pOutThreadInfo Thread information of the current thread.
*
* @result       nn::cdr::Error_Ok if successful, else an error code.
*/
CDR_DLL_DECL Error GetCurrentThread( Thread* pOutThreadInfo );


//------------------------------------------------------------------------------
/**
* @brief        Gets the Application ID set by the nmeta file of the nsp.
* @param[out]   pOutApplicationID The application id.
*
* @result       nn::cdr::Error_Ok if successful, else an error code.
*/
CDR_DLL_DECL Error GetApplicationId( uint64_t* pOutApplicationID);

//------------------------------------------------------------------------------
/**
* @brief        Set the path to where screenshots will be stored to.
* @param[out]   pPath directory path of where to store the screenshots to.
*
* @result       nn::cdr::Error_Ok if successful, else an error code.
*/
CDR_DLL_DECL Error SetScreenshotDirectoryPath( const char* pPath );

//------------------------------------------------------------------------------
/**
* @brief        Gets the images embedded in the crash dump and stores them.
*               By default they are stored relative to host executable.
*               A screenshot dir can be set prior to this using
*               SetScreenshotDirectoryPath()
*
* @result       nn::cdr::Error_Ok if successful, else an error code.
*/
CDR_DLL_DECL Error ExtractImages( );

//------------------------------------------------------------------------------
/**
* @brief        Gets the number of embedded images.
* @param[out]   pOutImageCount The number of images embedded.
*
* @result       nn::cdr::Error_Ok if successful, else an error code.
*/
CDR_DLL_DECL Error GetNumberOfImages( int32_t* pOutImageCount );

//------------------------------------------------------------------------------
/**
* @brief        Gets the details of an image at the given index.
* @param[in]    AtIndex The index of the image we want details for.
* @param[out]   pType The file type of image that is embedded.
* @param[out]   pType The size  of the image that is embedded.
*
* @result       nn::cdr::Error_Ok if successful, else an error code.
*/
CDR_DLL_DECL Error GetImageDetails( uint32_t AtIndex, ScreenshotType* pType, uint64_t* pDataSize );

//------------------------------------------------------------------------------
/**
* @brief        Gets the buffer of the image.
* @param[in]    AtIndex The index of the image we want the buffer for.
* @param[out]   pToBuffer The buffer to fill with the image data.
* @param[in]    DataSize The amount the user allocated for the buffer.
*
* @result       nn::cdr::Error_Ok if successful, else an error code.
*/
CDR_DLL_DECL Error GetImageBuffer( uint32_t AtIndex, void* pToBuffer, uint64_t DataSize );

//------------------------------------------------------------------------------
/**
* @brief        Gets the video embedded in the crash dump and stores it.
*               By default they are stored relative to host executable.
*               A video dir can be set prior to this using
*               SetScreenshotDirectoryPath()
* @param[in]    pFileName What the video file will be called.
*
* @result       nn::cdr::Error_Ok if successful, else an error code.
*/
CDR_DLL_DECL Error ExtractVideo( const char* pFileName );

//------------------------------------------------------------------------------
/**
* @brief        Gets the size of log content
* @param[out]   pOutLogSize The size of the log content
*
* @result       nn::cdr::Error_Ok if successful, else an error code.
*/
CDR_DLL_DECL Error GetLogContentSize( uint64_t* pOutLogSize );

//------------------------------------------------------------------------------
/**
* @brief        Gets the buffer of the log content.
* @param[out]   pToBuffer The buffer to fill with the log data.
* @param[in]    DataSize The amount the user allocated for the buffer.
*
* @result       nn::cdr::Error_Ok if successful, else an error code.
*/
CDR_DLL_DECL Error GetLogBuffer( char* pToBuffer, uint64_t DataSize );

}

}

