﻿// ==========================================================================
//  Copyright (C) Deja Tools, LLC.  All rights reserved.
// ==========================================================================

#include "stdafx.h"
#include <stdio.h>
#include <stdlib.h>
#include <cstdarg>
#include <cstdlib>
#include <string.h>

#if defined(NN_NINTENDO_SDK)
#include <nn.h>
#include <nn/nn_Log.h>
#include <nn/os.h>
#include <nn/tma/tma.h>
#include <nn/htcs.h>
#if defined(_WIN32)
#include <Windows.h>
#endif
#else
#include <stdint.h>
#include <tchar.h>
#include <WinSock2.h>
#include <ws2tcpip.h>
#include <Windows.h>
#include "../HTCSDirectory/HTCSDirectory.h"
#endif

#include "../StressTestsCommon.h"
#include "../Utils/Utils.h"

#if !defined(WIN32)
using Socket = int;
using ThreadReturn = int32_t;
const int INVALID_SOCKET{ -1 };
#else
using Socket = SOCKET;
using ThreadReturn = DWORD;
#endif

namespace
{
    const char     HostIPAddress[]{ "127.0.0.1" };
    const uint16_t HostPort{ 5430 };
    const char     RegisterRequestName[]{ "StressTestsCommonRequest" };

#if !defined(NN_NINTENDO_SDK)
    htcs_directory* s_pHTCSDirectory{ nullptr };
    HANDLE          s_hHTCSDirectoryMutex{ INVALID_HANDLE_VALUE };
    class LockMutex
    {
    private:
        HANDLE m_Mutex;
    public:
        LockMutex( HANDLE Mutex )
        {
            m_Mutex = Mutex;
            const DWORD WaitResult{ ::WaitForSingleObject( m_Mutex, INFINITE ) };
            if( WaitResult != WAIT_OBJECT_0 )
            {
                const DWORD LastError{ ::GetLastError() };
                REPORT( "[%s] ERROR: Wait for Mutex failed.  LastError: %u.  Result: %u.\n", StressTestsCommon::ApplicationName, LastError, WaitResult );
            }
        }
        ~LockMutex()
        {
            const BOOL bReleased{ ::ReleaseMutex( m_Mutex ) };
            if( bReleased == FALSE )
            {
                const DWORD LastError{ ::GetLastError() };
                REPORT( "[%s] ERROR: ReleaseMutex failed.  LastError: %u.\n", StressTestsCommon::ApplicationName, LastError );
            }
        }
    };
#endif

    const uint32_t ListenQueue{ 1024 };

    // Designed for Thread Safe testing.
    struct TestInformation
    {
        int32_t m_ExpectedNumberOfCommands{ 0 };
        int32_t m_ExpectedNumberOfSendCommands{ 0 };
        int32_t m_ExpectedNumberOfReceiveCommands{ 0 };
        int32_t m_ActualNumberOfSendCommands{ 0 };
        int32_t m_ActualNumberOfReceiveCommands{ 0 };
        int32_t m_NumberOfSendBytes{ 0 };
        int32_t m_NumberOfReceiveBytes{ 0 };

        uint8_t m_DataBuffer[StressTestsCommon::MAXIMUM_NUMBER_OF_BYTES_TO_SEND];
    };
}

// Forward declarations.
bool HandleCommands( Socket Socket );
ThreadReturn ServerThread( void* lpServerThreadArgument );


// --------------------------------------------------------------------------

int32_t HTCSGeneratePortName( char* PortName, int32_t PortNameSize, int32_t PortSuffix )
{
    int32_t Result{ 0 };

    if( Result == 0 )
    {
#if defined(WIN32)
        sprintf_s( PortName, PortNameSize, "%s_%d", StressTestsCommon::PortName, PortSuffix );
#else
        (void)PortNameSize;
        sprintf( PortName, "%s_%d", StressTestsCommon::PortName, PortSuffix );
#endif
    }

    return Result;
}


#if !defined(NN_NINTENDO_SDK)

// --------------------------------------------------------------------------

int32_t HTCSRegisterEntry( int32_t PortNumber )
{
    int32_t Result{ 0 };

    // Generate the PortName.
    const int32_t BufferSize{ 1024 };
    char LocalPortName[BufferSize];
    if( Result == 0 )
    {
        HTCSGeneratePortName( LocalPortName, BufferSize, PortNumber );
    }

    // Generate the HostPortString.
    char LocalHostPort[BufferSize];
    if( Result == 0 )
    {
        sprintf_s( LocalHostPort, BufferSize, "%d", HostPort + PortNumber );
    }

    if( Result == 0 )
    {
        LockMutex Locked{ s_hHTCSDirectoryMutex };
        if( !s_pHTCSDirectory->RegisterEntry( StressTestsCommon::ApplicationName, LocalPortName, HostIPAddress, LocalHostPort, RegisterRequestName ) )
        {
            Result = EXIT_FAILURE;
        }
        REPORT( "RegisterEntry '%s', '%s'? '%s'.\n", LocalPortName, LocalHostPort, (Result == 0) ? "Passed" : "Failed" );
    }

    return Result;
}

// --------------------------------------------------------------------------

int32_t HTCSUnregisterEntry( int32_t PortNumber )
{
    int32_t Result{ 0 };

    // Generate the PortName and the HostPortString.
    const int32_t BufferSize{ 1024 };
    char LocalHostPort[BufferSize];
    if( Result == 0 )
    {
        sprintf_s( LocalHostPort, BufferSize, "%d", HostPort + PortNumber );
    }

    if (Result == 0)
    {
        LockMutex Locked{ s_hHTCSDirectoryMutex };
        if( !s_pHTCSDirectory->UnRegisterEntry( StressTestsCommon::ApplicationName, LocalHostPort ) )
        {
            Result = EXIT_FAILURE;
        }
        REPORT( "UnRegisterEntry '%s'?  '%s'.\n", LocalHostPort, (Result == 0) ? "Passed" : "Failed" );
    }

    return Result;
}
#endif

// --------------------------------------------------------------------------
//  Creates a listening socket for our server.  Returns < 0 if there's an error.
//  In: PortSuffix: The number appended to StressTestsCommon::PortName.
//                  Example: "%s_%d", PortName, PortSuffix.
int CreateListeningSocketFromPortSuffix( int32_t PortSuffix )
{
    // ----------------------------------------------------------------------
    //  Create our socket
#if defined(NN_NINTENDO_SDK)
    int Ret = ::nn::htcs::Socket();
#else
    SOCKET Ret = socket( AF_INET, SOCK_STREAM, IPPROTO_TCP );
#endif
    if( Ret < 0 )
    {
        REPORT( "Error creating socket.\n" );
        return Ret;
    }

    // ----------------------------------------------------------------------
    // Get our address ready for the bind.
#if defined(NN_NINTENDO_SDK)
    nn::htcs::SockAddrHtcs ServerAddress;
    memset( &ServerAddress, 0, sizeof(ServerAddress) );
    ServerAddress.family = nn::htcs::HTCS_AF_HTCS;
    strcpy( ServerAddress.peerName.name, StressTestsCommon::ApplicationName );
    sprintf( ServerAddress.portName.name, "%s_%d", StressTestsCommon::PortName, PortSuffix );
#else
    struct sockaddr_in ServerAddress;

    ServerAddress.sin_family = AF_INET;
    ServerAddress.sin_addr.s_addr = INADDR_ANY;
    ServerAddress.sin_port = htons( HostPort + PortSuffix );
#endif

    // ----------------------------------------------------------------------
    // Bind our socket addresss to the  listening socket
#if defined(NN_NINTENDO_SDK)
    int Res = ::nn::htcs::Bind( Ret, &ServerAddress );
#else
    int Res = bind( Ret, (struct sockaddr *)&ServerAddress, sizeof(ServerAddress) );
#endif
    if( Res >= 0 )
    {
        // ----------------------------------------------------------------------
        // Call listen()
#if defined(NN_NINTENDO_SDK)
        Res = nn::htcs::Listen( Ret, ListenQueue );
#else
        Res = listen( Ret, ListenQueue );
#endif
        if( Res >= 0 )
        {
            //=========================================
            // No problems, so return the socket.
            return Ret;
        }
    }

    //=================================================
    // Problem creating our socket, so close it and
    // return the error code.
    //=================================================
#if defined(NN_NINTENDO_SDK)
    nn::htcs::Close( Ret );
#else
    closesocket( Ret );
#endif

    return Res;
}

// --------------------------------------------------------------------------
// Reads from the passed socket and puts it into the passed buffer.  Returns the amount
// read, or the error code if there was a problem.
int Receive( int Socket, void* pDataBuffer, size_t ReadAmount )
{
    int AmountRead = 0;
    int AmountToRead = ReadAmount;
    uint8_t* pBuffer{ reinterpret_cast<uint8_t *>(pDataBuffer) };
    while( AmountRead < (int)ReadAmount )
    {
#if defined(NN_NINTENDO_SDK)
        int Res = nn::htcs::Recv( Socket, pBuffer, AmountToRead, HTCS_MSG_WAITALL );
#else
        int Res = recv( Socket, reinterpret_cast<char *>(pBuffer), AmountToRead, 0 );
#endif

        if( Res > 0  )
        {
            AmountRead += Res;
            AmountToRead -= Res;
            pBuffer += Res;
        }
        else
        {
            return Res;
        }
    }

    return AmountRead;
}

// --------------------------------------------------------------------------
// Sends the passed data to the passed socket.  Returns the amount sent, or the error code
// if there was a problem.
int Send( int Socket, const void* pData, size_t SizeOfData )
{
    int TotalWritten = 0;
    const char* pBuffer = reinterpret_cast<const char *>(pData);
    size_t AmountLeft  = SizeOfData;

    while( AmountLeft > 0 )
    {
#if defined(NN_NINTENDO_SDK)
        int WrittenThisPass = nn::htcs::Send( Socket, pBuffer, AmountLeft, HTCS_MSG_WAITALL );
#else
        int WrittenThisPass = send( Socket, pBuffer, AmountLeft, 0 );
#endif
        if ( WrittenThisPass  <= 0 )
        {
            return WrittenThisPass;
        }

        TotalWritten += WrittenThisPass;
        AmountLeft -= WrittenThisPass;
        pBuffer += WrittenThisPass;
    }

    return TotalWritten;
}

// --------------------------------------------------------------------------

int32_t SendHeaderAndData( int Socket, const StressTestsCommon::Header& Header, const void* pDataBuffer, int32_t* pActualByteCount = nullptr )
{
    int32_t Result{ 0 };

    if( Result == 0 )
    {
        int32_t Sent = Send( Socket, &Header, sizeof(Header) );
        if( Sent != sizeof(Header) )
        {
            REPORT( "SendHeaderAndData: Error 0x%x calling Send.\n", Sent );
            Result = EXIT_FAILURE;
        }
    }

    int32_t ActualByteCount{ 0 };
    if( Result == 0 )
    {
        ActualByteCount = Send( Socket, pDataBuffer, Header.m_DataLength );
        if( ActualByteCount != Header.m_DataLength )
        {
            REPORT( "SendHeaderAndData: Failed to Send: %d != %d bytes.\n", ActualByteCount, Header.m_DataLength );
            Result = EXIT_FAILURE;
        }
    }

    // Return the Actual Byte Count, if requested.
    if( pActualByteCount != nullptr )
    {
        *pActualByteCount = ActualByteCount;
    }

    return Result;
}

// --------------------------------------------------------------------------
// Initializes the static variables, in preparation for the next test.
void InitializeTest( TestInformation& TestInformation, const StressTestsCommon::InitializeTest& InitializeTest )
{
    TestInformation.m_ExpectedNumberOfCommands        = InitializeTest.m_ExpectedNumberOfCommands;
    TestInformation.m_ExpectedNumberOfSendCommands    = InitializeTest.m_ExpectedNumberOfSendCommands;
    TestInformation.m_ExpectedNumberOfReceiveCommands = InitializeTest.m_ExpectedNumberOfReceiveCommands;
    TestInformation.m_ActualNumberOfSendCommands      = 0;
    TestInformation.m_ActualNumberOfReceiveCommands   = 0;
    TestInformation.m_NumberOfSendBytes               = 0;
    TestInformation.m_NumberOfReceiveBytes            = 0;
}

// --------------------------------------------------------------------------
// Fills out a StressTestsCommon::FinalizeTest with the current results.
void FillOutFinalizeTest( StressTestsCommon::FinalizeTest& FinalizeTest, const TestInformation& TestInformation )
{
    FinalizeTest.m_ExpectedNumberOfCommands        = TestInformation.m_ExpectedNumberOfCommands;
    FinalizeTest.m_ExpectedNumberOfSendCommands    = TestInformation.m_ExpectedNumberOfSendCommands;
    FinalizeTest.m_ExpectedNumberOfReceiveCommands = TestInformation.m_ExpectedNumberOfReceiveCommands;
    FinalizeTest.m_ActualNumberOfCommands          = TestInformation.m_ActualNumberOfSendCommands + TestInformation.m_ActualNumberOfReceiveCommands;
    FinalizeTest.m_ActualNumberOfSendCommands      = TestInformation.m_ActualNumberOfSendCommands;
    FinalizeTest.m_ActualNumberOfReceiveCommands   = TestInformation.m_ActualNumberOfReceiveCommands;
    FinalizeTest.m_NumberOfBytesTransferred        = TestInformation.m_NumberOfSendBytes + TestInformation.m_NumberOfReceiveBytes;
    FinalizeTest.m_NumberOfSendBytes               = TestInformation.m_NumberOfSendBytes;
    FinalizeTest.m_NumberOfReceiveBytes            = TestInformation.m_NumberOfReceiveBytes;
}

// --------------------------------------------------------------------------
// Does a Send command using the passed socket.  Returns FALSE if there
// was a problem, TRUE if not.
int32_t HandleInitializeServerTestResults( TestInformation& TestInformation, Socket Socket, const StressTestsCommon::Header& Header )
{
    int32_t Result{ 0 };

    // Wait to receive the StressTestsCommon::InitializeTest data.
    StressTestsCommon::InitializeTest InitializeTest{};
    if( Result == 0 )
    {
        int32_t BytesReceived{ 0 };
        while( (BytesReceived < Header.m_DataLength) && (Result == 0) )
        {
            const int32_t TotalBytesLeft{ Header.m_DataLength - BytesReceived };
            const int32_t ActualBytesRead{ Receive( Socket, &InitializeTest, TotalBytesLeft ) };
            if( ActualBytesRead < 0 )
            {
                REPORT( "HandleInitializeServerTestResults: Failed to Receivie %d bytes on socket %d.  Received: %d of %d.\n", Header.m_DataLength, Socket, BytesReceived, Header.m_DataLength );
                Result = EXIT_FAILURE;
            }

            BytesReceived += ActualBytesRead;
        }
    }

    // Process the InitializeTest data.
    if( Result == 0 )
    {
        ::InitializeTest( TestInformation, InitializeTest );
    }

    return Result;
}

// --------------------------------------------------------------------------
// Does a Send command using the passed socket.  Returns FALSE if there
// was a problem, TRUE if not.
int32_t HandleRequestServerTestResults( Socket Socket, const StressTestsCommon::Header& Header, const TestInformation& TestInformation )
{
    int32_t Result{ 0 };
    REPORT( "HandleRequestServerTestResults:  Handling ServerTestResults on socket %d.\n", Socket );

    // Fill out the Server's test results.
    StressTestsCommon::FinalizeTest FinalizeTest{};
    if( Result == 0 )
    {
        FillOutFinalizeTest( FinalizeTest, TestInformation );
    }

    // Fill out the Header to precede the ServerTestResults data.
    const StressTestsCommon::Header ServerTestResultsHeader
    {
        StressTestsCommon::Command::ServerTestResults,
        0,
        sizeof(FinalizeTest)
    };

    if ( Result == 0 )
    {
        Result = SendHeaderAndData( Socket, ServerTestResultsHeader, &FinalizeTest );
    }

    return Result;
}

// --------------------------------------------------------------------------
// Does a Send command using the passed socket.  Returns FALSE if there
// was a problem, TRUE if not.
int32_t HandleSendCommand( Socket Socket, const StressTestsCommon::Header& Header, TestInformation& TestInformation )
{
    int32_t Result{ 0 };

    //REPORT( "[%s] %5d/%5d) Socket: %d. Sending %d bytes on socket %d.\n", StressTestsCommon::ApplicationName, TestInformation.m_ActualNumberOfSendCommands + 1, TestInformation.m_ExpectedNumberOfSendCommands, Socket, Header.m_DataLength, Socket );

    // Regardless if the Send succeeds, the Send command *was* received.
    TestInformation.m_ActualNumberOfSendCommands++;

    // Write the "data pattern" to the send buffer.
    if( Result == 0 )
    {
        // Clear the memory - to help catch corruption on the "Remote host".
        memset( TestInformation.m_DataBuffer, 0x99, Header.m_DataLength );
        Result = StressTestsCommon::WritePatternFill( TestInformation.m_DataBuffer, Header.m_DataLength );
    }

    // Send the data to the Client.
    if( Result == 0 )
    {
        const int ActualBytesWritten{ Send( Socket, TestInformation.m_DataBuffer, Header.m_DataLength ) };
        if( ActualBytesWritten >= 0 )
        {
            TestInformation.m_NumberOfSendBytes += ActualBytesWritten;
        }
        else
        {
            Result = EXIT_FAILURE;
        }
    }

    return Result;
}

// --------------------------------------------------------------------------
// Does a Receive command using the passed socket.  Returns FALSE if there
// was a problem, TRUE if not.
int32_t HandleReceiveCommand( Socket Socket, const StressTestsCommon::Header& Header, TestInformation& TestInformation )
{
    int32_t Result{ 0 };

    //REPORT( "[%s] %5d/%5d) Socket: %d. Receiving %d bytes on socket %d.\n", StressTestsCommon::ApplicationName, TestInformation.m_ActualNumberOfReceiveCommands + 1, TestInformation.m_ExpectedNumberOfReceiveCommands, Socket, Header.m_DataLength, Socket );

    // Regardless if the Receive succeeds, the Receive command *was* received.
    TestInformation.m_ActualNumberOfReceiveCommands++;

    // Receive the Data.
    if( Result == 0 )
    {
        // Clear the memory - to help catch corruption from the "Remote host".
        memset( TestInformation.m_DataBuffer, 0xAA, Header.m_DataLength );
        const int32_t ActualBytesRead{ Receive( Socket, TestInformation.m_DataBuffer, Header.m_DataLength ) };
        if( ActualBytesRead >= 0 )
        {
            TestInformation.m_NumberOfReceiveBytes += ActualBytesRead;
        }
        else
        {
            Result = EXIT_FAILURE;
        }
    }

    // Validate the Received data.
    StressTestsCommon::ReceiveResult ReceiveResult{};
    if (Result == 0)
    {
        int32_t PatternMatches{ 0 };
        uint8_t ActualValue{ 0 };
        uint8_t ExpectedValue{ 0 };
        Result = StressTestsCommon::ValidatePatternFill( TestInformation.m_DataBuffer, Header.m_DataLength, ReceiveResult.m_DiscrepancyIndex, ActualValue, ExpectedValue );
        if( Result == 0 )
        {
            PatternMatches = (ReceiveResult.m_DiscrepancyIndex == -1) ? 1 : 0;
        }
        else
        {
            Result = EXIT_FAILURE;
            REPORT( "[%s] Socket: %d. HandleReceiveCommand. Failed to validate pattern.  Discrepancy byte index: %d/%d.  Byte 0x%02X != 0x%02X.", StressTestsCommon::ApplicationName, Socket, ReceiveResult.m_DiscrepancyIndex, Header.m_DataLength, ActualValue, ExpectedValue );
        }
        ReceiveResult.m_PatternMatched = PatternMatches;
        ReceiveResult.m_ActualValue    = ActualValue;
        ReceiveResult.m_ExpectedValue  = ExpectedValue;
    }

    // Send the ReceiveResult back to the Client.
    // *Always* attempt to send the result to the Client.
//    if( Result == 0 )
    {
        const StressTestsCommon::Header ReceiveResultHeader
        {
            StressTestsCommon::Command::ReceiveServerResult,
            0,
            sizeof(ReceiveResult),
        };
        const int32_t SendResult{ SendHeaderAndData( Socket, ReceiveResultHeader, &ReceiveResult ) };
        if( Result == 0 )
        {
            Result = SendResult;
        }
    }

    return Result;
}

// --------------------------------------------------------------------------
struct ServerThreadArgument
{
    int32_t             m_PortSuffix;
#if defined(NN_NINTENDO_SDK)
    nn::os::ThreadType  m_hThread;
    void*               m_pStack;
#else
    HANDLE              m_hThread;
    DWORD               m_ThreadId;
#endif
    int32_t             m_Result;
};
ThreadReturn ServerThread( void* lpServerThreadArgument )
{
    ::ServerThreadArgument &ServerThreadArgument{ *reinterpret_cast<::ServerThreadArgument *>(lpServerThreadArgument) };
    int32_t &Result{ ServerThreadArgument.m_Result };

    //  Create a listening socket for this thread.
    int ListeningSocket = CreateListeningSocketFromPortSuffix( ServerThreadArgument.m_PortSuffix );
    if( ListeningSocket == INVALID_SOCKET )
    {
        Result = EXIT_FAILURE;
    }

#if !defined(NN_NINTENDO_SDK)
    // PC-based applications have to tell HTCS who we are and where we can be found.
    // This htcs_directory object provides an interface for doing that.
    // (You can find the source in the Samples\HTCS\HTCSDirectory folder.)
    if( Result == 0 )
    {
        Result = HTCSRegisterEntry( ServerThreadArgument.m_PortSuffix );
    }
#endif

    // ----------------------------------------------------------------------
    // Do our command loop
    bool ShutDown{ false };
    while( !ShutDown && (Result == 0) )
    {
        // ----------------------------------------------------------------------
        //  Wait for a connection, then accept() it
        REPORT( "[%s] Listening socket: %d.  Waiting on connection\n", StressTestsCommon::ApplicationName, ListeningSocket );
#if defined(NN_NINTENDO_SDK)
        nn::htcs::SockAddrHtcs AtAddress;
        Socket ConnectingSocket = nn::htcs::Accept( ListeningSocket, &AtAddress );
#else
        Socket ConnectingSocket = accept( ListeningSocket, nullptr, nullptr );
#endif

        if( ConnectingSocket == INVALID_SOCKET )
        {
            REPORT( "[%s] Listening socket  Error calling accept()\n", StressTestsCommon::ApplicationName, ConnectingSocket );
            break;
        }

        REPORT( "[%s] Accepted connection on socket %d\n", StressTestsCommon::ApplicationName, ConnectingSocket );
        ShutDown = HandleCommands( ConnectingSocket );
    }

    REPORT( "[%s] Listening socket: %d.  Shutting down server.\n", StressTestsCommon::ApplicationName, ListeningSocket );

#if !defined(NN_NINTENDO_SDK)
    // Done now, so PC-based applications have to tell HTCS we are no longer available.
    // This htcs_directory object provides an interface for doing that.
    // (You can find the source in the Samples\HTCS\HTCSDirectory folder.)
    HTCSUnregisterEntry( HostPort );
#endif
    CloseSocket( ListeningSocket );
    ListeningSocket = INVALID_SOCKET;

    return 0;
}

// --------------------------------------------------------------------------

int32_t HandleThreadsTest( Socket Socket, const StressTestsCommon::Header& Header )
{
    int32_t Result{ 0 };

    REPORT( "[%s] Socket: %d. Starting ThreadsTest.\n", StressTestsCommon::ApplicationName, Socket );

    // Validate the header's data length.
    StressTestsCommon::ThreadsTest ThreadsTest{};
    if( Result == 0 )
    {
        if( Header.m_DataLength != sizeof(ThreadsTest) )
        {
            Result = EXIT_FAILURE;
        }
    }

    // Receive the Data.
    if( Result == 0 )
    {
        const int32_t ActualBytesRead{ Receive( Socket, &ThreadsTest, Header.m_DataLength ) };
        if( ActualBytesRead != Header.m_DataLength )
        {
            REPORT( "[%d] Socket: %d.  Failed to receive ThreadsTest data. %d != %d bytes.\n", StressTestsCommon::ApplicationName, Socket, ActualBytesRead, Header.m_DataLength );
            Result = EXIT_FAILURE;
        }
    }

    // Allocate the TestThreadArgument array.
    ServerThreadArgument *ServerThreadArguments{ nullptr };
    if( Result == 0 )
    {
        ServerThreadArguments = new ServerThreadArgument[ThreadsTest.m_NumberOfSocketThreads];
        for( int32_t Index = 0; Index < ThreadsTest.m_NumberOfSocketThreads; Index++ )
        {
            ServerThreadArguments[Index].m_PortSuffix = Index + 1;
#if defined(WIN32)
            ServerThreadArguments[Index].m_hThread    = INVALID_HANDLE_VALUE;
            ServerThreadArguments[Index].m_ThreadId   = 0;
#else
            ServerThreadArguments[Index].m_pStack     = nullptr;
#endif
            ServerThreadArguments[Index].m_Result     = 0;
        }
    }

    // Start the new Server Thread(s).
    if( Result == 0 )
    {
        for( int32_t Index = 0; Index < ThreadsTest.m_NumberOfSocketThreads; Index++ )
        {
#if defined(WIN32)
            ServerThreadArguments[Index].m_hThread = ::CreateThread( nullptr, 0, reinterpret_cast<PTHREAD_START_ROUTINE>(ServerThread), ServerThreadArguments + Index, 0, &ServerThreadArguments[Index].m_ThreadId );
#else
            char LocalPortName[64]{};
            const uint32_t StackSize{ 1024 * 64 };
            const uint32_t AllocateStackSize{ ((StackSize + (nn::os::StackRegionAlignment - 1)) & ~(nn::os::StackRegionAlignment - 1)) + (nn::os::StackRegionAlignment - 1) };
            if( Result == 0 )
            {
                ServerThreadArguments[Index].m_pStack = new uint8_t[AllocateStackSize];
                void *pAlignedStack{ (void*)((((uint64_t)ServerThreadArguments[Index].m_pStack) - 1 + nn::os::StackRegionAlignment) & ~(nn::os::StackRegionAlignment - 1)) };
                const nn::Result StartResult{ nn::os::CreateThread( &ServerThreadArguments[Index].m_hThread, reinterpret_cast<nn::os::ThreadFunction>(ServerThread), ServerThreadArguments + Index, pAlignedStack, StackSize, 0 ) };
                if( StartResult.IsSuccess() )
                {
                    HTCSGeneratePortName( LocalPortName, sizeof(LocalPortName), Index + 1 );
                    nn::os::SetThreadName( &ServerThreadArguments[Index].m_hThread, LocalPortName );
                    nn::os::StartThread( &ServerThreadArguments[Index].m_hThread );
                }
                else
                {
                    Result = EXIT_FAILURE;
                }
//                REPORT( "[%s] ThreadedTest: %d/%d) Created thread for Socket: %d.  Result: %d.\n", ApplicationName, Index + 1, NumberOfSockets, TestThreadArguments[Index].m_Socket, Result );
            }
#endif
        }
    }

    // Validate that the HTCS Ports are visible.
#if !defined(NN_NINTENDO_SDK)
    if( Result == 0 )
    {
        const int32_t BufferSize{ 64 };
        char LocalPortName[BufferSize];
        bool bFoundAllPorts{ false };
        while( !bFoundAllPorts )
        {
            htcs_directory HTCSDirectory;

            bFoundAllPorts = true;
            for( int32_t Index = 0; (Index < ThreadsTest.m_NumberOfSocketThreads) && (Result == Result) && bFoundAllPorts; Index++ )
            {
                Result = HTCSGeneratePortName( LocalPortName, BufferSize, ServerThreadArguments[Index].m_PortSuffix );
                if( Result == 0 )
                {
                    char DummyIPAddress[BufferSize]{ "" };
                    int32_t DummyPort{ 0 };
                    bFoundAllPorts = HTCSDirectory.ResolveTcpAddress( StressTestsCommon::ApplicationName, LocalPortName, DummyIPAddress, &DummyPort ) != FALSE;
                }
            }
        }
    }
#endif

    // *Always* acknowledge the ThreadsTest with a ThreadsTestResponse.
    {
        const StressTestsCommon::ThreadsTestResponse ThreadsTestResponse
        {
            Result,                                             // m_Result.
        };
        const StressTestsCommon::Header ThreadsTestResponseHeader
        {
            StressTestsCommon::Command::ThreadsTestResponse,    // m_Command.
            0,                                                  // m_SequenceNumber.
            sizeof(ThreadsTestResponse),                        // m_DataLength.
        };
        const int32_t ResponseResult{ SendHeaderAndData( Socket, ThreadsTestResponseHeader, &ThreadsTestResponse ) };
        if( Result == 0 )
        {
            Result = ResponseResult;
        }
    }

    // Wait for the Threads to finish.
    if( Result == 0 )
    {
        for( int32_t Index = 0; (Index < ThreadsTest.m_NumberOfSocketThreads) && (Result == 0); Index++ )
        {
            ::ServerThreadArgument& ServerThreadArgument{ ServerThreadArguments[Index] };
#if defined(WIN32)
            const DWORD TimeOutMS{ INFINITE };
            const DWORD WaitResult{ ::WaitForSingleObject( ServerThreadArgument.m_hThread, TimeOutMS ) };
            if( WaitResult != WAIT_OBJECT_0 )
            {
                Result = EXIT_FAILURE;
                REPORT( "[%s] Socket: %d.  Failed to WaitForSingleObject: 0x%08X.\n", StressTestsCommon::ApplicationName, Socket, ServerThreadArgument.m_hThread );
            }
#else
            // Wait for the thread to exit.
            uint8_t State = ServerThreadArgument.m_hThread._state;
            if( ( State == nn::os::ThreadType::State_Initialized ) ||
                ( State == nn::os::ThreadType::State_Started ) ||
                ( State == nn::os::ThreadType::State_Exited) )
            {
                nn::os::WaitThread( &ServerThreadArgument.m_hThread );
                nn::os::DestroyThread( &ServerThreadArgument.m_hThread );
            }

            delete[] reinterpret_cast<uint8_t *>(ServerThreadArgument.m_pStack);
            ServerThreadArgument.m_pStack = nullptr;
#endif
        }
    }

    // Close the sockets.
    if( Result == 0 )
    {
        REPORT( "[%s] Socket: %d. Close %d threads.\n", StressTestsCommon::ApplicationName, Socket, ThreadsTest.m_NumberOfSocketThreads );
    }

    // Unregister the HTCS entries.
#if !defined(NN_NINTENDO_SDK)
    if( Result == 0 )
    {
        for( int32_t Index = 0; (Index < ThreadsTest.m_NumberOfSocketThreads) && (Result == 0); Index++ )
        {
            Result = HTCSUnregisterEntry( Index + 1 );
            REPORT( "[%s] Socket: %d. UnRegister Entry: %d\n", StressTestsCommon::ApplicationName, Socket, Index + 1, (Result == 0) ? "Yes" : "No" );
        }
    }
#endif

    return Result;
}

// --------------------------------------------------------------------------
// Handles all the commands that come over the passed socket.
bool HandleCommands( Socket Socket )
{
    bool ShutDownServer = false;
    REPORT( "[%s] Socket: %d. Starting command session.\n", StressTestsCommon::ApplicationName, Socket );

    TestInformation* pTestInformation{ new TestInformation };

    // The Client will tell the Server when it should shutdown, using StressTestsCommon::Command::CloseServer.
    StressTestsCommon::Header Header;
    bool ShutDownSession{ false };
    int Result{ 0 };
    while ( !ShutDownServer && (Result == 0) )
    {
        // Wait for commands.
        int BytesRead = Receive( Socket, (char*)&Header, sizeof(Header) );
        if( BytesRead == sizeof(Header) )
        {
            switch( static_cast<StressTestsCommon::Command>(Header.m_Command) )
            {
            case StressTestsCommon::Command::InitializeServerTest:
            {
                Result = HandleInitializeServerTestResults( *pTestInformation, Socket, Header );
            }
            break;

            case StressTestsCommon::Command::RequestServerTestResults:
            {
                Result = HandleRequestServerTestResults( Socket, Header, *pTestInformation );
            }
            break;

            case StressTestsCommon::Command::Send:
            {
                Result = HandleSendCommand( Socket, Header, *pTestInformation );
            }
            break;

            case StressTestsCommon::Command::Receive:
            {
                Result = HandleReceiveCommand( Socket, Header, *pTestInformation );
            }
            break;

            default:
            case StressTestsCommon::Command::Close:
            {
                REPORT( "[%s] Socket: %d.  Received close command.\n", StressTestsCommon::ApplicationName, Socket );
                ShutDownSession = true;
            }
            break;

            case StressTestsCommon::Command::CloseServer:
            {
                REPORT( "[%s] Socket: %d.  Received close server command.\n", StressTestsCommon::ApplicationName, Socket );
                // Send the same Header back to the Client.
                Result = SendHeaderAndData( Socket, Header, nullptr );
                ShutDownSession = true;
                ShutDownServer  = true;
            }
            break;

            case StressTestsCommon::Command::ThreadsTest:
            {
                Result = HandleThreadsTest( Socket, Header );
            }
            break;
            }
        }
        else
        {
            //Couldn't read, so we're done.
            break;
        }
    }

    //  Close the connected socket
#if defined(NN_NINTENDO_SDK)
    nn::htcs::Close( Socket );
#else
    closesocket( Socket );
#endif
    REPORT( "[%s] Socket: %d. Completed session.\n", StressTestsCommon::ApplicationName, Socket );

    // Deallocate the TestInformation.
    delete pTestInformation;
    pTestInformation = nullptr;

    return ShutDownServer;
}

// --------------------------------------------------------------------------

#if defined(NN_NINTENDO_SDK)
extern "C" void nnMain( void )
#else
int _tmain(int argc, _TCHAR* argv[])
#endif
{
    int32_t Result{ 0 };

    InitSystem();

    // Make sure HTCS is up and running (also implies that Target Manager is running).
    // NOTE: It appears that a global variable is needed.  If it gets destroyed it removes
    //       HTCS ports from Target Manager...
#if !defined(NN_NINTENDO_SDK)
    // Create the HTCS Directory mutex.
    if( Result == 0 )
    {
        s_hHTCSDirectoryMutex = ::CreateMutex( nullptr, false, nullptr );
    }

    if( Result == 0 )
    {
        while( s_pHTCSDirectory == nullptr )
        {
            s_pHTCSDirectory = new htcs_directory();
            if( s_pHTCSDirectory->HasError() )
            {
                delete s_pHTCSDirectory;
                s_pHTCSDirectory = nullptr;
                Sleep(100);
            }
        }
    }
#endif

    // Start the initial "connection".
    if( Result == 0 )
    {
        ::ServerThreadArgument ServerThreadArgument{};
        ServerThreadArgument.m_PortSuffix = 0;
#if defined(NN_NINTENDO_SDK)
//        ServerThreadArgument.m_hThread; // This is a class.  Let it initialize itself.
        ServerThreadArgument.m_pStack    = nullptr;
#else
        ServerThreadArgument.m_hThread   = INVALID_HANDLE_VALUE;
        ServerThreadArgument.m_ThreadId  = 0;
#endif
        ServerThreadArgument.m_Result    = 0;

        // Technically, this does not need to be a thread, so call the Thread function directly.
        ServerThread( &ServerThreadArgument );
        Result = ServerThreadArgument.m_Result;
    }

    // ----------------------------------------------------------------------

#if !defined(NN_NINTENDO_SDK)
    delete s_pHTCSDirectory;
    s_pHTCSDirectory = nullptr;

    // Destroy the HTCS Directory Mutex.
    if( s_hHTCSDirectoryMutex != INVALID_HANDLE_VALUE )
    {
        ::CloseHandle( s_hHTCSDirectoryMutex );
        s_hHTCSDirectoryMutex = INVALID_HANDLE_VALUE;
    }
#endif

    CloseSystem();

#if !defined(NN_NINTENDO_SDK)
    return 0;
#endif
}

// --------------------------------------------------------------------------
