﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt.h>
#include <nn/vi.h>
#include <nn/vi/profiler/vi_Profiler.h>
#include <nnt/viUtil/testVi_Profiler.h>
#include <nnt/viUtil/testVi_Fixture.h>

namespace
{
    const int TestCount = 1024;

    class Profiler : public nnt::vi::util::Fixture
    {
        public:
            virtual ~Profiler() NN_NOEXCEPT
            {
                nn::vi::profiler::Finalize();
            }
    };
}

TEST_F(Profiler, Basic)
{
    NNT_ASSERT_RESULT_SUCCESS(nn::vi::profiler::Initialize());

    nn::os::SystemEvent vsync;
    NNT_ASSERT_RESULT_SUCCESS(nn::vi::profiler::GetVsyncEvent(vsync.GetBase()));

    for( int i = 0; i < TestCount; ++i )
    {
        EXPECT_TRUE(vsync.TimedWait(nn::TimeSpan::FromSeconds(1)));
    }

    nn::vi::profiler::Finalize();

    vsync.Clear();
    // event should no longer fire
    EXPECT_FALSE(vsync.TimedWait(nn::TimeSpan::FromSeconds(1)));
}

TEST_F(Profiler, AppInitFirst_AppFinalizeFirst)
{
    nn::vi::Initialize();

    nn::vi::Display* pDisplay;
    NNT_ASSERT_RESULT_SUCCESS(nn::vi::OpenDefaultDisplay(&pDisplay));

    nn::os::SystemEvent appVsync;
    NNT_ASSERT_RESULT_SUCCESS(nn::vi::GetDisplayVsyncEvent(appVsync.GetBase(), pDisplay));

    NNT_ASSERT_RESULT_SUCCESS(nn::vi::profiler::Initialize());

    nn::os::SystemEvent debugVsync;
    NNT_ASSERT_RESULT_SUCCESS(nn::vi::profiler::GetVsyncEvent(debugVsync.GetBase()));

    nnt::vi::util::TestVsyncSignals(TestCount, &appVsync, &debugVsync);

    nn::vi::Finalize();

    appVsync.Clear();
    // app vsync event should no longer fire
    EXPECT_FALSE(appVsync.TimedWait(nn::TimeSpan::FromSeconds(1)));

    for( int i = 0; i < TestCount; ++i )
    {
        EXPECT_TRUE(debugVsync.TimedWait(nn::TimeSpan::FromSeconds(1)));
    }

    nn::vi::profiler::Finalize();

    debugVsync.Clear();
    // profiler vsync event should no longer fire
    EXPECT_FALSE(debugVsync.TimedWait(nn::TimeSpan::FromSeconds(1)));
}

TEST_F(Profiler, AppInitFirst_ProfilerFinalizeFirst)
{
    nn::vi::Initialize();

    nn::vi::Display* pDisplay;
    NNT_ASSERT_RESULT_SUCCESS(nn::vi::OpenDefaultDisplay(&pDisplay));

    nn::os::SystemEvent appVsync;
    NNT_ASSERT_RESULT_SUCCESS(nn::vi::GetDisplayVsyncEvent(appVsync.GetBase(), pDisplay));

    NNT_ASSERT_RESULT_SUCCESS(nn::vi::profiler::Initialize());

    nn::os::SystemEvent debugVsync;
    NNT_ASSERT_RESULT_SUCCESS(nn::vi::profiler::GetVsyncEvent(debugVsync.GetBase()));

    nnt::vi::util::TestVsyncSignals(TestCount, &appVsync, &debugVsync);

    nn::vi::profiler::Finalize();

    debugVsync.Clear();
    // profiler vsync event should no longer fire
    EXPECT_FALSE(debugVsync.TimedWait(nn::TimeSpan::FromSeconds(1)));

    for( int i = 0; i < TestCount; ++i )
    {
        EXPECT_TRUE(appVsync.TimedWait(nn::TimeSpan::FromSeconds(1)));
    }

    nn::vi::Finalize();

    appVsync.Clear();
    // app vsync event should no longer fire
    EXPECT_FALSE(appVsync.TimedWait(nn::TimeSpan::FromSeconds(1)));
}

TEST_F(Profiler, ProfilerInitFirst_AppFinalizeFirst)
{
    NNT_ASSERT_RESULT_SUCCESS(nn::vi::profiler::Initialize());

    nn::os::SystemEvent debugVsync;
    NNT_ASSERT_RESULT_SUCCESS(nn::vi::profiler::GetVsyncEvent(debugVsync.GetBase()));

    nn::vi::Initialize();

    nn::vi::Display* pDisplay;
    NNT_ASSERT_RESULT_SUCCESS(nn::vi::OpenDefaultDisplay(&pDisplay));

    nn::os::SystemEvent appVsync;
    NNT_ASSERT_RESULT_SUCCESS(nn::vi::GetDisplayVsyncEvent(appVsync.GetBase(), pDisplay));

    nnt::vi::util::TestVsyncSignals(TestCount, &appVsync, &debugVsync);

    nn::vi::Finalize();

    appVsync.Clear();
    // app vsync event should no longer fire
    EXPECT_FALSE(appVsync.TimedWait(nn::TimeSpan::FromSeconds(1)));

    for( int i = 0; i < TestCount; ++i )
    {
        EXPECT_TRUE(debugVsync.TimedWait(nn::TimeSpan::FromSeconds(1)));
    }

    nn::vi::profiler::Finalize();

    debugVsync.Clear();
    // profiler vsync event should no longer fire
    EXPECT_FALSE(debugVsync.TimedWait(nn::TimeSpan::FromSeconds(1)));
}

TEST_F(Profiler, ProfilerInitFirst_ProfilerFinalizeFirst)
{
    NNT_ASSERT_RESULT_SUCCESS(nn::vi::profiler::Initialize());

    nn::os::SystemEvent debugVsync;
    NNT_ASSERT_RESULT_SUCCESS(nn::vi::profiler::GetVsyncEvent(debugVsync.GetBase()));

    nn::vi::Initialize();

    nn::vi::Display* pDisplay;
    NNT_ASSERT_RESULT_SUCCESS(nn::vi::OpenDefaultDisplay(&pDisplay));

    nn::os::SystemEvent appVsync;
    NNT_ASSERT_RESULT_SUCCESS(nn::vi::GetDisplayVsyncEvent(appVsync.GetBase(), pDisplay));

    nnt::vi::util::TestVsyncSignals(TestCount, &appVsync, &debugVsync);

    nn::vi::profiler::Finalize();

    debugVsync.Clear();
    // profiler vsync event should no longer fire
    EXPECT_FALSE(debugVsync.TimedWait(nn::TimeSpan::FromSeconds(1)));

    for( int i = 0; i < TestCount; ++i )
    {
        EXPECT_TRUE(appVsync.TimedWait(nn::TimeSpan::FromSeconds(1)));
    }

    nn::vi::Finalize();

    appVsync.Clear();
    // app vsync event should no longer fire
    EXPECT_FALSE(appVsync.TimedWait(nn::TimeSpan::FromSeconds(1)));
}
