﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt.h>
#include <nnt/viUtil/testVi_Fixture.h>
#include <nn/vi.private.h>

typedef nnt::vi::util::Fixture SetDisplayContrastRatio;

TEST_F(SetDisplayContrastRatio, Default_Any)
{
    nn::vi::Initialize();

    nn::vi::Display* pDisplay;
    NNT_ASSERT_RESULT_SUCCESS(nn::vi::OpenDefaultDisplay(&pDisplay));

    nn::vi::CmuMode revert;
    NNT_ASSERT_RESULT_SUCCESS(nn::vi::GetDisplayCmuMode(&revert, pDisplay));

    NNT_ASSERT_RESULT_SUCCESS(nn::vi::SetDisplayCmuMode(pDisplay, nn::vi::CmuMode_HighContrast));

    NNT_EXPECT_RESULT_FAILURE(nn::vi::ResultInvalidRange, nn::vi::SetDisplayContrastRatio(pDisplay, std::numeric_limits<float>::lowest()));
    NNT_EXPECT_RESULT_FAILURE(nn::vi::ResultInvalidRange, nn::vi::SetDisplayContrastRatio(pDisplay, std::numeric_limits<float>::max()));

    NNT_EXPECT_RESULT_SUCCESS(nn::vi::SetDisplayContrastRatio(pDisplay, 1.f));
    float current = 0.f;
    NNT_EXPECT_RESULT_SUCCESS(nn::vi::GetDisplayContrastRatio(&current, pDisplay));
    EXPECT_EQ(1.f, current);

    NNT_EXPECT_RESULT_SUCCESS(nn::vi::SetDisplayContrastRatio(pDisplay, 0.f));
    NNT_EXPECT_RESULT_SUCCESS(nn::vi::GetDisplayContrastRatio(&current, pDisplay));
    EXPECT_EQ(0.f, current);

    NNT_EXPECT_RESULT_SUCCESS(nn::vi::SetDisplayCmuMode(pDisplay, revert));

    nn::vi::CloseDisplay(pDisplay);
}

TEST_F(SetDisplayContrastRatio, Default_NotHighContrast_Any)
{
    nn::vi::Initialize();

    nn::vi::Display* pDisplay;
    NNT_ASSERT_RESULT_SUCCESS(nn::vi::OpenDefaultDisplay(&pDisplay));

    nn::vi::CmuMode revert;
    NNT_ASSERT_RESULT_SUCCESS(nn::vi::GetDisplayCmuMode(&revert, pDisplay));

    NNT_ASSERT_RESULT_SUCCESS(nn::vi::SetDisplayCmuMode(pDisplay, nn::vi::CmuMode_Default));
    NNT_EXPECT_RESULT_FAILURE(nn::vi::ResultNotSupported, nn::vi::SetDisplayContrastRatio(pDisplay, 0.f));

    NNT_ASSERT_RESULT_SUCCESS(nn::vi::SetDisplayCmuMode(pDisplay, revert));

    nn::vi::CloseDisplay(pDisplay);
}

TEST_F(SetDisplayContrastRatio, External_IcosaCopper)
{
    nn::vi::Initialize();

    nn::vi::Display* pDisplay;
    NNT_ASSERT_RESULT_SUCCESS(nn::vi::OpenDisplay(&pDisplay, "External"));

    nn::vi::CmuMode revert;
    NNT_ASSERT_RESULT_SUCCESS(nn::vi::GetDisplayCmuMode(&revert, pDisplay));

    NNT_ASSERT_RESULT_SUCCESS(nn::vi::SetDisplayCmuMode(pDisplay, nn::vi::CmuMode_HighContrast));

    NNT_EXPECT_RESULT_FAILURE(nn::vi::ResultInvalidRange, nn::vi::SetDisplayContrastRatio(pDisplay, std::numeric_limits<float>::lowest()));
    NNT_EXPECT_RESULT_FAILURE(nn::vi::ResultInvalidRange, nn::vi::SetDisplayContrastRatio(pDisplay, std::numeric_limits<float>::max()));

    NNT_EXPECT_RESULT_SUCCESS(nn::vi::SetDisplayContrastRatio(pDisplay, 1.f));
    float current = 0.f;
    NNT_EXPECT_RESULT_SUCCESS(nn::vi::GetDisplayContrastRatio(&current, pDisplay));
    EXPECT_EQ(1.f, current);

    NNT_EXPECT_RESULT_SUCCESS(nn::vi::SetDisplayContrastRatio(pDisplay, 0.f));
    NNT_EXPECT_RESULT_SUCCESS(nn::vi::GetDisplayContrastRatio(&current, pDisplay));
    EXPECT_EQ(0.f, current);

    NNT_EXPECT_RESULT_SUCCESS(nn::vi::SetDisplayCmuMode(pDisplay, revert));

    nn::vi::CloseDisplay(pDisplay);
}

TEST_F(SetDisplayContrastRatio, External_NotHighContrast_IcosaCopper)
{
    nn::vi::Initialize();

    nn::vi::Display* pDisplay;
    NNT_ASSERT_RESULT_SUCCESS(nn::vi::OpenDisplay(&pDisplay, "External"));

    nn::vi::CmuMode revert;
    NNT_ASSERT_RESULT_SUCCESS(nn::vi::GetDisplayCmuMode(&revert, pDisplay));

    NNT_ASSERT_RESULT_SUCCESS(nn::vi::SetDisplayCmuMode(pDisplay, nn::vi::CmuMode_Default));
    NNT_EXPECT_RESULT_FAILURE(nn::vi::ResultNotSupported, nn::vi::SetDisplayContrastRatio(pDisplay, 0.f));

    NNT_ASSERT_RESULT_SUCCESS(nn::vi::SetDisplayCmuMode(pDisplay, revert));

    nn::vi::CloseDisplay(pDisplay);
}

TEST_F(SetDisplayContrastRatio, Edid_IcosaCopper)
{
    nn::vi::Initialize();

    nn::vi::Display* pDisplay;
    NNT_ASSERT_RESULT_SUCCESS(nn::vi::OpenDisplay(&pDisplay, "Edid"));

    int underscan = 0;
    NNT_EXPECT_RESULT_FAILURE(nn::vi::ResultNotSupported, nn::vi::SetDisplayContrastRatio(pDisplay, underscan));

    nn::vi::CloseDisplay(pDisplay);
}

TEST(SetDisplayContrastRatioPreconditions, ViNotInitialized_Any)
{
    nn::vi::Display* pDisplay = reinterpret_cast<nn::vi::Display*>(0x1234);
    float contrast = 0.f;
    ASSERT_DEATH_IF_SUPPORTED(nn::vi::SetDisplayContrastRatio(pDisplay, contrast), "");
}

TEST(SetDisplayContrastRatioPreconditions, pDisplayNull_Any)
{
    nn::vi::Initialize();

    float contrast = 0.f;
    EXPECT_DEATH_IF_SUPPORTED(nn::vi::SetDisplayContrastRatio(nullptr, contrast), "");

    nn::vi::Finalize();
}

TEST(SetDisplayContrastRatioPreconditions, pDisplayInvalidHandle)
{
    nn::vi::Initialize();

    nn::vi::Display* pDisplay = nullptr;
    NNT_EXPECT_RESULT_SUCCESS(nn::vi::OpenDefaultDisplay(&pDisplay));

    if( pDisplay != nullptr )
    {
        nn::vi::CloseDisplay(pDisplay);
        float contrast = 0.f;
        EXPECT_DEATH_IF_SUPPORTED(nn::vi::SetDisplayContrastRatio(pDisplay, contrast), "");
    }

    nn::vi::Finalize();
}
