﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt.h>
#include <nnt/viUtil/testVi_Fixture.h>
#include <nn/vi.private.h>

typedef nnt::vi::util::Fixture SetDisplayCmuMode;

namespace
{
    nn::vi::CmuMode g_SupportedCmuModes[] =
    {
        nn::vi::CmuMode_Default,
        nn::vi::CmuMode_Grayscale,
        nn::vi::CmuMode_HighContrast,
        nn::vi::CmuMode_InvertColor,
    };

    nn::vi::CmuMode g_UnsupportedCmuModes[] =
    {
        nn::vi::CmuMode_Disabled,
    };
}

TEST_F(SetDisplayCmuMode, Default_Any)
{
    nn::vi::Initialize();

    nn::vi::Display* pDisplay;
    NNT_ASSERT_RESULT_SUCCESS(nn::vi::OpenDefaultDisplay(&pDisplay));

    nn::vi::CmuMode revert;
    NNT_ASSERT_RESULT_SUCCESS(nn::vi::GetDisplayCmuMode(&revert, pDisplay));

    for( int i = 0; i < sizeof(g_SupportedCmuModes) / sizeof(g_SupportedCmuModes[0]); ++i )
    {
        NNT_EXPECT_RESULT_SUCCESS(nn::vi::SetDisplayCmuMode(pDisplay, g_SupportedCmuModes[i]));

        nn::vi::CmuMode current;
        NNT_EXPECT_RESULT_SUCCESS(nn::vi::GetDisplayCmuMode(&current, pDisplay));
        EXPECT_EQ(g_SupportedCmuModes[i], current);
    }

    for( int i = 0; i < sizeof(g_UnsupportedCmuModes) / sizeof(g_UnsupportedCmuModes[0]); ++i )
    {
        nn::vi::CmuMode before;
        NNT_EXPECT_RESULT_SUCCESS(nn::vi::GetDisplayCmuMode(&before, pDisplay));

        NNT_EXPECT_RESULT_FAILURE(nn::vi::ResultNotSupported, nn::vi::SetDisplayCmuMode(pDisplay, g_UnsupportedCmuModes[i]));

        nn::vi::CmuMode after;
        NNT_EXPECT_RESULT_SUCCESS(nn::vi::GetDisplayCmuMode(&after, pDisplay));
        EXPECT_EQ(before, after);
    }

    NNT_EXPECT_RESULT_SUCCESS(nn::vi::SetDisplayCmuMode(pDisplay, revert));

    nn::vi::CloseDisplay(pDisplay);
}

TEST_F(SetDisplayCmuMode, External_IcosaCopper)
{
    nn::vi::Initialize();

    nn::vi::Display* pDisplay;
    NNT_ASSERT_RESULT_SUCCESS(nn::vi::OpenDisplay(&pDisplay, "External"));

    nn::vi::CmuMode revert;
    NNT_ASSERT_RESULT_SUCCESS(nn::vi::GetDisplayCmuMode(&revert, pDisplay));

    for( int i = 0; i < sizeof(g_SupportedCmuModes) / sizeof(g_SupportedCmuModes[0]); ++i )
    {
        NNT_EXPECT_RESULT_SUCCESS(nn::vi::SetDisplayCmuMode(pDisplay, g_SupportedCmuModes[i]));

        nn::vi::CmuMode current;
        NNT_EXPECT_RESULT_SUCCESS(nn::vi::GetDisplayCmuMode(&current, pDisplay));
        EXPECT_EQ(g_SupportedCmuModes[i], current);
    }

    for( int i = 0; i < sizeof(g_UnsupportedCmuModes) / sizeof(g_UnsupportedCmuModes[0]); ++i )
    {
        nn::vi::CmuMode before;
        NNT_EXPECT_RESULT_SUCCESS(nn::vi::GetDisplayCmuMode(&before, pDisplay));

        NNT_EXPECT_RESULT_FAILURE(nn::vi::ResultNotSupported, nn::vi::SetDisplayCmuMode(pDisplay, g_UnsupportedCmuModes[i]));

        nn::vi::CmuMode after;
        NNT_EXPECT_RESULT_SUCCESS(nn::vi::GetDisplayCmuMode(&after, pDisplay));
        EXPECT_EQ(before, after);
    }

    NNT_EXPECT_RESULT_SUCCESS(nn::vi::SetDisplayCmuMode(pDisplay, revert));

    nn::vi::CloseDisplay(pDisplay);
}

TEST_F(SetDisplayCmuMode, Edid_IcosaCopper)
{
    nn::vi::Initialize();

    nn::vi::Display* pDisplay;
    NNT_ASSERT_RESULT_SUCCESS(nn::vi::OpenDisplay(&pDisplay, "Edid"));

    for( int i = 0; i < sizeof(g_SupportedCmuModes) / sizeof(g_SupportedCmuModes[0]); ++i )
    {
        NNT_EXPECT_RESULT_FAILURE(nn::vi::ResultNotSupported, nn::vi::SetDisplayCmuMode(pDisplay, g_SupportedCmuModes[i]));
    }

    for( int i = 0; i < sizeof(g_UnsupportedCmuModes) / sizeof(g_UnsupportedCmuModes[0]); ++i )
    {
        NNT_EXPECT_RESULT_FAILURE(nn::vi::ResultNotSupported, nn::vi::SetDisplayCmuMode(pDisplay, g_UnsupportedCmuModes[i]));
    }

    nn::vi::CloseDisplay(pDisplay);
}

TEST(SetDisplayCmuModePreconditions, ViNotInitialized_Any)
{
    nn::vi::Display* pDisplay = reinterpret_cast<nn::vi::Display*>(0x1234);
    ASSERT_DEATH_IF_SUPPORTED(nn::vi::SetDisplayCmuMode(pDisplay, nn::vi::CmuMode_Default), "");
}

TEST(SetDisplayCmuModePreconditions, pDisplayNull_Any)
{
    nn::vi::Initialize();

    EXPECT_DEATH_IF_SUPPORTED(nn::vi::SetDisplayCmuMode(nullptr, nn::vi::CmuMode_Default), "");

    nn::vi::Finalize();
}

TEST(SetDisplayCmuModePreconditions, pDisplayInvalidHandle_Any)
{
    nn::vi::Initialize();

    nn::vi::Display* pDisplay = nullptr;
    NNT_EXPECT_RESULT_SUCCESS(nn::vi::OpenDefaultDisplay(&pDisplay));

    if( pDisplay != nullptr )
    {
        nn::vi::CloseDisplay(pDisplay);

        EXPECT_DEATH_IF_SUPPORTED(nn::vi::SetDisplayCmuMode(pDisplay, nn::vi::CmuMode_Default), "");
    }

    nn::vi::Finalize();
}
