﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt.h>
#include <nnt/viUtil/testVi_Fixture.h>
#include <nn/vi.h>

namespace
{
    class ScalingMode : public nnt::vi::util::Fixture
                               , public ::testing::WithParamInterface<nn::vi::ScalingMode>
    {
    };
}

typedef ScalingMode Supported;
typedef ScalingMode Unsupported;

INSTANTIATE_TEST_CASE_P(SetLayerScalingMode_Any,
                        Supported,
                        ::testing::Values(nn::vi::ScalingMode_FitToLayer,
                                          nn::vi::ScalingMode_PreserveAspectRatio));

TEST_P(Supported, ScalingMode)
{
    nn::vi::Initialize();

    nn::vi::Display* pDisplay;
    NNT_ASSERT_RESULT_SUCCESS(nn::vi::OpenDefaultDisplay(&pDisplay));

    nn::vi::Layer* pLayer;
    NNT_ASSERT_RESULT_SUCCESS(nn::vi::CreateLayer(&pLayer, pDisplay));

    NNT_EXPECT_RESULT_SUCCESS(nn::vi::SetLayerScalingMode(pLayer, GetParam()));

    nn::vi::DestroyLayer(pLayer);
    nn::vi::CloseDisplay(pDisplay);
}

INSTANTIATE_TEST_CASE_P(SetLayerScalingMode_Any,
                        Unsupported,
                        ::testing::Values(nn::vi::ScalingMode_Exact,
                                          nn::vi::ScalingMode_ScaleAndCrop,
                                          nn::vi::ScalingMode_None));

TEST_P(Unsupported, ScalingMode)
{
    nn::vi::Initialize();

    nn::vi::Display* pDisplay;
    NNT_ASSERT_RESULT_SUCCESS(nn::vi::OpenDefaultDisplay(&pDisplay));

    nn::vi::Layer* pLayer;
    NNT_ASSERT_RESULT_SUCCESS(nn::vi::CreateLayer(&pLayer, pDisplay));

    NNT_EXPECT_RESULT_FAILURE(nn::vi::ResultNotSupported, nn::vi::SetLayerScalingMode(pLayer, GetParam()));

    nn::vi::DestroyLayer(pLayer);
    nn::vi::CloseDisplay(pDisplay);
}

TEST(SetLayerScalingModePreconditions, ViNotInitialized_Any)
{
    // need some non-null value to not trip other preconditions
    nn::vi::Layer* pLayer = reinterpret_cast<nn::vi::Layer*>(0x1234);

    ASSERT_DEATH_IF_SUPPORTED(nn::vi::SetLayerScalingMode(pLayer, nn::vi::ScalingMode_FitToLayer), "");
}

TEST(SetLayerScalingModePreconditions, pLayerNull_Any)
{
    nn::vi::Initialize();

    EXPECT_DEATH_IF_SUPPORTED(nn::vi::SetLayerScalingMode(nullptr, nn::vi::ScalingMode_FitToLayer), "");

    nn::vi::Finalize();
}

TEST(SetLayerScalingModePreconditions, pLayerInvalidHandle_Any)
{
    nn::vi::Initialize();

    nn::vi::Display* pDisplay = nullptr;
    NNT_EXPECT_RESULT_SUCCESS(nn::vi::OpenDefaultDisplay(&pDisplay));

    if( pDisplay != nullptr )
    {
        nn::vi::Layer* pLayer = nullptr;
        NNT_EXPECT_RESULT_SUCCESS(nn::vi::CreateLayer(&pLayer, pDisplay));

        if( pLayer != nullptr )
        {
            nn::vi::DestroyLayer(pLayer);

            EXPECT_DEATH_IF_SUPPORTED(nn::vi::SetLayerScalingMode(pLayer, nn::vi::ScalingMode_FitToLayer), "");
        }

        nn::vi::CloseDisplay(pDisplay);
    }

    nn::vi::Finalize();
}
