﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt.h>
#include <nnt/viUtil/testVi_Fixture.h>
#include <nn/vi.h>

typedef nnt::vi::util::Fixture CreateLayer;

TEST_F(CreateLayer, Default_Any)
{
    nn::vi::Initialize();

    nn::vi::Display* pDisplay;
    NNT_ASSERT_RESULT_SUCCESS(nn::vi::OpenDefaultDisplay(&pDisplay));

    nn::vi::Layer* pLayer;
    NNT_ASSERT_RESULT_SUCCESS(nn::vi::CreateLayer(&pLayer, pDisplay));
    ASSERT_NE(nullptr, pLayer);

    nn::vi::DestroyLayer(pLayer);
    nn::vi::CloseDisplay(pDisplay);
}

TEST_F(CreateLayer, DefaultTooManyLayers_Any)
{
    nn::vi::Initialize();

    nn::vi::Display* pDisplay;
    NNT_ASSERT_RESULT_SUCCESS(nn::vi::OpenDefaultDisplay(&pDisplay));

    nn::vi::Layer* pLayer1;
    NNT_ASSERT_RESULT_SUCCESS(nn::vi::CreateLayer(&pLayer1, pDisplay));

    nn::vi::Layer* pLayer2;
    NNT_ASSERT_RESULT_FAILURE(nn::vi::ResultOperationFailed, nn::vi::CreateLayer(&pLayer2, pDisplay));
    EXPECT_EQ(pLayer2, nullptr);

    nn::vi::DestroyLayer(pLayer1);
    nn::vi::CloseDisplay(pDisplay);
}

TEST_F(CreateLayer, FramebufferDimensions_Any)
{
    nn::vi::Initialize();

    nn::vi::Display* pDisplay;
    NNT_ASSERT_RESULT_SUCCESS(nn::vi::OpenDefaultDisplay(&pDisplay));

    nn::vi::Layer* pLayer;
    NNT_ASSERT_RESULT_SUCCESS(nn::vi::CreateLayer(&pLayer, pDisplay, 1920, 1080));
    ASSERT_NE(nullptr, pLayer);

    nn::vi::DestroyLayer(pLayer);
    nn::vi::CloseDisplay(pDisplay);
}

TEST_F(CreateLayer, FramebufferDimensions_InvalidDimensions_Any)
{
    nn::vi::Initialize();

    nn::vi::Display* pDisplay;
    NNT_ASSERT_RESULT_SUCCESS(nn::vi::OpenDefaultDisplay(&pDisplay));

    nn::vi::Layer* pLayer;
    NNT_EXPECT_RESULT_FAILURE(nn::vi::ResultInvalidDimensions, nn::vi::CreateLayer(&pLayer, pDisplay, 0, 1080));
    EXPECT_EQ(nullptr, pLayer);

    NNT_EXPECT_RESULT_FAILURE(nn::vi::ResultInvalidDimensions, nn::vi::CreateLayer(&pLayer, pDisplay, 1920, 0));
    EXPECT_EQ(nullptr, pLayer);

    NNT_EXPECT_RESULT_FAILURE(nn::vi::ResultInvalidDimensions, nn::vi::CreateLayer(&pLayer, pDisplay, 1920, std::numeric_limits<int>::min()));
    EXPECT_EQ(nullptr, pLayer);

    NNT_EXPECT_RESULT_FAILURE(nn::vi::ResultInvalidDimensions, nn::vi::CreateLayer(&pLayer, pDisplay, std::numeric_limits<int>::min(), 1080));
    EXPECT_EQ(nullptr, pLayer);

    nn::vi::CloseDisplay(pDisplay);
}

TEST(CreateLayerPreconditions, ViNotInitialized_Any)
{
    // just need some non-null value to not trip other preconditions
    nn::vi::Display* pDisplay = reinterpret_cast<nn::vi::Display*>(0x1234);
    nn::vi::Layer* pLayer;

    ASSERT_DEATH_IF_SUPPORTED(nn::vi::CreateLayer(&pLayer, pDisplay), "");
}

TEST(CreateLayerPreconditions, pOutLayerNull_Any)
{
    nn::vi::Initialize();

    nn::vi::Display* pDisplay = nullptr;
    NNT_EXPECT_RESULT_SUCCESS(nn::vi::OpenDefaultDisplay(&pDisplay));

    if( pDisplay != nullptr )
    {
        EXPECT_DEATH_IF_SUPPORTED(nn::vi::CreateLayer(nullptr, pDisplay), "");

        nn::vi::CloseDisplay(pDisplay);
    }

    nn::vi::Finalize();
}

TEST(CreateLayerPreconditions, pDisplayNull_Any)
{
    nn::vi::Initialize();

    nn::vi::Layer* pLayer;
    EXPECT_DEATH_IF_SUPPORTED(nn::vi::CreateLayer(&pLayer, nullptr), "");

    nn::vi::Finalize();
}

TEST(CreateLayerPreconditions, pDisplayInvalidHandle_Any)
{
    nn::vi::Initialize();

    nn::vi::Display* pDisplay = nullptr;
    NNT_EXPECT_RESULT_SUCCESS(nn::vi::OpenDefaultDisplay(&pDisplay));

    if( pDisplay != nullptr )
    {
        nn::vi::CloseDisplay(pDisplay);

        nn::vi::Layer* pLayer;
        EXPECT_DEATH_IF_SUPPORTED(nn::vi::CreateLayer(&pLayer, pDisplay), "");
    }

    nn::vi::Finalize();
}
