﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>

#include <nn/ae/ae.h>
#include <nn/applet/applet.h>
#include <nn/fatal/fatal_Api.h>


void ProgramMain(void*) NN_NOEXCEPT;

namespace {
    // アプレットパラメータ
    char g_ParamBuffer[ nn::applet::StartupParamSizeMax ] = { 0 };
    // Program Winding するかどうか
    bool g_IsProgramWinding = true;

    //--------------------------------------------

    void SaveCallContext() NN_NOEXCEPT
    {
        nn::applet::StorageHandle storageHandle;
        NN_ABORT_UNLESS_RESULT_SUCCESS( nn::applet::CreateStorage(&storageHandle, sizeof(g_ParamBuffer) ) );
        NN_ABORT_UNLESS_RESULT_SUCCESS( nn::applet::WriteToStorage(storageHandle, 0, &g_ParamBuffer, sizeof(g_ParamBuffer) ) );
        nn::ae::PushToContextStack(storageHandle);
    }

    nn::ae::LibraryAppletStartHookResult LibraryAppletStartHook(nn::applet::AppletId appletId, nn::applet::LibraryAppletMode libraryAppletMode, void *) NN_NOEXCEPT
    {
        if ( !g_IsProgramWinding )
        {
            return nn::ae::LibraryAppletStartHookResult_Normal;
        }
        SaveCallContext();
        return nn::ae::LibraryAppletStartHookResult_WindProgram;
    }

    nn::applet::LibraryAppletHandle PrepareLibraryAppletImpl( nn::applet::AppletId id, const char* inData, size_t inDataSize, bool isAllForeground ) NN_NOEXCEPT
    {
        // 作成
        nn::applet::LibraryAppletHandle appletHandle;
        NN_ABORT_UNLESS_RESULT_SUCCESS( nn::applet::CreateLibraryApplet( &appletHandle, id,
            isAllForeground ? nn::applet::LibraryAppletMode_AllForeground : nn::applet::LibraryAppletMode_PartialForeground ) );

        // 入力データの push
        {
            // ストレージの作成
            nn::applet::StorageHandle storageHandle;
            NN_ABORT_UNLESS_RESULT_SUCCESS( nn::applet::CreateStorage(&storageHandle, inDataSize) );
            // ストレージへの書き込み
            NN_ABORT_UNLESS_RESULT_SUCCESS( nn::applet::WriteToStorage(storageHandle, 0, inData, inDataSize) );
            // 入力チャンネルへの push
            nn::applet::PushToInChannel( appletHandle, storageHandle );
        }
        return appletHandle;
    }

    nn::applet::LibraryAppletHandle CreateLibraryAppletForDevelop() NN_NOEXCEPT
    {
        return PrepareLibraryAppletImpl( nn::applet::AppletId_LibraryAppletSwkbd, g_ParamBuffer, sizeof(g_ParamBuffer), false );
    }

    void LibraryAppletMain( const nn::ae::LibraryAppletSelfInfo& info ) NN_NOEXCEPT
    {
        auto p = info;
        ProgramMain(&p);
    }

}

extern "C" void nnMain() NN_NOEXCEPT
{
    nn::fatal::SetFatalPolicy(nn::fatal::FatalPolicy_OnlyErrorReport);
    nn::ae::SetLibraryAppletStartHook( LibraryAppletStartHook );
    nn::ae::SetCreateSelfLibraryAppletForDevelop( CreateLibraryAppletForDevelop );
    nn::ae::InvokeLibraryAppletMain( LibraryAppletMain );
}
