﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "PanelRenderer.h"
#include "PanelRendererImpl.h"

bool PanelRenderer::DrawImplClearColor(
    nn::gfx::CommandBuffer* pCommandBuffer,
    const std::shared_ptr<IPanel>& panel,
    const RenderContext& ctxt
    ) NN_NOEXCEPT
{
    auto rect = panel->GetGlobalRectangle();
    auto srect = rect.Product(ctxt.cropRect);
    Framework::MakeSetViewportCommand(pCommandBuffer, rect);
    Framework::MakeSetScissorCommand(pCommandBuffer, srect);
    Framework::MakeClearCommand(pCommandBuffer, panel->GetColor());

    if(ctxt.isRegisterRegionRequired)
    {
        Framework::RegisterClearRegion(srect);
    }

    return true;
}

//---------------------------------------------------------------------------------
namespace {
    struct VertexConstants
    {
        //float position[4];
        float textureRegion[4];
        //float size[2];
        int isFlipYEnabled;
    };

    struct PixelConstants
    {
        float color[4];
        int isTextureEnabled;
    };

    class PanelInstanceState{
    public:
        PanelInstanceState() NN_NOEXCEPT
        {
            auto& impl = panel::g_RendererImpl;
            {
                nn::gfx::BufferInfo info;
                info.SetDefault();
                info.SetSize(sizeof(VertexConstants));
                info.SetGpuAccessFlags(nn::gfx::GpuAccess_ConstantBuffer);
                if( NN_STATIC_CONDITION( nn::gfx::Buffer::IsMemoryPoolRequired ) )
                {
                    size_t size = sizeof(VertexConstants);
                    size_t alignment = nn::gfx::Buffer::GetBufferAlignment(Framework::GetDevice(), info);
                    ptrdiff_t offset = Framework::AllocateConstantBufferPoolMemory(size, alignment);
                    m_VertexConstantBuffer.Initialize(Framework::GetDevice(), info, Framework::GetConstantBufferMemoryPool(), offset, size);
                    m_VertexConstantOffset = offset;
                }
                else
                {
                    m_VertexConstantBuffer.Initialize(Framework::GetDevice(), info, nullptr, 0, 0);
                    m_VertexConstantOffset = -1;
                }
            }
            {
                nn::gfx::BufferInfo info;
                info.SetDefault();
                info.SetSize(sizeof(PixelConstants));
                info.SetGpuAccessFlags(nn::gfx::GpuAccess_ConstantBuffer);
                if( NN_STATIC_CONDITION( nn::gfx::Buffer::IsMemoryPoolRequired ) )
                {
                    size_t size = sizeof(PixelConstants);
                    size_t alignment = nn::gfx::Buffer::GetBufferAlignment(Framework::GetDevice(), info);
                    ptrdiff_t offset = Framework::AllocateConstantBufferPoolMemory(size, alignment);
                    m_PixelConstantBuffer.Initialize(Framework::GetDevice(), info, Framework::GetConstantBufferMemoryPool(), offset, size);
                    m_PixelConstantOffset = offset;
                }
                else
                {
                    m_PixelConstantBuffer.Initialize(Framework::GetDevice(), info, nullptr, 0, 0);
                    m_PixelConstantOffset = -1;
                }
            }
            m_ObjectId = impl.objectId++;
        };

        ~PanelInstanceState() NN_NOEXCEPT
        {
            m_VertexConstantBuffer.Finalize(Framework::GetDevice());
            m_PixelConstantBuffer.Finalize(Framework::GetDevice());
            if(m_VertexConstantOffset != -1)
            {
                Framework::FreeConstantBufferPoolMemory(m_VertexConstantOffset);
            }
            if(m_PixelConstantOffset != -1)
            {
                Framework::FreeConstantBufferPoolMemory(m_PixelConstantOffset);
            }
        };

        nn::gfx::Buffer* GetVertexConstantBuffer() NN_NOEXCEPT
        {
            return &m_VertexConstantBuffer;
        }

        nn::gfx::Buffer* GetPixelConstantBuffer() NN_NOEXCEPT
        {
            return &m_PixelConstantBuffer;
        }

        VertexConstants* MapVertexConstants() NN_NOEXCEPT
        {
            return m_VertexConstantBuffer.Map<VertexConstants>();
        }

        void UnmapVertexConstants() NN_NOEXCEPT
        {
            m_VertexConstantBuffer.FlushMappedRange(0, sizeof(VertexConstants));
            m_VertexConstantBuffer.Unmap();
        }

        PixelConstants* MapPixelConstants() NN_NOEXCEPT
        {
            return m_PixelConstantBuffer.Map<PixelConstants>();
        }

        void UnmapPixelConstants() NN_NOEXCEPT
        {
            m_PixelConstantBuffer.FlushMappedRange(0, sizeof(PixelConstants));
            m_PixelConstantBuffer.Unmap();
        }

        void SetTexture(const std::shared_ptr<framework::PoolTexture> value) NN_NOEXCEPT
        {
            m_Texture = value;
        }

    private:
        nn::gfx::Buffer m_VertexConstantBuffer;
        nn::gfx::Buffer m_PixelConstantBuffer;
        ptrdiff_t m_VertexConstantOffset;
        ptrdiff_t m_PixelConstantOffset;
        std::shared_ptr<framework::PoolTexture> m_Texture;
        int m_ObjectId; // for debug
    };
}

bool PanelRenderer::DrawImplShader(
    nn::gfx::CommandBuffer* pCommandBuffer,
    const std::shared_ptr<IPanel>& panel,
    const std::shared_ptr<framework::PoolTexture>& texture,
    const RenderContext& ctxt
) NN_NOEXCEPT
{
    if(panel->GetColor().GetA() == 0 && texture == nullptr)
    {
        return false;
    }

    auto& impl = panel::g_RendererImpl;
    auto rect = panel->GetGlobalRectangle();
    auto srect = rect.Product(ctxt.cropRect);
    Framework::MakeSetViewportCommand(pCommandBuffer, rect);
    Framework::MakeSetScissorCommand(pCommandBuffer, srect);
    pCommandBuffer->SetShader(impl.panelShader.GetShader(0), nn::gfx::ShaderStageBit_All);
    pCommandBuffer->SetRasterizerState(&impl.rasterizerState);
    pCommandBuffer->SetBlendState(impl.blendState.Get());
    pCommandBuffer->SetDepthStencilState(&impl.depthStencilState);

    auto pInstanceState = std::make_shared<PanelInstanceState>();
    Framework::LockFrameCommandResource(ctxt.resourceLockIndex, pInstanceState);
    {
        auto p = pInstanceState->MapVertexConstants();
        p->textureRegion[0] = 0;
        p->textureRegion[1] = 0;
        p->textureRegion[2] = 0;
        p->textureRegion[3] = 0;
        if(texture != nullptr)
        {
            float poolWidth  = static_cast<float>(texture->GetPoolWidth());
            float poolHeight = static_cast<float>(texture->GetPoolHeight());
            float x = static_cast<float>(texture->GetOriginX());
            float y = static_cast<float>(texture->GetOriginY());
            float w = static_cast<float>(texture->GetWidth());
            float h = static_cast<float>(texture->GetHeight());
            p->textureRegion[0] = /*0*/ x / poolWidth;
            p->textureRegion[1] = /*0*/ y / poolHeight;
            p->textureRegion[2] = /*1*/ w / poolWidth;
            p->textureRegion[3] = /*1*/ h / poolHeight;
        }
        p->isFlipYEnabled = panel->GetFlipYEnabled() ? 1 : 0;
        pInstanceState->UnmapVertexConstants();
        nn::gfx::GpuAddress a;
        pInstanceState->GetVertexConstantBuffer()->GetGpuAddress(&a);
        pCommandBuffer->SetConstantBuffer(0, nn::gfx::ShaderStage_Vertex, a, sizeof(VertexConstants));
    }
    {
        auto p = pInstanceState->MapPixelConstants();
        auto color = panel->GetColor();
        p->color[0] = color.GetR();
        p->color[1] = color.GetG();
        p->color[2] = color.GetB();
        p->color[3] = color.GetA();
        p->isTextureEnabled = (texture != nullptr) ? 1 : 0;
        pInstanceState->UnmapPixelConstants();
        nn::gfx::GpuAddress a;
        pInstanceState->GetPixelConstantBuffer()->GetGpuAddress(&a);
        pCommandBuffer->SetConstantBuffer(0, nn::gfx::ShaderStage_Pixel, a, sizeof(PixelConstants));
    }
    if(texture)
    {
        pInstanceState->SetTexture(texture);
        pCommandBuffer->SetTextureAndSampler(0, nn::gfx::ShaderStage_Pixel, texture->GetDescriptorSlot(), Framework::GetDefaultSamplerDescriptorSlot());
    }

    pCommandBuffer->InvalidateMemory(nn::gfx::GpuAccess_ConstantBuffer);
    pCommandBuffer->Draw(nn::gfx::PrimitiveTopology_TriangleStrip, 4, 0);

    if(ctxt.isRegisterRegionRequired)
    {
        Framework::RegisterClearRegion(srect);
    }

    return true;
}

bool PanelRenderer::DrawImplTextWriter(
    nn::gfx::CommandBuffer* pCommandBuffer,
    const std::shared_ptr<IPanel>& panel,
    const std::shared_ptr<framework::PoolTextWriterHolder>& textWriter,
    const RenderContext& ctxt
) NN_NOEXCEPT
{
    auto& impl = panel::g_RendererImpl;
    auto rect = panel->GetGlobalRectangle();
    auto srect = rect.Product(ctxt.cropRect);
    Framework::MakeSetViewportCommand(pCommandBuffer, rect);
    Framework::MakeSetScissorCommand(pCommandBuffer, srect);
    pCommandBuffer->SetRasterizerState(&impl.rasterizerState);
    pCommandBuffer->SetBlendState(impl.blendState.Get());
    pCommandBuffer->SetDepthStencilState(&impl.depthStencilState);
    textWriter->Get()->MakeCommand(pCommandBuffer);

    Framework::LockFrameCommandResource(ctxt.resourceLockIndex, textWriter);

    if(ctxt.isRegisterRegionRequired)
    {
        Framework::RegisterClearRegion(srect);
    }

    return true;
}

