﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "PanelContainer.h"

#include <algorithm>
#include <nn/nn_Assert.h>

PanelContainer::PanelContainer()
    : PanelBase(PanelType_Container)
    , m_IsCroppingEnabled(true)
{
}

void PanelContainer::SetupDummyContainer() NN_NOEXCEPT
{
    SetPosition(0, 0, 0);
    SetSize(0, 0);
    SetColor(nn::util::Color4f(0, 0, 0, 0));
    SetTransparent(true);
    SetCroppingEnabled(false);
}

int PanelContainer::GetChildCount() const NN_NOEXCEPT
{
    return static_cast<int>(m_Children.size());
}

std::shared_ptr<IPanel> PanelContainer::GetChild(int index) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_RANGE(index, 0, GetChildCount());
    return m_Children[index];
}

void PanelContainer::AddChild(const std::shared_ptr<IPanel>& child) NN_NOEXCEPT
{
    m_Children.push_back(child);
    child->SetParent(shared_from_this());
}

void PanelContainer::RemoveChild(const std::shared_ptr<IPanel>& child) NN_NOEXCEPT
{
    NN_ASSERT_EQUAL(shared_from_this(), child->GetParent());
    auto it = std::remove_if(m_Children.begin(), m_Children.end(), [&child](const std::shared_ptr<IPanel>& p){ return p == child; });
    m_Children.resize(it - m_Children.begin());
    child->SetParent(nullptr);
}

void PanelContainer::ClearChildren() NN_NOEXCEPT
{
    for(auto& p : m_Children)
    {
        p->SetParent(nullptr);
    }
    m_Children.clear();
}

bool PanelContainer::IsCroppingEnabled() const NN_NOEXCEPT
{
    return m_IsCroppingEnabled;
}

void PanelContainer::SetCroppingEnabled(bool value) NN_NOEXCEPT
{
    m_IsCroppingEnabled = value;
}
