﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>
#include <nnt.h>

#include "testVi_Context.h"
#include "testVi_Macro.h"

static const int DefaultLoopCount = 100;
static const int DefaultSlotIndex = 0;
static const int DefaultFrameBufferCount = 4;

NNT_VI_TEST_SERVERSTATE(SBCreate_SBDestroy)
{
    const int LoopCount = DefaultLoopCount;
    const int slot = DefaultSlotIndex;
    const int bufferCount = DefaultFrameBufferCount;
    NN_LOG("SBCreate>SBDestroy-------------------------------\n");
    Context context;

    context.ConnectService();
    for(int i = 0; i < LoopCount; i++)
    {
        NN_LOG("LOOP %d/%d\n", i + 1, LoopCount);
        auto h = context.CreateSharedBuffer(slot, bufferCount);
        context.DestroySharedBuffer(slot, h);
    }
    context.DisconnectService();
}

NNT_VI_TEST_SERVERSTATE(ShimInit_SBCreate_ShimExit)
{
    const int LoopCount = DefaultLoopCount;
    const int slot = DefaultSlotIndex;
    const int bufferCount = DefaultFrameBufferCount;
    NN_LOG("ShimInit>SBCreate>ShimExit-------------------------------\n");
    Context context;

    for(int i = 0; i < LoopCount; i++)
    {
        NN_LOG("LOOP %d/%d\n", i + 1, LoopCount);
        context.ConnectService();
        context.CreateSharedBuffer(slot, bufferCount);
        //context.DestroySharedBuffer(h);
        context.DisconnectService();
        context.CleanSharedBuffer(slot);
    }
}

NNT_VI_TEST_SERVERSTATE(SBExport_SBUnexport)
{
    const int LoopCount = DefaultLoopCount;
    const int slot = DefaultSlotIndex;
    const int bufferCount = DefaultFrameBufferCount;
    Context context;
    context.ConnectService();
    auto h = context.CreateSharedBuffer(slot, bufferCount);
    for(int i = 0; i < LoopCount; i++)
    {
        NN_LOG("LOOP %d/%d\n", i + 1, LoopCount);
        NNT_EXPECT_RESULT_SUCCESS(context.RegisterSharedBufferImporterAruid(h, nn::applet::GetAppletResourceUserId()));
        NNT_EXPECT_RESULT_SUCCESS(context.UnregisterSharedBufferImporterAruid(h, nn::applet::GetAppletResourceUserId()));
    }
    context.DestroySharedBuffer(slot, h);
    context.DisconnectService();
}


NNT_VI_TEST_SERVERSTATE(SBExport_SBUnexport_Multiple)
{
    const int LoopCount = DefaultLoopCount;
    const int slot = DefaultSlotIndex;
    const int bufferCount = DefaultFrameBufferCount;
    Context context;
    context.ConnectService();
    auto h = context.CreateSharedBuffer(slot, bufferCount);
    for(int i = 0; i < LoopCount; i++)
    {
        NN_LOG("LOOP %d/%d\n", i + 1, LoopCount);
        NNT_EXPECT_RESULT_SUCCESS(context.RegisterSharedBufferImporterAruid(h, nn::applet::GetAppletResourceUserId()));
        NNT_EXPECT_RESULT_FAILURE(nn::vi::ResultAlreadyOpened, context.RegisterSharedBufferImporterAruid(h, nn::applet::GetAppletResourceUserId()));
        NNT_EXPECT_RESULT_FAILURE(nn::vi::ResultAlreadyOpened, context.RegisterSharedBufferImporterAruid(h, nn::applet::GetAppletResourceUserId()));
        NNT_EXPECT_RESULT_FAILURE(nn::vi::ResultAlreadyOpened, context.RegisterSharedBufferImporterAruid(h, nn::applet::GetAppletResourceUserId()));
        NNT_EXPECT_RESULT_SUCCESS(context.UnregisterSharedBufferImporterAruid(h, nn::applet::GetAppletResourceUserId()));
        NNT_EXPECT_RESULT_FAILURE(nn::vi::ResultNotFound, context.UnregisterSharedBufferImporterAruid(h, nn::applet::GetAppletResourceUserId()));
        NNT_EXPECT_RESULT_FAILURE(nn::vi::ResultNotFound, context.UnregisterSharedBufferImporterAruid(h, nn::applet::GetAppletResourceUserId()));
        NNT_EXPECT_RESULT_FAILURE(nn::vi::ResultNotFound, context.UnregisterSharedBufferImporterAruid(h, nn::applet::GetAppletResourceUserId()));
    }
    context.DestroySharedBuffer(slot, h);
    context.DisconnectService();
}

NNT_VI_TEST_SERVERSTATE(SBExport_ShimFinalize)
{
    const int LoopCount = DefaultLoopCount;
    const int slot = DefaultSlotIndex;
    const int bufferCount = DefaultFrameBufferCount;
    Context context;
    for(int i = 0; i < LoopCount; i++)
    {
        NN_LOG("LOOP %d/%d\n", i + 1, LoopCount);
        context.ConnectService();
        auto h = context.CreateSharedBuffer(slot, bufferCount);
        NNT_EXPECT_RESULT_SUCCESS(context.RegisterSharedBufferImporterAruid(h, nn::applet::GetAppletResourceUserId()));
        NNT_EXPECT_RESULT_FAILURE(nn::vi::ResultAlreadyOpened, context.RegisterSharedBufferImporterAruid(h, nn::applet::GetAppletResourceUserId()));
        //context.DestroySharedBuffer(h);
        context.DisconnectService();
        context.CleanSharedBuffer(slot);
    }
}

NNT_VI_TEST_SERVERSTATE(MLBind_MLUnbind)
{
    const int LoopCount = DefaultLoopCount;
    const int slot = DefaultSlotIndex;
    const int bufferCount = DefaultFrameBufferCount;
    NN_LOG("MLBind>MLUnbind-------------------------------\n");
    Context context;

    context.ConnectService();
    auto h = context.CreateSharedBuffer(slot, bufferCount);
    auto displayId = context.OpenDisplay();
    auto layerId = context.CreateManagedLayer(displayId);
    for(int i = 0; i < LoopCount; i++)
    {
        NN_LOG("LOOP %d/%d\n", i + 1, LoopCount);
        NN_ABORT_UNLESS_RESULT_SUCCESS(context.BindLowLevelLayerToManagedLayer(layerId));
        NN_ABORT_UNLESS_RESULT_SUCCESS(context.UnbindLowLevelLayer(layerId));
    }
    context.DestroyManagedLayer(layerId);
    context.CloseDisplay(displayId);
    context.DestroySharedBuffer(slot, h);
    context.DisconnectService();
}

NNT_VI_TEST_SERVERSTATE(MLConnect_MLDisconnect)
{
    const int LoopCount = DefaultLoopCount;
    const int slot = DefaultSlotIndex;
    const int bufferCount = DefaultFrameBufferCount;
    NN_LOG("MLConnect>MLDisconncect-------------------------------\n");
    Context context;

    context.ConnectService();
    auto h = context.CreateSharedBuffer(slot, bufferCount);
    auto displayId = context.OpenDisplay();
    auto layerId = context.CreateManagedLayer(displayId);
    NN_ABORT_UNLESS_RESULT_SUCCESS(context.BindLowLevelLayerToManagedLayer(layerId));
    for(int i = 0; i < LoopCount; i++)
    {
        NN_LOG("LOOP %d/%d\n", i + 1, LoopCount);
        NN_ABORT_UNLESS_RESULT_SUCCESS(context.ConnectLowLevelLayerToSharedBuffer(layerId, h));
        NN_ABORT_UNLESS_RESULT_SUCCESS(context.DisconnectLowLevelLayerFromSharedBuffer(layerId));
    }
    NN_ABORT_UNLESS_RESULT_SUCCESS(context.UnbindLowLevelLayer(layerId));
    context.DestroyManagedLayer(layerId);
    context.CloseDisplay(displayId);
    context.DestroySharedBuffer(slot, h);
    context.DisconnectService();
}

NNT_VI_TEST_SERVERSTATE(MLConnect_MLUnbind)
{
    const int LoopCount = DefaultLoopCount;
    const int slot = DefaultSlotIndex;
    const int bufferCount = DefaultFrameBufferCount;
    NN_LOG("MLConnect>MLUnbind-------------------------------\n");
    Context context;

    context.ConnectService();
    auto h = context.CreateSharedBuffer(slot, bufferCount);
    auto displayId = context.OpenDisplay();
    auto layerId = context.CreateManagedLayer(displayId);
    for(int i = 0; i < LoopCount; i++)
    {
        NN_LOG("LOOP %d/%d\n", i + 1, LoopCount);
        NN_ABORT_UNLESS_RESULT_SUCCESS(context.BindLowLevelLayerToManagedLayer(layerId));
        NN_ABORT_UNLESS_RESULT_SUCCESS(context.ConnectLowLevelLayerToSharedBuffer(layerId, h));
        NN_ABORT_UNLESS_RESULT_SUCCESS(context.UnbindLowLevelLayer(layerId));
        NNT_VI_EXPECT_ERROR(nn::vi::ResultNotFound, context.DisconnectLowLevelLayerFromSharedBuffer(layerId));
    }
    context.DestroyManagedLayer(layerId);
    context.CloseDisplay(displayId);
    context.DestroySharedBuffer(slot, h);
    context.DisconnectService();
}

NNT_VI_TEST_SERVERSTATE(MLConnect_SBDestroy)
{
    const int LoopCount = DefaultLoopCount;
    const int slot = DefaultSlotIndex;
    NN_LOG("MLConnect>MLUnbind-------------------------------\n");
    Context context;

    context.ConnectService();
    auto displayId = context.OpenDisplay();
    auto layerId = context.CreateManagedLayer(displayId);
    for(int i = 0; i < LoopCount; i++)
    {
        int bufferCount = DefaultFrameBufferCount - (i % 2);
        NN_LOG("LOOP %d/%d: buffrCount=%d\n", i + 1, LoopCount, bufferCount);
        auto h = context.CreateSharedBuffer(slot, bufferCount);
        NN_ABORT_UNLESS_RESULT_SUCCESS(context.BindLowLevelLayerToManagedLayer(layerId));
        NN_ABORT_UNLESS_RESULT_SUCCESS(context.ConnectLowLevelLayerToSharedBuffer(layerId, h));
        NN_ABORT_UNLESS_RESULT_SUCCESS(context.UnbindLowLevelLayer(layerId));
        context.DestroySharedBuffer(slot, h);
        NNT_VI_EXPECT_ERROR(nn::vi::ResultNotFound, context.DisconnectLowLevelLayerFromSharedBuffer(layerId));
    }
    context.DestroyManagedLayer(layerId);
    context.CloseDisplay(displayId);
    context.DisconnectService();
}
