﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Result.h>

#include <nn/fs.h>
#include <nn/ssl.h>

#include <pugixml.hpp>
#include <rapidjson/document.h>

enum ReadFileResult
{
    ReadFileResult_Success = 0,
    ReadFileResult_FileOpenError,
    ReadFileResult_FileSizeError,
    ReadFileResult_OutOfMemoryError,
    ReadFileResult_FileReadError,
    ReadFileResult_FileParseError
};

struct FileReaderInfo
{
    nn::fs::FileHandle handle;
    size_t bytesRead = 0;
};

struct FileWriterInfo
{
    nn::fs::FileHandle handle;
    size_t bytesWritten = 0;
};

struct StringReaderInfo
{
    const char *pData = nullptr;
    size_t offset     = 0;
    size_t dataSize   = 0;
};

// Callback used to read a file for upload
// The userdata pointer should be a FileReaderInfo structure.
size_t FileReaderCallback(char *buffer, size_t size, size_t nitems, void *instream);

// Callback used to write a libcurl response to a file.
// The userdata pointer should be a FileWriterInfo structure.
size_t FileWriterCallback(char *ptr, size_t size, size_t nmemb, void *userdata);

// Callback that returns a dummy file containing null bytes.
// The userdata pointer should be the size of the dummy file as a size_t.
size_t NullReaderCallback(char *buffer, size_t size, size_t nitems, void *instream);

// Callback that voids any data that gets sent to it.
// The userdata pointer is ignored.
size_t NullWriterCallback(char *ptr, size_t size, size_t nmemb, void *userdata);

// Callback that reads a "file" from a string in memory.
// The userdata pointer should be a StringReaderInfo structure.
size_t StringReaderCallback(char *buffer, size_t size, size_t nitems, void *instream);

// Callback that stores received data in memory.
// The userdata pointer should be an empty C++ StringStream object.
size_t StringWriterCallback(char *ptr, size_t size, size_t nmemb, void *userdata);

// Create and/or open a blank file for writing. If the specified
// file does not exist, it will be created. If it does exist, it
// will be truncated to 0 bytes and overwritten.
nn::Result CreateFileForWriting(nn::fs::FileHandle *handle, const char *filename);

// Open an existing file for reading.
nn::Result OpenFileForReading(nn::fs::FileHandle *handle, const char *filename);

// Load the testing CA certificate into a given SSL context.
nn::Result LoadCaCertIntoContext(nn::ssl::Context *ctx);

// Parse an XML document in the filesystem.
ReadFileResult ReadXmlDocFromFile(pugi::xml_document *xmlDoc, const char *filename);

// Parse a JSON document in the filesystem.
ReadFileResult ReadJsonDocFromFile(rapidjson::Document *jsonDoc, const char *filename);
