﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "testFixtures.h"
#include "utils.h"

#include <rapidjson/document.h>

#include <sstream>


/*
    This test uploads a file from ROM to the test server and expects a JSON-encoded
    object as a response. The JSON should contain the length of the data the server
    received as well as the MD5 checksum of that data. The client then compares
    the received metadata with what it knows to be correct about the file it sent.
*/
TEST_F(CurlTest, PostFileUploadTest)
{
    CURLcode cResult;
    nn::Result nResult;

    const char *testUrl = "http://ntdlibcurltesting01.ntd.nintendo.com/postfileupload.php";

    // Open the file from ROM for reading
    nn::fs::FileHandle readFile;
    const char *inputFilename = "rom:/nintendo_switch_logo.png";
    nResult = OpenFileForReading(&readFile, inputFilename);
    ASSERT_TRUE(nResult.IsSuccess());

    FileReaderInfo frInfo;
    frInfo.handle = readFile;
    frInfo.bytesRead = 0;

    int64_t fileSize;
    nResult = nn::fs::GetFileSize(&fileSize, readFile);
    ASSERT_TRUE(nResult.IsSuccess());

    // Prepare the form submission
    CURLFORMcode cfResult;
    curl_httppost *form = nullptr;
    curl_httppost *tail = nullptr;
    cfResult = curl_formadd(&form, &tail,
        CURLFORM_COPYNAME, "imagefile",
        CURLFORM_FILENAME, "nintendo_switch_logo.png",
        CURLFORM_STREAM, &frInfo,
        CURLFORM_CONTENTSLENGTH, fileSize,
        CURLFORM_CONTENTTYPE, "image/png",
        CURLFORM_END);
    ASSERT_EQ(CURL_FORMADD_OK, cfResult);

    // Set up and fire the POST request
    cResult = curl_easy_setopt(cHandle, CURLOPT_URL, testUrl);
    ASSERT_EQ(CURLE_OK, cResult);

    cResult = curl_easy_setopt(cHandle, CURLOPT_HTTPPOST, form);
    ASSERT_EQ(CURLE_OK, cResult);

    cResult = curl_easy_setopt(cHandle, CURLOPT_READFUNCTION, FileReaderCallback);
    ASSERT_EQ(CURLE_OK, cResult);

    cResult = curl_easy_setopt(cHandle, CURLOPT_WRITEFUNCTION, StringWriterCallback);
    ASSERT_EQ(CURLE_OK, cResult);

    std::stringstream responseStream;
    cResult = curl_easy_setopt(cHandle, CURLOPT_WRITEDATA, &responseStream);
    ASSERT_EQ(CURLE_OK, cResult);

    cResult = curl_easy_perform(cHandle);
    ASSERT_EQ(CURLE_OK, cResult);

    // Check the response code
    long responseCode;
    cResult = curl_easy_getinfo(cHandle, CURLINFO_RESPONSE_CODE, &responseCode);
    ASSERT_EQ(CURLE_OK, cResult);
    ASSERT_EQ(200, responseCode);

    // Close the file handle
    nn::fs::CloseFile(readFile);

    // Parse the returned JSON into a document
    rapidjson::Document jsonDoc;
    jsonDoc.Parse(responseStream.str().c_str());
    ASSERT_FALSE(jsonDoc.HasParseError());

    // Verify the metadata returned from the server
    ASSERT_EQ(fileSize, jsonDoc["size"].GetInt64());
    ASSERT_STREQ("nintendo_switch_logo.png", jsonDoc["filename"].GetString());
    ASSERT_STREQ("e55931149170d8d22ea6793958d8a7ea", jsonDoc["md5sum"].GetString());
}

/*
    This function verifies that uploading data from a memory buffer using HTTP POST
    works correctly. The test server calculates the MD5 hash of the data it receives
    and sends it back as JSON for the client to compare with the original.
*/
TEST_F(CurlTest, PostBufferUploadTest)
{
    CURLcode cResult;
    nn::Result nResult;

    const char *testUrl = "http://ntdlibcurltesting01.ntd.nintendo.com/postfileupload.php";

    // Read the sample file into a memory buffer
    nn::fs::FileHandle fHandle;
    nResult = OpenFileForReading(&fHandle, "rom:/nintendo_switch_logo.png");
    ASSERT_TRUE(nResult.IsSuccess());

    int64_t fSize;
    nResult = nn::fs::GetFileSize(&fSize, fHandle);
    ASSERT_TRUE(nResult.IsSuccess());

    char *fBuffer = new char[fSize];
    ASSERT_NE(nullptr, fBuffer);

    nResult = nn::fs::ReadFile(fHandle, 0, fBuffer, fSize);
    ASSERT_TRUE(nResult.IsSuccess());

    nn::fs::CloseFile(fHandle);

    // Prepare the form submission
    CURLFORMcode cfResult;
    curl_httppost *form = nullptr;
    curl_httppost *tail = nullptr;
    cfResult = curl_formadd(&form, &tail,
        CURLFORM_COPYNAME, "imagefile",
        CURLFORM_CONTENTTYPE, "image/png",
        CURLFORM_BUFFER, "nintendo_switch_logo.png",
        CURLFORM_BUFFERLENGTH, fSize,
        CURLFORM_BUFFERPTR, fBuffer,
        CURLFORM_END);
    ASSERT_EQ(CURL_FORMADD_OK, cfResult);

    // Open a file to receive the response
    nResult = CreateFileForWriting(&fHandle, "sd:/postupload.json");
    ASSERT_TRUE(nResult.IsSuccess());

    FileWriterInfo fwInfo;
    fwInfo.handle = fHandle;
    fwInfo.bytesWritten = 0;

    // Prepare and send the request
    cResult = curl_easy_setopt(cHandle, CURLOPT_URL, testUrl);
    ASSERT_EQ(CURLE_OK, cResult);

    cResult = curl_easy_setopt(cHandle, CURLOPT_HTTPPOST, form);
    ASSERT_EQ(CURLE_OK, cResult);

    cResult = curl_easy_setopt(cHandle, CURLOPT_WRITEFUNCTION, FileWriterCallback);
    ASSERT_EQ(CURLE_OK, cResult);

    cResult = curl_easy_setopt(cHandle, CURLOPT_WRITEDATA, &fwInfo);
    ASSERT_EQ(CURLE_OK, cResult);

    cResult = curl_easy_perform(cHandle);
    ASSERT_EQ(CURLE_OK, cResult);

    // Close the response file
    nn::fs::CloseFile(fHandle);

    // Verify the response code
    long responseCode;
    cResult = curl_easy_getinfo(cHandle, CURLINFO_RESPONSE_CODE, &responseCode);
    ASSERT_EQ(CURLE_OK, cResult);
    ASSERT_EQ(200, responseCode);

    // Parse the response file
    rapidjson::Document jDocument;
    ReadFileResult rfResult = ReadJsonDocFromFile(&jDocument, "sd:/postupload.json");
    ASSERT_EQ(ReadFileResult_Success, rfResult);

    // Verify the response data
    ASSERT_EQ(fSize, jDocument["size"].GetInt64());
    ASSERT_STREQ("nintendo_switch_logo.png", jDocument["filename"].GetString());
    ASSERT_STREQ("e55931149170d8d22ea6793958d8a7ea", jDocument["md5sum"].GetString());

    // Clean up allocated memory
    delete[] fBuffer;
}

/*
    This test verifies that libcurl can correctly upload files using the HTTP PUT method
    by uploading an image file and checking the filesize and MD5 hash returned by the server.

    The test relies on a server-side handler that computes the MD5 checksum of the
    uploaded data and returns it along with the data size as a JSON stream.
*/
TEST_F(CurlTest, PutFileUploadTest)
{
    CURLcode cResult;
    nn::Result nResult;

    const char *testUrl = "http://ntdlibcurltesting01.ntd.nintendo.com/uploads/image.png";

    // Set up reading file from ROM
    nn::fs::FileHandle readFile;
    const char *inputFilename = "rom:/nintendo_switch_logo.png";
    nResult = OpenFileForReading(&readFile, inputFilename);
    ASSERT_TRUE(nResult.IsSuccess());

    FileReaderInfo frInfo;
    frInfo.handle = readFile;
    frInfo.bytesRead = 0;

    int64_t fileSize;
    nResult = nn::fs::GetFileSize(&fileSize, readFile);
    ASSERT_TRUE(nResult.IsSuccess());

    // Set up writing file to SD card
    nn::fs::FileHandle writeFile;
    const char *outputFilename = "sd:/put_upload.json";
    nResult = CreateFileForWriting(&writeFile, outputFilename);
    ASSERT_TRUE(nResult.IsSuccess());

    FileWriterInfo fwInfo;
    fwInfo.handle = writeFile;
    fwInfo.bytesWritten = 0;

    // Set up the request
    cResult = curl_easy_setopt(cHandle, CURLOPT_URL, testUrl);
    ASSERT_EQ(CURLE_OK, cResult);

    cResult = curl_easy_setopt(cHandle, CURLOPT_UPLOAD, 1);
    ASSERT_EQ(CURLE_OK, cResult);

    cResult = curl_easy_setopt(cHandle, CURLOPT_READFUNCTION, FileReaderCallback);
    ASSERT_EQ(CURLE_OK, cResult);

    cResult = curl_easy_setopt(cHandle, CURLOPT_READDATA, &frInfo);
    ASSERT_EQ(CURLE_OK, cResult);

    cResult = curl_easy_setopt(cHandle, CURLOPT_INFILESIZE_LARGE, fileSize);
    ASSERT_EQ(CURLE_OK, cResult);

    cResult = curl_easy_setopt(cHandle, CURLOPT_WRITEFUNCTION, FileWriterCallback);
    ASSERT_EQ(CURLE_OK, cResult);

    cResult = curl_easy_setopt(cHandle, CURLOPT_WRITEDATA, &fwInfo);
    ASSERT_EQ(CURLE_OK, cResult);

    // Perform the request
    cResult = curl_easy_perform(cHandle);
    ASSERT_EQ(CURLE_OK, cResult);

    // Close the file handles
    nn::fs::CloseFile(readFile);
    nn::fs::CloseFile(writeFile);

    // Check the response code
    long responseCode;
    cResult = curl_easy_getinfo(cHandle, CURLINFO_RESPONSE_CODE, &responseCode);
    ASSERT_EQ(CURLE_OK, cResult);
    ASSERT_EQ(200, responseCode);

    // Parse the response
    rapidjson::Document jDocument;
    ReadFileResult rfResult = ReadJsonDocFromFile(&jDocument, outputFilename);
    ASSERT_EQ(ReadFileResult_Success, rfResult);

    // Verify that the file was sent correctly
    ASSERT_EQ(fileSize, jDocument["size"].GetInt64());
    ASSERT_STREQ("e55931149170d8d22ea6793958d8a7ea", jDocument["md5sum"].GetString());
}
