﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Result.h>
#include <nn/xcd/xcd.h>
#pragma warning(push)
#pragma warning(disable: 4668)
#include <opencv2/opencv.hpp>
#pragma warning(pop)
#include <test_Common.h>
#include <test_TaskBase.h>
#include <test_TextWriter.h>
#include <test_Perf.h>
#include "../TeraIrsensorDriver.h"

//============================================================
class ProcessorTaskBase : public test::TaskBase
{
public:
    ProcessorTaskBase(const char* name, uptr arg)  // arg is unused. you can set NULL when you call the constructor.
    : test::TaskBase(name, NULL)
    , m_IsInitialized(false)
    , m_IsLogEnabled(false)
    , m_IsVisualLogEnabled(false)
    , m_SkipFrame(0)
    , m_TotalFrame(0)
    , m_TotalFrameCount(0)
    , m_CaptureFrameCount(0)
    , m_LogMenuSelect(0)
    , m_pImageMenu(NULL)
    , m_Cursor(0)
    , m_ImageSizeIndex(0)
    , m_SamplingResult()
    //, m_IsImageSizeChanged(false)
    , m_TextWriter()
    , m_Perf()
    , m_ActivePadState()
    , m_Config()
    , m_CurrentConfig()
    {
        m_pDriver = reinterpret_cast<nnt::XcdIrSensorDriver*>(arg);
        for (auto i = 0; i < FORMAT_SIZE_COUNT_MAX; i++)
        {
            m_pImageCaptured[i] = NULL;
            m_pImageCopied[i] = NULL;
        }
    }

    virtual ~ProcessorTaskBase()
    {}

public:
    enum
    {
        MENU_ITEM_EXPOSURE_TIME,
        MENU_ITEM_DIGITAL_GAIN,
        MENU_ITEM_POSTPROCESS_MODE,
        MENU_ITEM_IR_LED_MODE,
        MENU_ITEM_IR_LED_INTENSITY,
        MENU_ITEM_ON_OFF_SUBTRACTION_MODE,
        MENU_ITEM_FPS_MODE,
        MENU_ITEM_HDR_MODE,
        MENU_ITEM_LENS_SHADING_MODE,
        MENU_ITEM_6AXIS_MODE,
        MENU_ITEM_LRA_MODE,
        MENU_ITEM_NUM
    };

    enum PostProcess
    {
        POSTPROCESS_NONE,
        POSTPROCESS_INVERSION,
        POSTPROCESS_MAX_NUM,
    };

    enum DigitalGain
    {
        GAIN_X1,
        GAIN_X2,
        GAIN_X4,
        GAIN_X8,
        GAIN_X16,
        GAIN_MAX_NUM,
    };

    enum IrLedMode
    {
        IR_LED_NONE,
        IR_LED_WIDE,
        IR_LED_NARROW,
        IR_LED_BOTH,
        IR_LED_WIDE_ALWAYS,
        IR_LED_NARROW_ALWAYS,
        IR_LED_BOTH_ALWAYS,
        IR_LED_MAX_NUM,
    };

    struct ProcessorConfig
    {
        int exposureTimeUsec;
        DigitalGain digitalGain;
        PostProcess postProcess;
        IrLedMode irLedMode;
        int irLedIntensity;
        bool isOnOffSubtractionEnabled;
        bool isHdrEnabled;
        bool isLensShadingEnabled;
        int fps;
        bool is6AxisEnabled;
        bool isLraEnabled;
    };

    static const s32 EXPOSURE_TIME_USEC_MAX = 600;
    static const s32 EXPOSURE_TIME_USEC_MIN = 7;
    static const s32 FPS_MAX = 200;
    static const s32 FPS_MIN = 10;
    static const s32 IR_LED_INTENSITY_MAX = 32;

    static const uint8_t DIGITAL_GAIN_SETTING_DATA[GAIN_MAX_NUM][2];
    static const char* MENU_ITEM_NAMES[MENU_ITEM_NUM];

private:
    ProcessorTaskBase(const ProcessorTaskBase& src);
    ProcessorTaskBase& operator=(const ProcessorTaskBase& rhs);

protected: // you have to implement
    virtual void DoInitializeCore(){}
    virtual void DoFinalizeCore(){}
    virtual void DoCalcCore(IplImage* pImage, IplImage* pDstImage)
    {
        NN_UNUSED(pDstImage);
        NN_UNUSED(pImage);
    }
    virtual void DoDrawCore(IplImage* pImage)
    {
        NN_UNUSED(pImage);
    }
    virtual void DoDrawMenuCore(IplImage* pImage)
    {
        NN_UNUSED(pImage);
    }
    virtual void DoCalcMenuCore(int* pMenuCnt)
    {
        NN_UNUSED(pMenuCnt);
    }
    virtual void DoWriteRegSettingCore(nn::xcd::IrWriteRegisterSetting& setting, int& index)
    {
        NN_UNUSED(setting);
        NN_UNUSED(index);
    }
    virtual void DoSetDefaultConfigCore(ProcessorConfig& config);

    virtual void DoCreateLogFileCore(){}
    virtual void DoCloseLogFileCore(){}

private:
    virtual void DoInitialize();
    virtual void DoFinalize();
    virtual void DoCalc();
    virtual void DoDraw();
    virtual void DoDrawMenu();
    virtual void DoCalcMenu();
    virtual void DoWriteRegSetting();
    virtual void DoSetDefaultConfig();

private:
    void InitializeProcessor();
    void FinalizeProcessor();

    void CalcMenu();
    void DrawMenu();

    void DoSetLog();

protected:
    enum
    {
        IMAGE_WIDTH = 320,
        IMAGE_HEIGHT = 240
    };

    // Get image config.
    CvSize GetImageSizeCv(int index) const
    {
        int width = IMAGE_WIDTH;
        int height = IMAGE_HEIGHT;
        for (int i = 0; i < index; i++)
        {
            width /= 2;
            height /= 2;
        }
        return cvSize(width, height);

    }

    // Access images used internally
    IplImage* GetImageCopied()  // RGB image copied from the image captured
    {
        TEST_ASSERT(m_IsInitialized);
        return m_pImageCopied[m_ImageSizeIndex];
    }

    void SaveImage(const IplImage* pImage) const;
    s32 GetKeycode() const;
    void PutText(IplImage* pImage, const s32 x, const s32 y, const CvScalar& color, const char* fmt, ...);

protected:
    static const s32 KEY_CODE_A = 'a';
    static const s32 KEY_CODE_B = 'b';
    static const s32 KEY_CODE_Y = 'y';
    static const s32 KEY_CODE_X = 'x';
    static const s32 KEY_CODE_R = 'r';
    static const s32 KEY_CODE_L = 'l';
    static const s32 KEY_CODE_UP = 2490368;
    static const s32 KEY_CODE_DOWN = 2621440;
    static const s32 KEY_CODE_LEFT = 2424832;
    static const s32 KEY_CODE_RIGHT = 2555904;

    static const int FORMAT_SIZE_COUNT_MAX = 3;

protected:
    bool m_IsLogEnabled;
    bool m_IsVisualLogEnabled;
    s32 m_SkipFrame;
    s32 m_TotalFrame;
    s32 m_TotalFrameCount;
    s32 m_CaptureFrameCount;
    s32 m_LogMenuSelect;
    nn::xcd::IrCommonData m_ImageHeader;
    s32 m_Cursor;
    test::TextWriter m_TextWriter;
    char m_DirectoryPath[256];
    s32 m_ImageSizeIndex;
    //bool m_IsImageSizeChanged;
    nn::Result m_SamplingResult;
    nnt::XcdIrSensorDriver* m_pDriver;

private:
    bool m_IsInitialized;
    ProcessorConfig m_Config;
    ProcessorConfig m_CurrentConfig;
    IplImage* m_pImageCaptured[FORMAT_SIZE_COUNT_MAX];
    IplImage* m_pImageCopied[FORMAT_SIZE_COUNT_MAX];
    IplImage* m_pImageMenu;
    test::Perf m_Perf;
    nn::xcd::PadState m_ActivePadState;
    nn::xcd::SensorCalibrationValue m_SensorCal;
}; //ImageTransferModeTaskBase

//============================================================
