﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "DpdModeTestTask.h"
#include <test_ConsoleFrameworkOpenCv.h>
#include <test_CvUtil.h>
#include <nn/xcd/xcd.h>
#include <nn/os/os_Tick.h>

namespace
{

    const CvScalar ColorRed = {36, 28, 255};
    const CvScalar ColorWhite    = { 255,255,255, 255 };
    const CvScalar ColorGray    = { 128,128,128, 255 };
    const CvScalar ColorCyan     = { 239,174,  0, 255 };
    const CvScalar ColorMagenta     = { 255,0,  255, 255 };
    const CvScalar ColorYellow     = {0, 255, 255, 255 };
    const CvScalar ColorGreen   = {0, 255, 0, 255 };

    void DrawLine(CvArr* pImage, int x1, int y1, int x2, int y2, CvScalar color, int width)
    {
        CvPoint p1 = cvPoint(x1, y1);
        CvPoint p2 = cvPoint(x2, y2);
        cvLine(pImage, p1, p2, color, width);
    }

    void DrawCircle(CvArr* pImage, int ox, int oy, int radius, CvScalar color, int width)
    {
        CvPoint p = cvPoint(ox, oy);
        cvCircle(pImage, p, radius, color, width);
    }

    const int DPD_PIXCNT_SIZE_MAX = 65535;

    const char* OUTPUT_IMAGE_FILE_NAME = "%s\\%s_%04d%02d%02d_%02d%02d%02d_%04d.png";
    const char* OUTPUT_DPD_FILE_NAME = "%s\\%s_%s.csv";
    FILE* fp;
    const size_t FILE_NAME_SIZE = 256;
    const size_t LINE_SIZE = 4096;
}

const char* DpdModeTestTask::MENU_ITEM_DPD_NAMES[MENU_ITEM_DPD_NUM] =
{
    "WOI",
};

//============================================================
void DpdModeTestTask::DoInitializeCore()
{
    auto trialCounter = 0;
    const int TrialCountMax = 50;
    while (!m_pDriver->RequestModeSet(nn::xcd::IrProcessorType::Dpd))
    {
        NN_SDK_ASSERT_LESS(trialCounter, TrialCountMax);
        nn::os::SleepThread(nn::TimeSpanType::FromMilliSeconds(15));
        trialCounter++;
    }
    m_ImageHeader.irProcessorType = nn::xcd::IrProcessorType::Dpd;
    // 内部パラメータの初期化
    SetDefaultDpdConfig(m_Config);
    SetDefaultDpdConfig(m_CurrentConfig);
} //DpdModeTestTask::DoInitializeCore()

//============================================================
void DpdModeTestTask::DoFinalizeCore()
{
    m_pDriver->SetSixAxisSampling(false);
    m_pDriver->SetLraSending(false);
    auto trialCounter = 0;
    const int TrialCountMax = 50;
    while (!m_pDriver->RequestModeSet(nn::xcd::IrProcessorType::Ready))
    {
        NN_SDK_ASSERT_LESS(trialCounter, TrialCountMax);
        nn::os::SleepThread(nn::TimeSpanType::FromMilliSeconds(15));
        trialCounter++;
    }
    m_ImageHeader.irProcessorType = nn::xcd::IrProcessorType::Ready;
} //DpdModeTestTask::DoFinalizeCore()

//============================================================
void DpdModeTestTask::DoCalcCore(IplImage* pImage, IplImage* pDstImage)
{
    NN_UNUSED(pImage);
    NN_UNUSED(pDstImage);

    m_SamplingResult = m_pDriver->GetDpdData(&m_ImageHeader, &m_State[0], &m_PacketCnt);
} //DpdModeTestTask::DoCalc()

//============================================================
void DpdModeTestTask::DoDrawCore(IplImage* pImage)
{
    cvSetZero(pImage);
    for (auto i = 0; i < m_PacketCnt; i++)
    {
        DrawDpd(m_State[i], pImage);
    }
    if (m_IsLogEnabled)
    {
        for (auto i = 0; i < m_PacketCnt; i++)
        {
            SaveData(pImage, m_State[i], i);
        }
    }
} //DpdModeTestTask::DoDraw()


void DpdModeTestTask::DrawDpd( nn::xcd::IrDpdProcessorState& dpdProcessorState, IplImage* pResultImage)
{
    int offsetX = 0;
    int offsetY = 0;

    nn::xcd::IrRect woi = {0, 0, 320, 240};
    int woiRightX = woi.x + woi.width - 1;
    int woiLeftX = woi.x;
    int woiUpY = woi.y;
    int woiDownY = woi.y + woi.height - 1;

    // WOI の表示
    DrawLine(pResultImage, woiLeftX, woiUpY, woiLeftX, woiDownY, ColorYellow, 1);
    DrawLine(pResultImage, woiLeftX, woiDownY, woiRightX, woiDownY, ColorYellow, 1);
    DrawLine(pResultImage, woiRightX, woiDownY, woiRightX, woiUpY, ColorYellow, 1);
    DrawLine(pResultImage, woiRightX, woiUpY, woiLeftX, woiUpY, ColorYellow, 1);

    // 検出されたオブジェクトを表示する
    for (int i = 0; i < nn::xcd::IrDpdProcessorObjectCountMax; i++)
    {
        const nn::xcd::IrDpdData& result = dpdProcessorState.objects[i];

        if (result.isDataValid)
        {
            DrawCircle(pResultImage,
                offsetX + static_cast<int>(result.centroid.x),
                offsetY + static_cast<int>(result.centroid.y),
                2, ColorGreen, -1);

            int leftX = offsetX + result.bound.x;
            int rightX = offsetX + result.bound.x + result.bound.width - 1;
            int upY = offsetY + result.bound.y;
            int downY = offsetY + result.bound.y + result.bound.height - 1;
            DrawLine(pResultImage, leftX, upY, leftX, downY, ColorRed, 2);
            DrawLine(pResultImage, leftX, downY, rightX, downY, ColorRed, 2);
            DrawLine(pResultImage, rightX, downY, rightX, upY, ColorRed, 2);
            DrawLine(pResultImage, rightX, upY, leftX, upY, ColorRed, 2);
        }
    }
}

//============================================================
void DpdModeTestTask::DoDrawMenuCore(IplImage* pImage)
{
    const CvSize FONT_SIZE = cvSize(16, 16);
    //const char* WOI_NAMES[] = {"QVGA", "QQVGA", "QQQVGA"};

    for (s32 i = MENU_ITEM_NUM; i < (MENU_ITEM_NUM + MENU_ITEM_DPD_NUM); i++)
    {
        //s32 tx = 2 * FONT_SIZE.width;
        s32 ty = i * FONT_SIZE.height;

        m_TextWriter.SetFontColor((m_Cursor == i) ? test::CONST_COLOR_RED : test::CONST_COLOR_WHITE);
        m_TextWriter.PutText(pImage, 0, ty, "%s", (m_Cursor == i) ? ">" : " ");

        //if (i == (MENU_ITEM_DPD_WOI + MENU_ITEM_NUM))
        //{
        //    m_TextWriter.PutText(pImage, tx, ty, "%s : %d", MENU_ITEM_DPD_NAMES[MENU_ITEM_DPD_WOI], m_Config.woi);
        //}
    }
} //DpdModeTestTask::DoDraw()


//============================================================
void DpdModeTestTask::DoCalcMenuCore(int* pMenuCnt)
{
    //static nn::os::Tick keyPressedTime = nn::os::Tick();
    //static int counter = 0;
    //s32 keycode = GetKeycode();
    if (m_Cursor == (MENU_ITEM_DPD_WOI + MENU_ITEM_NUM))
    {
        //if (keycode == KEY_CODE_LEFT)
        //{
        //    m_Config.woi = static_cast<SizeFormat>(m_Format - 1);
        //    if (m_Format < 0)
        //    {
        //        m_Format = static_cast<SizeFormat>(SIZE_FORMAT_MAX_NUM - 1);
        //    }
        //}
        //if (keycode == KEY_CODE_RIGHT)
        //{
        //    m_Format= static_cast<SizeFormat>(m_Format + 1);
        //    if (m_Format >= SIZE_FORMAT_MAX_NUM)
        //    {
        //        m_Format = static_cast<SizeFormat>(0);
        //    }
        //}
    }
    *pMenuCnt = MENU_ITEM_DPD_NUM;
}//NOLINT(readability/fn_size)

void DpdModeTestTask::DoWriteRegSettingCore(nn::xcd::IrWriteRegisterSetting& setting, int& index)
{
    NN_UNUSED(setting);
    NN_UNUSED(index);
    //if (m_Config.woi != m_CurrentConfig.woi)
    //{
    //    //WOI 変更の修正 (TO BE IMPLEMENTED)
    //    setting.registerBlock[index].bankId = 1;
    //    setting.registerBlock[index].address = 0x30;
    //    setting.registerBlock[index].data = 0x00;
    //    index++;
    //    setting.registerCount = index;
    //}

    m_CurrentConfig = m_Config;
}

void DpdModeTestTask::DoSetDefaultConfigCore(ProcessorConfig& config)
{
    // Dpdモード設定時のデフォルト設定
    config.digitalGain = GAIN_X2;
    config.exposureTimeUsec = 200;
    config.irLedMode = IR_LED_BOTH;
    config.irLedIntensity = 16;
    config.postProcess = POSTPROCESS_NONE;
    config.isOnOffSubtractionEnabled = true;
    config.isHdrEnabled = false;
    config.isLensShadingEnabled = true;
    config.fps = 133;
}

void DpdModeTestTask::SetDefaultDpdConfig(DpdConfig& config)
{
    config.woi.x = 0;
    config.woi.y = 0;
    config.woi.width = 320;
    config.woi.height = 240;
}

void DpdModeTestTask::SaveData(IplImage* pImage, nn::xcd::IrDpdProcessorState& state, int index)
{
    char path[1024];

    SYSTEMTIME st;
    GetLocalTime(&st);

    m_TotalFrameCount++;
    bool isCaptureEnabled = false;
    if (m_SkipFrame == 0 || (m_TotalFrameCount % m_SkipFrame == 0))
    {
        m_CaptureFrameCount++;
        isCaptureEnabled = true;
    }

    if (isCaptureEnabled)
    {
        if (m_IsVisualLogEnabled && m_SamplingResult.IsSuccess())
        {
            // 画像は1枚目のみ保存する
            if (index == 0)
            {
                sprintf_s(path, OUTPUT_IMAGE_FILE_NAME, m_DirectoryPath, GetName(), st.wYear, st.wMonth, st.wDay, st.wHour, st.wMinute, st.wSecond, st.wMilliseconds);
                s32 success = cvSaveImage(path, pImage);
                if (success == 0)
                {
                    TEST_PRINTF("Failed to save an image: %s\n", path);
                    return;
                }
            }
        }

        // データログ
        if (fp != NULL)
        {
            char data[LINE_SIZE];
            sprintf_s(data, "%04d%02d%02d_%02d%02d%02d_%04d", st.wYear, st.wMonth, st.wDay, st.wHour, st.wMinute, st.wSecond, st.wMilliseconds);
            if (m_SamplingResult.IsSuccess())
            {
                sprintf_s(data, "%s,%d,%d", data, state.samplingNumber, state.diffTimeStampCount);
                for (int i = 0; i < nn::xcd::IrDpdProcessorObjectCountMax; i++)
                {
                    sprintf_s(data, "%s,%f,%f,%f,%d,%d,%d,%d,%d,%d",
                        data,
                        state.objects[i].averageIntensity,
                        state.objects[i].centroid.x,
                        state.objects[i].centroid.y,
                        state.objects[i].pixelCount,
                        state.objects[i].bound.x,
                        state.objects[i].bound.y,
                        state.objects[i].bound.width,
                        state.objects[i].bound.height,
                        state.objects[i].isDataValid
                    );
                }
                sprintf_s(data, "%s\n", data);
            }
            else
            {
                sprintf_s(data, "%s,SamplingFailed,%0X\n", data, m_SamplingResult.GetInnerValueForDebug());
            }
            fputs(data, fp);
        }
    }
}

void DpdModeTestTask::DoCreateLogFileCore()
{
    char path[FILE_NAME_SIZE];
    sprintf_s(path, OUTPUT_DPD_FILE_NAME, m_DirectoryPath, GetName(), m_DirectoryPath);
    if ((fp = fopen(path, "w")) == NULL)
    {
        TEST_PRINTF("Failed to create file\n");
    }
    char data[LINE_SIZE];
    sprintf_s(data, "time,samplingNumber,deltaTime");
    for (int i = 0; i< nn::xcd::IrDpdProcessorObjectCountMax; i++)
    {
        sprintf_s(data, "%s,objects[%d].averageIntensity,objects[%d].centroidX,objects[%d].centoridY,objects[%d].pixelCount,objects[%d].boundX,objects[%d].boundY,objects[%d].boundWidth,objects[%d].boundHeight,objects[%d].isValid", data, i, i, i, i, i, i, i, i, i);
    }
    sprintf_s(data, "%s\n", data);
    fputs(data, fp);
}

void DpdModeTestTask::DoCloseLogFileCore()
{
    fclose(fp);
}

